package com.biz.crm.dms.business.policy.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutor;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutorLadder;
import com.biz.crm.dms.business.policy.local.mapper.SalePolicyExecutorMapper;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 和标品中默认的优惠政策执行信息有关的数据层功能汇总
 * @author yinwenjie
 */
@Component
public class SalePolicyExecutorRepository extends ServiceImpl<SalePolicyExecutorMapper, SalePolicyExecutor> {
  @Autowired(required = false)
  private SalePolicyExecutorJpaRepository salePolicyExecutorJpaRepository;
  
  /**
   * 按照指定的优惠政策业务编码和执行编码，在指定的二级租户范围内查询优惠政策执行详情信息（包括关联的阶梯和阶梯变量信息）
   * @param salePolicyCode 指定的优惠政策业务编码
   * @param executorCode 指定的执行器业务编号
   * @param tenantCode 指定的二级租户
   * @return
   */
  public SalePolicyExecutor findBySalePolicyCodeAndExecutorCode(String salePolicyCode ,String executorCode , String tenantCode) {
    SalePolicyExecutor salePolicyExecutor = this.salePolicyExecutorJpaRepository.findBySalePolicyCodeAndExecutorCode(salePolicyCode, executorCode, tenantCode);
    if(salePolicyExecutor == null) {
      return null;
    }
    // 注意其中的阶梯信息，要进行排序（按照indexs进行排序，因为indexs越小，说明阶梯执行顺序越靠前）
    salePolicyExecutor.setSalePolicyExecutorLadders(this.sortedLadders(salePolicyExecutor.getSalePolicyExecutorLadders()));
    return salePolicyExecutor;
  }
  
  /**
   * 对一组阶梯进行排序
   * @param salePolicyExecutorLadders
   * @return
   */
  private Set<SalePolicyExecutorLadder> sortedLadders(Set<SalePolicyExecutorLadder> salePolicyExecutorLadders) {
    if(!CollectionUtils.isEmpty(salePolicyExecutorLadders)) {
      List<SalePolicyExecutorLadder> sortedSalePolicyExecutorLadders = salePolicyExecutorLadders.stream().sorted((source , target) -> source.getIndexs() - target.getIndexs()).collect(Collectors.toList());
      return Sets.newLinkedHashSet(sortedSalePolicyExecutorLadders);
    }
    return salePolicyExecutorLadders;
  }
  
  /**
   * 按照指定的优惠政策业务编码和执行编码，在指定的二级租户范围内查询优惠政策执行详情信息（这里只查询数量）
   * @param salePolicyCode 指定的优惠政策业务编码
   * @param executorCode 指定的执行器业务编号
   * @param tenantCode 指定的二级租户
   * @return
   */
  public Integer countBySalePolicyCodeAndExecutorCode(String salePolicyCode ,String executorCode , String tenantCode) {
    return this.lambdaQuery().eq(SalePolicyExecutor::getSalePolicyCode, salePolicyCode)
    .eq(SalePolicyExecutor::getExecutorCode , executorCode)
    .eq(SalePolicyExecutor::getTenantCode, tenantCode)
    .count();
  }
  
  /**
   * 按照指定的优惠政策业务编码，在指定的二级租户范围内查询优惠政策执行详情信息（包括关联的阶梯和阶梯变量信息，且可能有多个）
   * @param salePolicyCode 指定的优惠政策业务编码
   * @param tenantCode 指定的二级租户
   * @return
   */
  public List<SalePolicyExecutor> findBySalePolicyCode(String salePolicyCode , String tenantCode , String executeStrategyCode) {
    Set<SalePolicyExecutor> salePolicyExecutors = this.salePolicyExecutorJpaRepository.findBySalePolicyCodeAndExecuteStrategyCode(salePolicyCode, tenantCode , executeStrategyCode);
    if(CollectionUtils.isEmpty(salePolicyExecutors)) {
      return null;
    }
    // 注意其中的阶梯信息，要进行排序
    for (SalePolicyExecutor salePolicyExecutor : salePolicyExecutors) {
      salePolicyExecutor.setSalePolicyExecutorLadders(this.sortedLadders(salePolicyExecutor.getSalePolicyExecutorLadders()));
    }
    return Lists.newArrayList(salePolicyExecutors);
  }  
  
  /**
   * 按照优惠政策执行策略的id进行删除
   */
  public void deleteByIds(Set<String> salePolicyExecutorIds) {
    LambdaQueryWrapper<SalePolicyExecutor> lambdaQuery = Wrappers.<SalePolicyExecutor>lambdaQuery();
    lambdaQuery.in(SalePolicyExecutor::getId, salePolicyExecutorIds);
    this.baseMapper.delete(lambdaQuery);
  }
}
