package com.biz.crm.dms.business.policy.local.scopestrategy;

import com.biz.crm.dms.business.policy.local.entity.SalePolicyScopeInfo;
import com.biz.crm.dms.business.policy.local.service.SalePolicyScopeInfoService;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyScopeChannelInfoVo;
import com.biz.crm.dms.business.policy.sdk.enums.ScopeSelectionMethod;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 该策略支持按照渠道信息来进行优惠政策范围圈定的处理方式
 * TODO 标品提供的默认用户选择方式，还要进行优化
 * @author yinwenjie
 */
@Component
public class ChannelForSalePolicyCustomerScopeStrategy extends AbstractSalePolicyCustomerScopeStrategy implements SalePolicyCustomerScopeStrategy<SalePolicyScopeChannelInfoVo> {

  @Autowired(required = false)
  private SalePolicyScopeInfoService salePolicyScopeInfoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  public static final String SCOPE_TYPE = "channelForSalePolicy";
  
  @Override
  public String getScopeType() {
    return SCOPE_TYPE;
  }

  @Override
  public String getScopeTypeDesc() {
    return "渠 道";
  }

  @Override
  public Class<SalePolicyScopeChannelInfoVo> getSalePolicyCustomerInfoClass() {
    return SalePolicyScopeChannelInfoVo.class;
  }
  
  @Override
  public void onSaveSalePolicyCustomerInfo(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy) {
    /*
     * 操作步骤可参考 #OrgForSalePolicyCustomerScopeStrategy.onSaveSalePolicyCustomerInfo 方法中的详细描述
     * */
    // 1、=====
    if(update) {
      Map<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> oldSalePolicyCustomerInfoMaps = oldSalePolicy.getCustomerScopeMapping();
      if(!CollectionUtils.isEmpty(oldSalePolicyCustomerInfoMaps) && !CollectionUtils.isEmpty(oldSalePolicyCustomerInfoMaps.get(getScopeType()))) {
        this.deleteSalePolicyCustomerInfos(getScopeType(), oldSalePolicyCustomerInfoMaps.get(getScopeType()));
      }
    }
    
    // 2、=====
    // 注意，只添加和自己相关的优惠政策信息
    Map<String,Set<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerInfoMaps = currentSalePolicy.getCustomerScopeMapping();
    if(CollectionUtils.isEmpty(salePolicyCustomerInfoMaps) || CollectionUtils.isEmpty(salePolicyCustomerInfoMaps.get(getScopeType()))) {
      return;
    }
    Set<? extends AbstractSalePolicyCustomerScopeInfo> currentSalePolicyCustomerInfos = salePolicyCustomerInfoMaps.get(getScopeType());
    // 修正每一个条信息的冗余值后，开始进行添加
    Set<SalePolicyScopeInfo> salePolicyScopeInfos = Sets.newLinkedHashSet();
    for (AbstractSalePolicyCustomerScopeInfo abstractSalePolicyCustomerInfo : currentSalePolicyCustomerInfos) {
      SalePolicyScopeChannelInfoVo salePolicyScopeChannelInfoVo = (SalePolicyScopeChannelInfoVo)abstractSalePolicyCustomerInfo;
      salePolicyScopeChannelInfoVo.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
      salePolicyScopeChannelInfoVo.setTenantCode(currentSalePolicy.getTenantCode());
      
      SalePolicyScopeInfo salePolicyScopeInfo = new SalePolicyScopeInfo();
      salePolicyScopeInfo.setSalePolicyCode(currentSalePolicy.getSalePolicyCode());
      salePolicyScopeInfo.setTenantCode(currentSalePolicy.getTenantCode());
      salePolicyScopeInfo.setCode(salePolicyScopeChannelInfoVo.getChannelCode());
      salePolicyScopeInfo.setName(salePolicyScopeChannelInfoVo.getChannelName());
      salePolicyScopeInfo.setCustomerScopeType(SCOPE_TYPE);
      salePolicyScopeInfo.setState(1);
      salePolicyScopeInfo.setSelectionMethod(salePolicyScopeChannelInfoVo.getSelectionMethod());
      salePolicyScopeInfos.add(salePolicyScopeInfo);
    }
    this.salePolicyScopeInfoService.createBatch(salePolicyScopeInfos);
  }

  /**
   * 按照渠道信息来进行优惠政策范围圈定的客户信息集合
   * @param tenantCode 指定的二级租户信息
   * @param salePolicyCode 当前构造的具体优惠政策业务编号
   * @return
   */
  @Override
  public Set<SalePolicyScopeChannelInfoVo> requestSalePolicyCustomerInfo(String tenantCode , String salePolicyCode) {
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode)) {
      return null;
    }
    //查询渠道信息范围的客户结果
    List<SalePolicyScopeInfo> salePolicyScopeInfos = this.salePolicyScopeInfoService.findByTenantCodeAndSalePolicyCodeAndCustomerScopeType(tenantCode, salePolicyCode, SCOPE_TYPE);
    if (CollectionUtils.isEmpty(salePolicyScopeInfos)) {
      return null;
    }
    //转成渠道信息范围客户结果vo集合
    Set<SalePolicyScopeChannelInfoVo> salePolicyScopeChannelInfoVos = new HashSet<>();
    for (SalePolicyScopeInfo salePolicyScopeInfo: salePolicyScopeInfos) {
      SalePolicyScopeChannelInfoVo salePolicyScopeChannelInfoVo = this.nebulaToolkitService.copyObjectByWhiteList(salePolicyScopeInfo, SalePolicyScopeChannelInfoVo.class, HashSet.class, ArrayList.class);
      salePolicyScopeChannelInfoVo.setChannelCode(salePolicyScopeInfo.getCode());
      salePolicyScopeChannelInfoVo.setChannelName(salePolicyScopeInfo.getName());
      salePolicyScopeChannelInfoVos.add(salePolicyScopeChannelInfoVo);
    }
    return salePolicyScopeChannelInfoVos;
  }

  /**
   * 按照渠道信息来进行优惠政策范围圈定的客户编码集合
   * @param salePolicyCode 需要查询的优惠政策业务编号
   * @param tenantCode 当前二级租户信息
   * @return
   */
  @Override
  public boolean matched(String customerCode,String tenantCode , String salePolicyCode, Set<SalePolicyScopeChannelInfoVo> salePolicyCustomerInfos) {
    /**
     * 操作步骤：
     * 1、选定方式分组转成渠道信息范围的渠道编码map
     * 2、从包含中的渠道编码过滤掉非包含渠道编码
     * 3、获取渠道范围的客户编码
     * 
     * 从 2022-1月开始，页面上不在传递“不包含”信息
     * 
     */
    if (StringUtils.isAnyBlank(tenantCode, salePolicyCode) || CollectionUtils.isEmpty(salePolicyCustomerInfos)) {
      return false;
    }
    //1、======
    Map<String, List<String>> selectionMethodMap = salePolicyCustomerInfos.stream()
        .filter(a -> StringUtils.isNoneBlank(a.getSelectionMethod(), a.getChannelCode()))
        .collect(Collectors.groupingBy(SalePolicyScopeChannelInfoVo::getSelectionMethod, Collectors.mapping(SalePolicyScopeChannelInfoVo::getChannelCode, Collectors.toList())));
    //2、======
    List<String> includeList = selectionMethodMap.get(ScopeSelectionMethod.INCLUDE.getCode());
    List<String> excludeList = selectionMethodMap.get(ScopeSelectionMethod.EXCLUDE.getCode());
    if (CollectionUtils.isEmpty(includeList)) {
      return false;
    }
    if (!CollectionUtils.isEmpty(excludeList)){
      includeList.removeAll(excludeList);
    }
    //3、======
    Boolean isExist = this.customerVoService.existByCustomerCodeAndChannels(includeList, customerCode);
    return isExist;
  }
}
