package com.biz.crm.dms.business.policy.local.service.internal;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecord;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordExcludePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordGift;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordProduct;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordProductSelectedPolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecordStepPolicy;
import com.biz.crm.dms.business.policy.local.service.SalePolicyRecordService;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicStrategy;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicVoter;
import com.biz.crm.dms.business.policy.sdk.config.SimplePolicyProperties;
import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.local.context.DefaultCycleExecuteContext;
import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.GiftResultInfo;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.context.StepType;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyExecuteService;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyVoService;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteShareStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyStickupListener;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyProductInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyThreshold;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import lombok.extern.slf4j.Slf4j;

/**
 * 优惠政策执行的默认实现
 * @author yinwenjie
 */
@Slf4j
@Service
public class SalePolicyExecuteServiceImpl implements SalePolicyExecuteService<DefaultPolicyExecuteContext> {
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private SimplePolicyProperties simplePolicyProperties;
  @Autowired(required = false)
  private SalePolicyVoService salePolicyVoService;
  @Autowired(required = false)
  private SalePolicyRecordService salePolicyRecordService;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  @Autowired(required = false)
  private Set<CharacteristicStrategy<?>> characteristicStrateis;
  
  @Override
  public DefaultPolicyExecuteContext preJudgment(Set<SalePolicyConProduct> initPolicyProducts, String tenantCode, String customerCode) {
    // 交由外部调用的预判定过程，无需关注“是否揉价”
    return this.preJudgment(initPolicyProducts, tenantCode, customerCode, false);
  }
  
  /**
   * 真正进行预判定逻辑的方法
   */
  private DefaultPolicyExecuteContext preJudgment(Set<SalePolicyConProduct> initPolicyProducts, String tenantCode, String customerCode, boolean kneading) {
    /*
     * 预判定过程，只是为了简单筛选出传入的一批商品在理论上可以使用哪些优惠政策。
     * 这些优惠政策只是采用基本有效期进行过滤，并匹配商品编号即可，注意：这里虽然要返回优惠上下文，但实际上并没有进行优惠政策的实际计算
     * 
     * 在完成了入参边界验证后（例如，initPolicyProducts中不能有规格编号重复的商品信息），处理过程为：
     * 
     * 1、首先查询当前经销商正在执行的有效营销活动（注意是有效的）
     * 2、对本次参与计算的本品initPolicyProducts，基于第1步查询得到的营销活动，进行归类。
     * 归类匹配的营销活动，以及营销活动下的本品信息。
     * 注意，此步骤还将过滤出那些被排除的营销活动，以及那些被排除的本品信息，以及排除原因，该步骤也对整单优惠进行筛选
     * 3、返回结果，由于返回的结果中数据量过大，且冗余数据较多，所以需要对返回对象进行精简，且不能影响到缓存中的原始信息
     * */
    
    Validate.notBlank(tenantCode, "进行优惠计算时，经销商信息必须传入!!");
    Validate.notBlank(customerCode, "进行优惠计算时，客户信息必须传入!!");
    Validate.isTrue(!CollectionUtils.isEmpty(initPolicyProducts), "进行优惠计算时，参与优惠的总体本品规格信息，必须传入!!");
    long initPolicyProductSpecificationCodeSize = initPolicyProducts.stream().map(SalePolicyConProduct::getProductCode).distinct().count();
    Validate.isTrue(initPolicyProductSpecificationCodeSize == initPolicyProducts.size(), "进行优惠计算时，参与优惠的总体本品规格信息，必须传入且不能重复!!");
    
    // 1、=======
    Set<SalePolicyVo> salePolicies = this.salePolicyVoService.findDetailsByProcessingAndCustomerCode(tenantCode, true, customerCode , null);
    if (CollectionUtils.isEmpty(salePolicies)) {
      return null;
    }
    
    // 2、======
    DefaultPolicyExecuteContext executreContext = this.buildDefaultPolicyExecuteContext(salePolicies, initPolicyProducts, tenantCode, customerCode , kneading , false , null);
    
    // 3、======
    return executreContext.streamlining();
  }
  
  /**
   * 该方法验证在指定的商品规格、指定的客户下，哪些营销活动可以使用，哪些优惠活动不能使用，不能使用的原因是什么</br>
   * 注意，该方法一定不会返回null，要么实在是数据错误无法处理，就直接抛出异常
   * @return 
   */
  private DefaultPolicyExecuteContext buildDefaultPolicyExecuteContext(Set<SalePolicyVo> initSalePolicys, Set<SalePolicyConProduct> initPolicyProducts , String tenantCode, String customerCode , boolean kneading, boolean formal , String relevanceCode) {
    /*
     * 分成三个步骤：
     * 1、根据传入的单据本品信息，计算商品的总小计和总数量
     * 2、如果当前优惠政策是商品优惠政策，则只需要匹配了initPolicyProducts中的任何商品，就认为可以使用
     * 3、如果当前优惠政策是整单优惠，则直接认为可以使用
     * */
    // 1、=======
    BigDecimal productsTotalAmount = BigDecimal.ZERO;
    BigDecimal productsTotalNumber = BigDecimal.ZERO;
    for (SalePolicyConProduct policyConProduct : initPolicyProducts) {
      productsTotalAmount = productsTotalAmount.add(policyConProduct.getSubtotalAmount());
      productsTotalNumber = productsTotalNumber.add(policyConProduct.getQuantity());
    }
    if(initSalePolicys == null) {
      initSalePolicys = Sets.newLinkedHashSet();
    }
    
    DefaultPolicyExecuteContext executreContext = new DefaultPolicyExecuteContext(initPolicyProducts , initSalePolicys , productsTotalAmount , productsTotalNumber.intValue() , tenantCode, customerCode , kneading , formal , relevanceCode);
    // 该Map代表经过后续的每次循环判定后，剩下的还没有参与任何优惠活动的本品信息
    Map<String, SalePolicyConProduct> remainingPolicyProducts = initPolicyProducts.stream().collect(Collectors.toMap(SalePolicyConProduct::getProductCode, item -> item));
    Set<String> targetPolicyProductCodes = remainingPolicyProducts.values().stream().map(SalePolicyConProduct::getProductCode).collect(Collectors.toSet());
    
    // 2、======
    for (SalePolicyVo salePolicyItem : initSalePolicys) {
      // 如果是整单优惠，则不用在第2步进行判定
      if(salePolicyItem.getWholePolicy()) {
        continue;
      }
      Set<? extends AbstractSalePolicyProductInfo> sourceProducts = salePolicyItem.getSalePolicyProductInfos();
      if(CollectionUtils.isEmpty(sourceProducts)) {
        continue;
      }
      Set<String> sourceProductCodes = sourceProducts.stream().filter(item -> StringUtils.isNotBlank(item.getProductCode())).map(AbstractSalePolicyProductInfo::getProductCode).collect(Collectors.toSet());
      // 判定两个商品业务编号的集合，是否存在交集，只要存在交集就算理论上可以使用的优惠
      Set<String> intersectionCodeValues = Sets.intersection(targetPolicyProductCodes, sourceProductCodes);
      if(!CollectionUtils.isEmpty(intersectionCodeValues)) {
        executreContext.addNewMathedPolicy(salePolicyItem, intersectionCodeValues.toArray(new String[] {}));
        for (String intersectionCode : intersectionCodeValues) {
          remainingPolicyProducts.remove(intersectionCode);
        }
      } else {
        // 计入上下文对象的“不能享受的优惠政策”区域，并进行后续判定
        String message = "未涵盖该政策范围内的商品信息.";
        executreContext.addExcludedPolicy(salePolicyItem, message);
      }
    }
    
    // 3、======
    for (SalePolicyVo salePolicyItem : initSalePolicys) {
      // 这里只做整单优惠的判定
      if(!salePolicyItem.getWholePolicy()) {
        continue;
      }
      // 认为整单优惠理论上都能用
      executreContext.addNewMathedPolicy(salePolicyItem, initPolicyProducts.stream().map(SalePolicyConProduct::getProductCode).toArray(String[]::new));
    }
    return executreContext;
  }
  
  @Override
  public DefaultPolicyExecuteContext preExecute(Set<SalePolicyConProduct> initPolicyProducts, Map<String, String[]> selectPolicyCodeMapping, String tenantCode, String customerCode, boolean kneading) {
    return this.execute(initPolicyProducts, selectPolicyCodeMapping, tenantCode, customerCode, kneading, false, null);
  }

  @Override
  public DefaultPolicyExecuteContext preExecute(Set<SalePolicyConProduct> initPolicyProducts, String tenantCode, String customerCode, boolean kneading) {
    return this.preExecute(initPolicyProducts, null, tenantCode, customerCode , kneading);
  }
  
  /**
   * 该私有方法用于根据最新的优惠政策本品情况，判定优惠政策的门槛
   */
  private Pair<Map<String,Set<String>>,String> validateSalePolicyThreshold(SalePolicyTempleteRegister salePolicyTempleteRegister , DefaultPolicyExecuteContext executeContext , Set<String> policyConProductCodes , SalePolicyVo currentSalePolicy) {
    SalePolicyStickupListener<? extends AbstractSalePolicyThreshold, ? extends AbstractSalePolicyProductInfo> salePolicyStickupListener = salePolicyTempleteRegister.getStickupListener();
    AbstractSalePolicyThreshold salePolicyThreshold = currentSalePolicy.getSalePolicyProductThreshold();
    // 注意：注意门槛要以当前本品的最新状态，进行门槛判定
    Set<SalePolicyConProduct> latestPolicyProducts = this.buildLatestPolicyProducts(policyConProductCodes, executeContext);
    Pair<Map<String,Set<String>>,String> thresholdMatchedPair = salePolicyStickupListener.executeThreshold(latestPolicyProducts , executeContext , salePolicyThreshold);
    return thresholdMatchedPair;
  }
  
  /**
   * 该私有方法负责按照最新的优惠政策执行情况构造最新的本品状态，这些商品一般用于负责进行后续优惠门槛的判定
   * @param productCodes 
   * @param executeContext 
   * @return
   */
  private Set<SalePolicyConProduct> buildLatestPolicyProducts(Set<String> productCodes , DefaultPolicyExecuteContext executeContext) {
    Set<SalePolicyConProduct> policyConProducts = Sets.newLinkedHashSet();
    for (String productCode : productCodes) {
      ProductPolicyStepResult productPolicyStepResult = executeContext.findLastProductStepResultByProductCode(productCode);
      SalePolicyConProduct defaultPolicyConProduct = new SalePolicyConProduct();
      defaultPolicyConProduct.setProductCode(productCode);
      defaultPolicyConProduct.setQuantity(new BigDecimal(productPolicyStepResult.getInitNumbers()));
      defaultPolicyConProduct.setSubtotalAmount(productPolicyStepResult.getLastSubtotal());
      defaultPolicyConProduct.setUnitPrice(productPolicyStepResult.getLastPrices());
      policyConProducts.add(defaultPolicyConProduct);
    }
    return policyConProducts;
  }
  
  /**
   * 对指定的一批优惠政策，按照SimplePolicyProperties中设定的排序要求，进行排序。</br>
   * 如果某些优惠政策对应的优惠政策模板（编号）并没有在这个设定中，则会在最后进行执行。</br>
   * 该排序还会自动将本品优惠和整单优惠分组进行排序，以便保证所有的本品性质的优惠在所有的整单优惠之前被执行
   * @param salePolicies 
   * @return 
   */
  private List<String> sortPolicy(Set<SalePolicyVo> salePolicies) {
    List<SalePolicyVo> productSalePolicies = salePolicies.stream().filter(item -> !item.getWholePolicy()).collect(Collectors.toList());
    List<SalePolicyVo> noConfigProductSalePolicies = Lists.newArrayList(productSalePolicies);
    // 1、====== 先做本品优惠政策的排序
    // 排序后的本品优惠政策在这里
    List<SalePolicyVo> sortedProductSalePolicies = Lists.newArrayList();
    String[] typesorts = simplePolicyProperties.getTypesorts();
    for (String typesort : typesorts) {
      for (SalePolicyVo productSalePolicy : productSalePolicies) {
        String type = productSalePolicy.getType();
        if(StringUtils.equals(typesort, type)) {
          sortedProductSalePolicies.add(productSalePolicy);
          noConfigProductSalePolicies.remove(productSalePolicy);
        }
      }
    }
    // 如果走到这里，noConfigProductSalePolicies还有优惠政策，那么说明这个优惠政策对应的type，并没有被设置到配置信息中
    if(!CollectionUtils.isEmpty(noConfigProductSalePolicies)) {
      sortedProductSalePolicies.addAll(noConfigProductSalePolicies);
    }
    
    // 2、====== 再做整单优惠政策的排序
    List<SalePolicyVo> wholePolicySalePolicies = salePolicies.stream().filter(item -> item.getWholePolicy()).collect(Collectors.toList());
    List<SalePolicyVo> noConfigWholeSalePolicies = Lists.newArrayList(wholePolicySalePolicies);
    // 排序后的整单优惠政策在这里
    List<SalePolicyVo> sortedWholeSalePolicies = Lists.newArrayList();
    for (String typesort : typesorts) {
      for (SalePolicyVo wholePolicySalePolicy : wholePolicySalePolicies) {
        String type = wholePolicySalePolicy.getType();
        if(StringUtils.equals(typesort, type)) {
          sortedWholeSalePolicies.add(wholePolicySalePolicy);
          noConfigWholeSalePolicies.remove(wholePolicySalePolicy);
        }
      }
    }
    // 如果走到这里，noConfigProductSalePolicies还有优惠政策，那么说明这个优惠政策对应的type，并没有被设置到配置信息中
    if(!CollectionUtils.isEmpty(noConfigWholeSalePolicies)) {
      sortedWholeSalePolicies.addAll(noConfigWholeSalePolicies);
    }
    
    // 组装结构
    List<SalePolicyVo> results = Lists.newArrayList(sortedProductSalePolicies);
    results.addAll(sortedWholeSalePolicies);
    return results.stream().map(SalePolicyVo::getSalePolicyCode).collect(Collectors.toList());
  }
  
  /**
   * 该私有方法，根据type获得正确的SalePolicyRegister营销策略注册器
   * @param cacheResults
   * @return
   */
  private SalePolicyTempleteRegister findSalePolicyTempleteRegister(String type) {
    Validate.notBlank("错误的优惠政策类型，请检查传入值");
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyTempleteRegisters) , "错误的SalePolicyRegister信息，请检查!!");
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = null;
    for (SalePolicyTempleteRegister item : salePolicyTempleteRegisters) {
      if(StringUtils.equals(type, item.getType())) {
        matchedSalePolicyTempleteRegister = item;
        break;
      }
    }
    Validate.notNull(matchedSalePolicyTempleteRegister , "未匹配任何SalePolicyTempleteRegister信息，请检查!!");
    return matchedSalePolicyTempleteRegister;
  }
  
  /**
   * 该私有方法，在指定的优惠政策模板注册器中，寻找正确的“阶梯执行规则”，因为递归循环要用
   * @return
   */
  private SalePolicyCycleRuleStrategy findCorrectCycleRuleStrategy(SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy , String cycleRuleCode) {
    Set<SalePolicyCycleRuleStrategy> bindableCycleRuleStrategies = this.findAllBindingCycleRuleStrategy(salePolicyExecuteStrategy);
    Validate.isTrue(!CollectionUtils.isEmpty(bindableCycleRuleStrategies) , "未发现当前优惠政策有至少一个能绑定的阶梯执行规则，请检查!!");
    HashMap<String, SalePolicyCycleRuleStrategy> bindableCycleRuleStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyCycleRuleStrategy salePolicyCycleRuleStrategy : bindableCycleRuleStrategies) {
      bindableCycleRuleStrategyMapping.put(salePolicyCycleRuleStrategy.getCycleRuleCode(), salePolicyCycleRuleStrategy);
    }
    
    // 寻找正确的策略处理器，并进行保存处理
    Validate.notBlank(cycleRuleCode , "至少一个将要保存的优惠执行策略没有指定cycleRuleCode信息，请检查!!");
    SalePolicyCycleRuleStrategy currentSalePolicyCycleRuleStrategy = bindableCycleRuleStrategyMapping.get(cycleRuleCode);
    Validate.notNull(currentSalePolicyCycleRuleStrategy , "未匹配到业务编号为:%s 的阶梯执行规则信息，请检查!!" , cycleRuleCode);
    return currentSalePolicyCycleRuleStrategy;
  }
  
  /**
   * 该私有方法用于基于特定的优惠政策模板查询所有已绑定的优惠执行策略
   * @param salePolicyTempleteRegister 
   * @return
   */
  private Set<SalePolicyCycleRuleStrategy> findAllBindingCycleRuleStrategy(SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy) {
    Collection<Class<? extends SalePolicyCycleRuleStrategy>> bindableCycleRuleStrategyClasses = salePolicyExecuteStrategy.getCycleRuleStrategyClasses();
    if(CollectionUtils.isEmpty(bindableCycleRuleStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些限价政策可以进行绑定
    Set<SalePolicyCycleRuleStrategy> bindableCycleRuleStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyCycleRuleStrategy> bindableCycleRuleStrategyClass : bindableCycleRuleStrategyClasses) {
      try {
        SalePolicyCycleRuleStrategy currentSalePolicyCycleRuleStrategy = applicationContext.getBean(bindableCycleRuleStrategyClass);
        bindableCycleRuleStrategySet.add(currentSalePolicyCycleRuleStrategy);
      } catch(RuntimeException e) {
        log.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableCycleRuleStrategySet;
  }
  
  /**
   * 该私有方法根据指定的优惠政策信息，按照优惠政策执行策略业务编码查询执行策略的基本信息 
   * @return 如果没有找到则返回null
   */
  private AbstractSalePolicyExecutorInfo findCorrectExecutorInfo(SalePolicyVo salePolicy , String executorCode) {
    Set<AbstractSalePolicyExecutorInfo> salePolicyExecutorInfos = salePolicy.getSalePolicyExecutorInfos();
    if(CollectionUtils.isEmpty(salePolicyExecutorInfos)) {
      return null;
    }
    for (AbstractSalePolicyExecutorInfo salePolicyExecutorInfo : salePolicyExecutorInfos) {
      if(StringUtils.equals(salePolicyExecutorInfo.getExecutorCode(), executorCode)) {
        return salePolicyExecutorInfo;
      }
    }
    return null;
  }
  
  /**
   * 该私有方法用于基于特定的优惠政策模板查询所有已绑定的优惠执行策略
   * @param salePolicyTempleteRegister 
   * @return
   */
  private Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> findAllBindingExecuteStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>>> bindableExecuteStrategyClasses = salePolicyTempleteRegister.getExecuteStrategyClasses();
    if(CollectionUtils.isEmpty(bindableExecuteStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些限价政策可以进行绑定
    Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategyClass : bindableExecuteStrategyClasses) {
      try {
        SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> currentSalePolicyExecuteStrategy = applicationContext.getBean(bindableExecuteStrategyClass);
        bindableExecuteStrategySet.add(currentSalePolicyExecuteStrategy);
      } catch(RuntimeException e) {
        log.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableExecuteStrategySet;
  }
  
  /**
   * 该私有方法用于查询符合要求的限价政策（基于特定的优惠政策）
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> findCorrectExecuteStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String executeStrategyCode) {
    Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategies = this.findAllBindingExecuteStrategy(salePolicyTempleteRegister);
    Validate.isTrue(!CollectionUtils.isEmpty(bindableExecuteStrategies) , "未发现当前优惠政策有至少一个能绑定的优惠政策执行策略，请检查!!");
    HashMap<String, SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy : bindableExecuteStrategies) {
      bindableExecuteStrategyMapping.put(salePolicyExecuteStrategy.getExecuteStrategyCode(), salePolicyExecuteStrategy);
    }
    
    // 寻找正确的策略处理器，并进行保存处理
    Validate.notBlank(executeStrategyCode , "至少一个将要保存的优惠执行策略没有指定executeStrategyCode信息，请检查!!");
    SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> currentSalePolicyExecuteStrategy = bindableExecuteStrategyMapping.get(executeStrategyCode);
    Validate.notNull(currentSalePolicyExecuteStrategy , "未匹配到业务编号为:%s 的优惠执行策略信息，可能是该政策不存在或者该优惠政策不允许关联这个执行政策，请检查!!" , executeStrategyCode);
    return currentSalePolicyExecuteStrategy;
  }
  
  /**
   * 该私有方法用于查询符合要求的限价政策（基于特定的优惠政策）
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> findCorrectLimitStrategy(SalePolicyTempleteRegister matchedSalePolicyTempleteRegister , String limitStrategyCode) {
    Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategies = this.findAllBindingLimitStrategy(matchedSalePolicyTempleteRegister);
    Validate.isTrue(!CollectionUtils.isEmpty(bindableLimitStrategies) , "未发现当前优惠政策有至少一个能绑定的限价政策，请检查!!");
    HashMap<String, SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy : bindableLimitStrategies) {
      bindableLimitStrategyMapping.put(salePolicyLimitStrategy.getLimitStrategyCode(), salePolicyLimitStrategy);
    }
    
    // 寻找正确的策略处理器，并进行保存处理
    Validate.notBlank(limitStrategyCode , "至少一个将要保存的现价策略没有指定code信息，请检查!!");
    SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = bindableLimitStrategyMapping.get(limitStrategyCode);
    Validate.notNull(currentSalePolicyLimitStrategy , "未匹配到业务编号为:%s 的限量政策信息，可能是该政策不存在或者改优惠政策不允许关联这个限量政策，请检查!!" , limitStrategyCode);
    return currentSalePolicyLimitStrategy;
  }
  
  /**
   * 该私有方法用于基于特定的优惠政策查询所有已绑定的限量政策
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> findAllBindingLimitStrategy(SalePolicyTempleteRegister matchedSalePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>>> bindableLimitStrategyClasses = matchedSalePolicyTempleteRegister.getBindableLimitStrategyClasses();
    if(CollectionUtils.isEmpty(bindableLimitStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些限价政策可以进行绑定
    Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyClass : bindableLimitStrategyClasses) {
      try {
        SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = applicationContext.getBean(bindableLimitStrategyClass);
        bindableLimitStrategySet.add(currentSalePolicyLimitStrategy);
      } catch(RuntimeException e) {
        log.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableLimitStrategySet;
  }

  @Override
  @Transactional
  public DefaultPolicyExecuteContext execute(String relevanceCode, Set<SalePolicyConProduct> initPolicyProducts, Map<String, String[]> selectPolicyCodeMapping, String tenantCode, String customerCode, boolean kneading) {
    /*
     * 正式计算的过程逻辑和预计算的过程逻辑基本相同，所以很多代码片段可以重用。
     * 两者最大的不同是，正式计算的结果流程需要被记录
     * */
    return this.execute(initPolicyProducts , selectPolicyCodeMapping, tenantCode, customerCode , kneading , true, relevanceCode);
  }
  
  /**
   * 正式执行和预执行的共用核心逻辑在这里（注意，该方法绝对不能返回null，即使参数不足，也要构建一个默认初始化的执行上下文）
   * @param formal 是否正式执行（false：预执行，true：正式执行）
   * @param relevanceCode 如果是正式执行，则必须传入对应的单据的业务编码信息
   */
  private DefaultPolicyExecuteContext execute(Set<SalePolicyConProduct> initPolicyProducts, Map<String, String[]> selectPolicyCodeMapping, String tenantCode, String customerCode, boolean kneading, boolean formal , String relevanceCode) {
    /*
     * 预执行和预执行过程，两者的对比是：
     * a、预处理过程不基于预判定过程，且预处理过程得到的优惠政策一定更接近实际执行的情况。
     * b、预处理过程瞄准“优惠政策维度”进行主要处理，然后在处理完成后，再对“商品本品维度”进行梳理
     * c、预执行不用保存执行结果（执行流水，但是正式执行过程需要）
     * 
     * 预处理过程是最关键的执行过程，处理逻辑如下：
     * 1、首先查询当前客户（经销商）正在执行的有效营销活动（注意是有效的），注意：这一步可以直接重用“预判定”功能的主方法
     * 
     * 2、判定selectPolicyCodeMapping集合中有没有客户传来的某些本品指定参与哪些优惠政策
     * 如果有，还要再预判定的基础上，再做一次计算前的数据准备工作，具体来说就是将指定本品参与优惠政策以外的关联去掉
     * 并按照这些执行策略的优先级，进行分组排序（注意：整单优惠政策一个排序组，商品优惠一个排序组）
     * 
     * 一、============= 以下进行商品优惠组的判定（每一个优惠政策依次进行如下计算）
     * 3、判定优惠政策的门槛（得到哪些门槛详情，以及对应的商品可以继续进行优惠）
     * 4、检查每一个门槛对应的商品信息都适用同一个优惠政策执行策略后，开始进行正式的计算
     * 5、基于最新的优惠政策执行上下文，进行选定的限量信息的预执行
     * 6、验证一个商品在一次优惠执行过程中，只能参与一种类型的优惠政策类型一次
     * 
     * 二、============= 以下进行整单优惠组的判定（每一个优惠政策依次进行如下计算）
     * 7、整单优惠政策，也应该判定总优惠门槛的问题
     * 8、检查每一个门槛对应的息都使用同一个优惠政策执行策略后，开始进行正式的计算
     * 9、基于最新的优惠政策执行上下文，进行选定的限量信息的预执行
     * 10、验证一个商品在一次优惠执行过程中，只能参与一种类型的优惠政策类型一次
     * 
     * 三、============= 以下过程是正式执行时需要额外处理的逻辑
     * 9、将优惠执行结果的流水进行记录
     * */
    
    // 1、=======
    Validate.notBlank(tenantCode, "进行优惠计算时，经销商信息必须传入!!");
    Validate.notBlank(customerCode, "进行优惠计算时，客户信息必须传入!!");
    Validate.isTrue(!CollectionUtils.isEmpty(initPolicyProducts), "进行优惠计算时，参与优惠的总体本品规格信息，必须传入!!");
    long initPolicyProductSpecificationCodeSize = initPolicyProducts.stream().map(SalePolicyConProduct::getProductCode).distinct().count();
    Validate.isTrue(initPolicyProductSpecificationCodeSize == initPolicyProducts.size(), "进行优惠计算时，参与优惠的总体本品规格信息，必须传入且不能重复!!");
    Set<SalePolicyVo> initSalePolicyies = this.salePolicyVoService.findDetailsByProcessingAndCustomerCode(tenantCode, true, customerCode , relevanceCode);
    // 说明当前租户、当前客户没有任何可执行的优惠政策，这是一种正常情况
    if(CollectionUtils.isEmpty(initSalePolicyies)) {
      return this.buildDefaultPolicyExecuteContext(null, initPolicyProducts, tenantCode, customerCode, kneading, formal, relevanceCode);
    }
    if(formal) {
      Validate.notBlank(relevanceCode , "当正式执行优惠政策计算时，发生优惠的业务单据编号必须传入，请检查!!");
      // 确认当前单据没有被执行任何优惠政策，否则需要报错
      Validate.isTrue(this.salePolicyRecordService.findByBillCode(relevanceCode, tenantCode, customerCode) == null , "当正式执行优惠政策计算时，发现指定单据已经执行过优惠，不能重复执行，请检查!!");
    }
    // 商品详情中的基础数据必须齐全
    for (SalePolicyConProduct initPolicyProductItem : initPolicyProducts) {
      BigDecimal quantity = initPolicyProductItem.getQuantity();
      BigDecimal unitPrice = initPolicyProductItem.getUnitPrice();
      BigDecimal subtotalAmount = initPolicyProductItem.getSubtotalAmount();
      Validate.isTrue(quantity != null && quantity.floatValue() > 0.0f, "进行优惠政策计算时，商品初始单价必须大于0！！");
      Validate.isTrue(unitPrice != null && unitPrice.floatValue() > 0, "进行优惠政策计算时，商品初始数量必须大于0！！");
      Validate.isTrue(subtotalAmount != null && subtotalAmount.floatValue() > 0, "进行优惠政策计算时，商品初始小计总价必须大于0！！");
    }
    DefaultPolicyExecuteContext executeContext = this.buildDefaultPolicyExecuteContext(initSalePolicyies, initPolicyProducts, tenantCode, customerCode , kneading , formal , relevanceCode);
    
    // 2、======
    if(!CollectionUtils.isEmpty(selectPolicyCodeMapping)) {
      /*
       * 处理过程如下：
       * a、检查集合中，这些本品指定参与的优惠政策是不是自己本身就允许参与的，如果不是则是数据错误，需要抛出异常
       * b、根据selectPolicyCodeMapping中设定的情况，将指定本品和其他优惠的映射关系去掉
       * */
      // a、======
      for (Map.Entry<String,String[]> item : selectPolicyCodeMapping.entrySet()) {
        String selectProductCode = item.getKey();
        String[] selectPolicyCodes = item.getValue();
        // 如果条件成立，就认为这个本品进行最大限度的优惠政策匹配
        if(selectPolicyCodes == null) {
          continue;
        } 
        // 如果条件成立，说明该本品不参与任何优惠政策
        if(selectPolicyCodes.length == 0) {
          selectPolicyCodes = new String[] {};
        }
        Set<String> selectPolicyCodeSet = Sets.newHashSet(selectPolicyCodes);
        Set<String> matchedPolicies = executeContext.findPolicyCodesByProductCode(selectProductCode);
        if(CollectionUtils.isEmpty(matchedPolicies)) {
          matchedPolicies = Sets.newLinkedHashSet();
        }
        // b、=====
        // 将存在于matchedPolicies集合中，但不存在于selectPolicyCodeSet集合中的所有优惠政策全部排出
        // 注意，Sets.difference中的集合引用基于matchedPolicies这个目标集合，所以要创建一个新的集合，以便边遍历边移除
        Set<String> mustExcludePolicyCodes = Sets.difference(matchedPolicies, selectPolicyCodeSet);
        if(CollectionUtils.isEmpty(mustExcludePolicyCodes)) {
          continue;
        }
        mustExcludePolicyCodes = Sets.newLinkedHashSet(mustExcludePolicyCodes);
        for (String mustExcludePolicyCode : mustExcludePolicyCodes) {
          executeContext.excludeMatchedPolicyMapping(selectProductCode, mustExcludePolicyCode);
        }
      }
    }
    
    // 步骤说明中的一、二大步，都在这个私有方法中，可参看
    DefaultPolicyExecuteContext preExecuteContext = this.doExecute(executeContext, initSalePolicyies, initPolicyProducts, tenantCode, customerCode , formal , relevanceCode);
    return preExecuteContext;
  }
  
  /**
   * 这是真正的预执行过程或者执行过程，其中的executeContext优惠政策执行上下文中的内容有可能直接来源于预判定逻辑，也可能又根据调用者传入的指定参与的优惠政策进行了一次过滤</br>
   * 执行过程参加public preExecute方法中的步骤说明
   * @param formal 是否正式执行（false：预执行，true：正式执行）
   * @param relevanceCode 如果是正式执行，则必须传入对应的单据的业务编码信息
   * @return
   */
  private DefaultPolicyExecuteContext doExecute(DefaultPolicyExecuteContext executeContext , Set<SalePolicyVo> initSalePolicyies , Set<SalePolicyConProduct> initPolicyProducts, String tenantCode, String customerCode , boolean formal , String relevanceCode) {
    // 建立以下两个映射，因为后续逻辑会用（一个是映射优惠政策编码和优惠政策详情、另一个是映射优惠政策编码和对应的优惠政策模板详情）
    Map<String , SalePolicyVo> initSalePolicyMapping = initSalePolicyies.stream().collect(Collectors.toMap(SalePolicyVo::getSalePolicyCode, item -> item));
    final Set<String> preJudgmentMatchedPolicyCodes = executeContext.getMatchedPolicyInfos();
    if(CollectionUtils.isEmpty(preJudgmentMatchedPolicyCodes)) {
      return executeContext;
    }
    Set<SalePolicyVo> preJudgmentSalePolicies = initSalePolicyies.stream().filter(salePolicyItem -> preJudgmentMatchedPolicyCodes.contains(salePolicyItem.getSalePolicyCode())).collect(Collectors.toSet());
    // 开始进行分组并排列执行顺序（策略注册机：SalePolicyRegister中sort越小的越先执行）
    List<String> sortedPreJudgmentSalePolicyCodes = this.sortPolicy(preJudgmentSalePolicies);
    
    // 2022-05-01 增加优惠特性执行前的干预判定：为优惠政策增加特性判定策略
    List<String> notExcludedSortedPreJudgmentSalePolicyCodes = Lists.newArrayList(sortedPreJudgmentSalePolicyCodes);
    if(!CollectionUtils.isEmpty(characteristicStrateis)) {
      for (CharacteristicStrategy<?> characteristicStrategy : characteristicStrateis) {
         for(int index = 0 ; index < notExcludedSortedPreJudgmentSalePolicyCodes.size() ; index++) {
           String currentSalePolicyCode = notExcludedSortedPreJudgmentSalePolicyCodes.get(index);
           SalePolicyVo preJudgmentSalePolicy = initSalePolicyMapping.get(currentSalePolicyCode);
           String characteristicCode = characteristicStrategy.characteristicCode();
           // 判定当前的优惠政策，是否设定了指定的特性执行策略characteristicCode
           Set<AbstractCharacteristicInfo> selectedCharacteristicInfos = preJudgmentSalePolicy.getCharacteristicInfos();
           if(CollectionUtils.isEmpty(selectedCharacteristicInfos)) {
             continue;
           }
           String[] selectedCharacteristicCodes = selectedCharacteristicInfos.stream().map(AbstractCharacteristicInfo::getCharacteristicCode).toArray(String[]::new);
           if(!StringUtils.equalsAny(characteristicCode, selectedCharacteristicCodes)) {
             continue;
           }
           // 开始进行判定
           CharacteristicVoter characteristicVoter = characteristicStrategy.make(executeContext, preJudgmentSalePolicy, customerCode, formal, relevanceCode);
           // 不排除任何操作
           if(characteristicVoter == null || characteristicVoter == CharacteristicVoter.GRANTED) {
             continue;
           } 
           // 排除自己
           else if(characteristicVoter == CharacteristicVoter.DENIED) {
             String errorMsg = characteristicStrategy.errorMsg(preJudgmentSalePolicy);
             executeContext.addExcludedPolicy(preJudgmentSalePolicy, errorMsg);
             notExcludedSortedPreJudgmentSalePolicyCodes.remove(index);
             index--;
             continue;
           }
           // 排除出自己以外的所有优惠政策
           else {
             String errorMsg = characteristicStrategy.errorMsg(preJudgmentSalePolicy);
             for (String notExcludedSortedPreJudgmentSalePolicyCode : notExcludedSortedPreJudgmentSalePolicyCodes) {
               if(!StringUtils.equals(notExcludedSortedPreJudgmentSalePolicyCode, currentSalePolicyCode)) {
                 SalePolicyVo exculudeSalePolicy = initSalePolicyMapping.get(notExcludedSortedPreJudgmentSalePolicyCode);
                 if(exculudeSalePolicy != null) {
                   executeContext.addExcludedPolicy(exculudeSalePolicy, errorMsg);
                 }
               } 
             } 
             notExcludedSortedPreJudgmentSalePolicyCodes.clear();
             notExcludedSortedPreJudgmentSalePolicyCodes.add(currentSalePolicyCode);
             break;
           }
        }
      }
    }
    
    // ======== 一、========
    // ====== 以下进行商品优惠组的判定（每一个优惠政策依次进行如下计算）
    TD: for (String sortedPreJudgmentSalePolicyCode : sortedPreJudgmentSalePolicyCodes) {
      SalePolicyVo preJudgmentSalePolicy = initSalePolicyMapping.get(sortedPreJudgmentSalePolicyCode);
      if(preJudgmentSalePolicy.getWholePolicy()) {
        continue;
      }
      // 查找最终参与这个优惠政策的本品信息
      Set<String> conProductCodes = executeContext.findProductCodesByPolicyCode(sortedPreJudgmentSalePolicyCode);
      if(CollectionUtils.isEmpty(conProductCodes)) {
        continue;
      }
      
      // 3、=======
      // 该变量表示在这个政策中，已经达到的优惠政策门槛编号（可能是一个，也可能是多个）, Key：门槛编号；value：达到这个门槛的商品本品业务编号
      SalePolicyTempleteRegister salePolicyTempleteRegister = this.findSalePolicyTempleteRegister(preJudgmentSalePolicy.getType());
      Pair<Map<String,Set<String>>,String> thresholdMatchedPair = this.validateSalePolicyThreshold(salePolicyTempleteRegister, executeContext, conProductCodes, preJudgmentSalePolicy);
      // 根据super.executeThreshold方法的工作特性，如果L侧的Map为空集合，那么R侧的错误提醒信息中，一定存在一条错误日志
      Map<String,Set<String>> thresholdMatchedMapping = thresholdMatchedPair.getLeft();
      String errorMsg = thresholdMatchedPair.getRight();
      // 条件成立，说明在当前的业务单据情况下，优惠政策的门槛并没有达到
      if(CollectionUtils.isEmpty(thresholdMatchedMapping)) {
        executeContext.addExcludedPolicy(preJudgmentSalePolicy, errorMsg);
        continue;
      } 
      // 否则说明该优惠政策中，至少有一组优惠门槛被全部或部分商品达到了
      else {
        Set<String> allMatchedProductCodes = Sets.newLinkedHashSet();
        for (Set<String> groupProductCodes : thresholdMatchedMapping.values()) {
          allMatchedProductCodes.addAll(groupProductCodes);
        }
        Set<String> initPolicyProductCodes = initPolicyProducts.stream().map(SalePolicyConProduct::getProductCode).collect(Collectors.toSet());
        executeContext.addNewMathedPolicy(preJudgmentSalePolicy, Sets.intersection(initPolicyProductCodes, allMatchedProductCodes).toArray(new String[] {}));
      }
      
      // 4、======
      // 首先检查每一个门槛对应的商品信息都使用同一个优惠政策执行策略
      Set<AbstractSalePolicyProductInfo> salePolicyProductInfos = preJudgmentSalePolicy.getSalePolicyProductInfos();
      Map<String , AbstractSalePolicyProductInfo> salePolicyProductInfoMapping = salePolicyProductInfos.stream().collect(Collectors.toMap(AbstractSalePolicyProductInfo::getProductCode, item->item));
      // productInfoExecutorGroupMapping表示按照本品信息所使用的优惠政策执行策略业务编码（executorCode）进行的分组，使用executorCode就可以找到AbstractSalePolicyExecutorInfo信息
      // 进而找到“阶梯循环规则”（cycleRuleStrategy）, 另外注意，这里得到的productInfoExecutorGroupMapping中的本品信息，一定存在于参与本次优惠执行的本品集合（initPolicyConProductCodes）中
      Map<String , List<AbstractSalePolicyProductInfo>> productInfoExecutorGroupMapping = Maps.newLinkedHashMap();
      for (Map.Entry<String,Set<String>> matchedThresholdItem : thresholdMatchedMapping.entrySet()) {
        Set<String> matchedThresholdProductCodes = matchedThresholdItem.getValue();
        Set<String> matchedProductCodes = salePolicyProductInfoMapping.entrySet().stream().filter(entry -> matchedThresholdProductCodes.contains(entry.getKey())).map(entry -> entry.getKey()).collect(Collectors.toSet());
        if(matchedProductCodes == null || matchedThresholdProductCodes.size() != matchedProductCodes.size()) {
          executeContext.addExcludedPolicy(preJudgmentSalePolicy, "优惠门槛中的商品信息和实际本品信息发生了冲突，请联系管理员");
          continue TD;
        }
        Map<String , List<AbstractSalePolicyProductInfo>> currentProductInfoExecutorGroupMapping = salePolicyProductInfoMapping.entrySet().stream()
            .filter(entry -> matchedThresholdProductCodes.contains(entry.getKey()))
            .filter(entry -> conProductCodes.contains(entry.getValue().getProductCode()))
            .map(entry -> entry.getValue()).collect(Collectors.groupingBy(AbstractSalePolicyProductInfo::getExecutorCode));
        if(currentProductInfoExecutorGroupMapping.size() != 1) {
          executeContext.addExcludedPolicy(preJudgmentSalePolicy, "优惠门槛中的商品信息并不属于同一种执行策略，请联系管理员");
          continue TD;
        }
        // 如果条件成立，说明当前执行编号进行的本品分组中，并没有任何本品涉及本次执行
        if(CollectionUtils.isEmpty(currentProductInfoExecutorGroupMapping.entrySet().iterator().next().getValue())) {
          continue TD;
        }
        productInfoExecutorGroupMapping.putAll(currentProductInfoExecutorGroupMapping);
      }
      // 开始进行每一个执行器的执行过程，其中最重要的是取得其中的阶梯循环规则（SalePolicyCycleRuleStrategy）
      TG: for (Map.Entry<String , List<AbstractSalePolicyProductInfo>> productInfoExecutorGroupItem : productInfoExecutorGroupMapping.entrySet()) {
        String executorCode = productInfoExecutorGroupItem.getKey();
        List<AbstractSalePolicyProductInfo> executorGroupProductInfos = productInfoExecutorGroupItem.getValue();
        Set<String> executorGroupProductCodes = executorGroupProductInfos.stream().map(AbstractSalePolicyProductInfo::getProductCode).collect(Collectors.toSet());
        AbstractSalePolicyExecutorInfo salePolicyExecutorInfo = this.findCorrectExecutorInfo(preJudgmentSalePolicy, executorCode);
        if(salePolicyExecutorInfo == null) {
          log.warn("在优惠政策(%s)中，没有找到业务编号为(%s)的优惠政策执行策略信息，请检查优惠政策设定", preJudgmentSalePolicy.getSalePolicyCode() , executorCode);
          continue TG;
        }
        SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy = this.findCorrectExecuteStrategy(salePolicyTempleteRegister, salePolicyExecutorInfo.getExecuteStrategyCode());
        SalePolicyCycleRuleStrategy salePolicyCycleRuleStrategy = this.findCorrectCycleRuleStrategy(salePolicyExecuteStrategy, salePolicyExecutorInfo.getCycleRuleCode());
        // 创建阶梯循环执行上下文
        Set<SalePolicyConProduct> initPolicyProductsForCycleExecutes = initPolicyProducts.stream().filter(item -> executorGroupProductCodes.contains(item.getProductCode())).collect(Collectors.toSet());
        AbstractCycleExecuteContext cycleExecuteContext = salePolicyExecuteStrategy.buildCycleExecuteContext(executeContext , executorCode, initPolicyProductsForCycleExecutes, customerCode);
        // 开始阶梯循环计算过程
        salePolicyCycleRuleStrategy.cycle(cycleExecuteContext, salePolicyExecuteStrategy , preJudgmentSalePolicy , salePolicyExecutorInfo);
        // 接着需要将整个优惠政策计算完成后的优惠结果分摊到参与优惠的每个商品上
        // 换句话说：需要按照本次优惠政策的执行结果上下文cycleExecuteContext，构造最新的整体优惠政策执行上下文policyExecuteContext
        Class<? extends SalePolicyExecuteShareStrategy> salePolicyExecuteShareStrategyClass = salePolicyExecuteStrategy.getSalePolicyExecuteShareStrategy();
        SalePolicyExecuteShareStrategy salePolicyExecuteShareStrategy = this.applicationContext.getBean(salePolicyExecuteShareStrategyClass);
        salePolicyExecuteShareStrategy.share(customerCode , preJudgmentSalePolicy , executeContext, cycleExecuteContext, executorGroupProductCodes);
        
        // 5、======
        // 注意执行限量策略的位置，由于商品性质的优惠会政策，可能一次涉及多组阶梯的执行，所以每组阶梯执行后，都要进行限量控制
        Set<AbstractSalePolicyLimitInfo> salePolicyLimitInfos = preJudgmentSalePolicy.getSalePolicyLimitInfos();
        if(!CollectionUtils.isEmpty(salePolicyLimitInfos)) {
          for (AbstractSalePolicyLimitInfo salePolicyLimitInfo : salePolicyLimitInfos) {
            String limitStrategyCode = salePolicyLimitInfo.getLimitStrategyCode();
            SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy = this.findCorrectLimitStrategy(salePolicyTempleteRegister, limitStrategyCode);
            Validate.notNull(salePolicyLimitStrategy , "优惠政策预执行时，发现错误的限量设定信息（%s），请检查!!" , limitStrategyCode);
            // 只要有一个这个优惠政策的限量设定验证没有通过（抛出了异常），则需要重新设定这个已经满足的优惠政策为“不匹配”
            try {
              salePolicyLimitStrategy.preValidate(executeContext , cycleExecuteContext , preJudgmentSalePolicy , salePolicyLimitInfo);
            } catch(RuntimeException e) {
              executeContext.resetExcludedPolicy(sortedPreJudgmentSalePolicyCode , executorGroupProductCodes , e.getMessage());
              continue TG;
            }
          }
        }
      }
      // 6、======
      try {
        this.analysisRepeatSalePolicy(preJudgmentSalePolicy, executeContext, conProductCodes);
      } catch(RuntimeException e) {
        executeContext.resetExcludedPolicy(sortedPreJudgmentSalePolicyCode, conProductCodes , e.getMessage());
        continue TD;
      }
    }
    
    // ======== 二、========
    TD: for (String sortedPreJudgmentSalePolicyCode : sortedPreJudgmentSalePolicyCodes) {
      SalePolicyVo preJudgmentSalePolicy = initSalePolicyMapping.get(sortedPreJudgmentSalePolicyCode);
      if(!preJudgmentSalePolicy.getWholePolicy()) {
        continue;
      }
      // 取得可以参与这个整单优惠的商品本品信息(因为虽然是整单优惠，但由于操作者的设定，指定了某些本品不参与整单优惠)
      Set<String> conProductCodes = executeContext.findProductCodesByPolicyCode(sortedPreJudgmentSalePolicyCode);
      // 如果条件成立，说明这个优惠政策虽然理论上默认命中所有本品，但由于操作者的设定，该次执行没有任何一个物资实际要求进行享受
      if(CollectionUtils.isEmpty(conProductCodes)) {
        continue;
      }
      Set<SalePolicyConProduct> initPolicyProductsForWholePolicy = initPolicyProducts.stream().filter(initPolicyProduct -> conProductCodes.contains(initPolicyProduct.getProductCode())).collect(Collectors.toSet());
      
      // 7、======
      // 该变量表示在这个政策中，已经达到的优惠政策门槛编号（可能是一个，也可能是多个）, Key：门槛编号；value：达到这个门槛的商品本品业务编号
      SalePolicyTempleteRegister salePolicyTempleteRegister = this.findSalePolicyTempleteRegister(preJudgmentSalePolicy.getType());
      Pair<Map<String,Set<String>>,String> thresholdMatchedPair = this.validateSalePolicyThreshold(salePolicyTempleteRegister, executeContext, conProductCodes , preJudgmentSalePolicy);
      // 根据super.executeThreshold方法的工作特性，如果L侧的Map为空集合，那么R侧的错误提醒信息中，一定存在一条错误日志
      Map<String,Set<String>> thresholdMatchedMapping = thresholdMatchedPair.getLeft();
      String errorMsg = thresholdMatchedPair.getRight();
      // 条件成立，说明在当前的业务单据情况下，优惠政策的门槛并没有达到（注意：整单优惠不会出现多组的情况）
      if(CollectionUtils.isEmpty(thresholdMatchedMapping)) {
        executeContext.addExcludedPolicy(preJudgmentSalePolicy, errorMsg);
        continue;
      } 
      
      // 8、======
      Set<AbstractSalePolicyExecutorInfo> salePolicyExecutorInfos = preJudgmentSalePolicy.getSalePolicyExecutorInfos();
      if(CollectionUtils.isEmpty(salePolicyExecutorInfos)) {
        log.warn("在整单优惠政策(%s)中，没有发现优惠执行信息，请检查优惠政策设定");
        continue;
      }
      // 整单优惠中，只可能有一条优惠策略执行信息
      if(salePolicyExecutorInfos.size() != 1) {
        log.warn("在整单优惠政策(%s)中，发现多条优惠执行信息，请检查优惠政策设定");
        continue;
      }
      AbstractSalePolicyExecutorInfo salePolicyExecutorInfo = salePolicyExecutorInfos.iterator().next();
      SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy = this.findCorrectExecuteStrategy(salePolicyTempleteRegister, salePolicyExecutorInfo.getExecuteStrategyCode());
      SalePolicyCycleRuleStrategy salePolicyCycleRuleStrategy = this.findCorrectCycleRuleStrategy(salePolicyExecuteStrategy, salePolicyExecutorInfo.getCycleRuleCode());
      BigDecimal initSubtotalAmountForCycleExecutes = BigDecimal.ZERO;
      BigDecimal initSubtotalNumberForCycleExecutes = BigDecimal.ZERO;
      BigDecimal preSubtotalAmountForCycleExecutes = BigDecimal.ZERO;
      BigDecimal preSurplusSubtotalAmountForCycleExecutes = BigDecimal.ZERO;
      BigDecimal preSurplusSubtotalNumberForCycleExecutes = BigDecimal.ZERO;
      for (SalePolicyConProduct initPolicyProductForWholePolicy : initPolicyProductsForWholePolicy) {
        ProductPolicyStepResult productPolicyStepResult = executeContext.findLastProductStepResultByProductCode(initPolicyProductForWholePolicy.getProductCode());
        initSubtotalAmountForCycleExecutes = initSubtotalAmountForCycleExecutes.add(productPolicyStepResult.getInitSubtotal());
        initSubtotalNumberForCycleExecutes = initSubtotalNumberForCycleExecutes.add(new BigDecimal(productPolicyStepResult.getInitNumbers()));
        preSubtotalAmountForCycleExecutes = preSubtotalAmountForCycleExecutes.add(productPolicyStepResult.getLastSubtotal());
        preSurplusSubtotalAmountForCycleExecutes = preSurplusSubtotalAmountForCycleExecutes.add(productPolicyStepResult.getLastSurplusTotalAmount());
        preSurplusSubtotalNumberForCycleExecutes = preSurplusSubtotalNumberForCycleExecutes.add(new BigDecimal(productPolicyStepResult.getLastSurplusTotalNumber()));
      }
      // 整单优惠，都是这个单据上的所有本品信息，都参与
      DefaultCycleExecuteContext cycleExecuteContext = new DefaultCycleExecuteContext(salePolicyExecutorInfo.getExecutorCode() , customerCode ,initPolicyProductsForWholePolicy , initSubtotalAmountForCycleExecutes, initSubtotalNumberForCycleExecutes.intValue(),
        preSubtotalAmountForCycleExecutes , preSurplusSubtotalAmountForCycleExecutes , preSurplusSubtotalNumberForCycleExecutes.intValue());
      // 开始阶梯循环计算过程
      salePolicyCycleRuleStrategy.cycle(cycleExecuteContext, salePolicyExecuteStrategy , preJudgmentSalePolicy , salePolicyExecutorInfo);
      // 接着需要将整个优惠政策计算完成后的优惠结果分摊到参与优惠的每个商品上
      // 换句话说：需要按照本次优惠政策的执行结果上下文cycleExecuteContext，构造最新的整体优惠政策执行上下文policyExecuteContext
      Class<? extends SalePolicyExecuteShareStrategy> salePolicyExecuteShareStrategyClass = salePolicyExecuteStrategy.getSalePolicyExecuteShareStrategy();
      SalePolicyExecuteShareStrategy salePolicyExecuteShareStrategy = this.applicationContext.getBean(salePolicyExecuteShareStrategyClass);
      salePolicyExecuteShareStrategy.share(customerCode , preJudgmentSalePolicy , executeContext, cycleExecuteContext, initPolicyProductsForWholePolicy.stream().map(SalePolicyConProduct::getProductCode).collect(Collectors.toSet()));
      
      // 9、======
      Set<AbstractSalePolicyLimitInfo> salePolicyLimitInfos = preJudgmentSalePolicy.getSalePolicyLimitInfos();
      if(!CollectionUtils.isEmpty(salePolicyLimitInfos)) {
        for (AbstractSalePolicyLimitInfo salePolicyLimitInfo : salePolicyLimitInfos) {
          String limitStrategyCode = salePolicyLimitInfo.getLimitStrategyCode();
          SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy = this.findCorrectLimitStrategy(salePolicyTempleteRegister, limitStrategyCode);
          Validate.notNull(salePolicyLimitStrategy , "优惠政策预执行时，发现错误的限量设定信息（%s），请检查!!" , limitStrategyCode);
          // 只要有一个这个优惠政策的限量设定验证没有通过（抛出了异常），则需要重新设定这个已经满足的优惠政策为“不匹配”
          try {
            salePolicyLimitStrategy.preValidate(executeContext , cycleExecuteContext , preJudgmentSalePolicy , salePolicyLimitInfo);
          } catch(RuntimeException e) {
            executeContext.resetExcludedPolicy(sortedPreJudgmentSalePolicyCode , null, e.getMessage());
            continue TD;
          }
        }
      }
      // 10、======
      try {
        this.analysisRepeatSalePolicy(preJudgmentSalePolicy, executeContext, conProductCodes);
      } catch(RuntimeException e) {
        executeContext.resetExcludedPolicy(sortedPreJudgmentSalePolicyCode , conProductCodes, e.getMessage());
        continue TD;
      }
    }
    
    if(!formal) {
      return executeContext;
    }
    
    // 三、======= 如果是正式执行，则记录优惠执行的流水信息
    // 9、======
    this.salePolicyRecordService.create(relevanceCode, executeContext);
    return executeContext;
  }
  
  /**
   * 判定这些参与本次优惠政策执行的商品（executorGroupProductInfos），在历史上并没有参与和本次优惠政策模板类型相同的优惠政策
   * @param salePolicyProductCodes
   */
  private void analysisRepeatSalePolicy(SalePolicyVo salePolicy , DefaultPolicyExecuteContext executeContext , Set<String> salePolicyProductCodes) {
    String salePolicyCode = salePolicy.getSalePolicyCode();
    String templeteRegisterType = salePolicy.getTempleteRegisterType();
    // 一般这种情况不会发生，但如果发生说明数据存在问题
    if(StringUtils.isBlank(templeteRegisterType)) {
      return;
    }
    for (String salePolicyProductCode : salePolicyProductCodes) {
      Deque<ProductPolicyStepResult> productPolicyStepResults  = executeContext.findPolicyStepResultsByProductCode(salePolicyProductCode);
      if(CollectionUtils.isEmpty(productPolicyStepResults)) {
        continue;
      }
      // 首先要确定，这个本品是真实产生了salePolicy所对应的步进
      ProductPolicyStepResult peek = productPolicyStepResults.peek();
      Set<String> salePolicyExecutorCodes = salePolicy.getSalePolicyExecutorInfos().stream().map(AbstractSalePolicyExecutorInfo::getExecutorCode).collect(Collectors.toSet());
      if(!StringUtils.equals(salePolicy.getSalePolicyCode(), peek.getSalePolicyCode()) || !salePolicyExecutorCodes.contains(peek.getExecuteCode())) {
        continue;
      }
      // 一个一个进行“templeteRegisterType是否重复”的判定
      for (ProductPolicyStepResult productPolicyStepResult : productPolicyStepResults) {
        String currentSalePolicyCode = productPolicyStepResult.getSalePolicyCode();
        // 自身和自身不需要对比，初始化步进也不需要对比
        if(StringUtils.isBlank(currentSalePolicyCode) || StringUtils.equals(currentSalePolicyCode, salePolicyCode)) {
          continue;
        }
        SalePolicyVo currentSalePolicyVo = this.salePolicyVoService.findDetailsByCode(currentSalePolicyCode);
        String currentTempleteRegisterType = currentSalePolicyVo.getTempleteRegisterType();
        // 如果条件成立，说明和当前优惠政策同一类型的其他优惠政策，已经被享受过
        // 需要将本次优惠政策的所有执行结果进行回退，并明确错误信息给前端调用者
        if(StringUtils.equals(currentTempleteRegisterType, templeteRegisterType)) {
          throw new IllegalArgumentException("同一种类型的优惠政策不能被重复参与");
        }
      }
    }
  }

  @Override
  public DefaultPolicyExecuteContext findByRelevanceCode(String relevanceCode, String tenantCode, String customerCode) {
    if(StringUtils.isAnyBlank(relevanceCode , tenantCode , customerCode)) {
      return null;
    }
    
    /*
     * 查询指定业务单据的历史优惠政策执行情况的过程如下：
     * 1、查询指定relevanceCode、tenantCode、customerCode是否存在优惠政策正式执行流水。
     * 如果不存在则返回null；存在则返回详细情况
     * 2、首先进行上下文基本信息的还原，主要为initPolicyProducts、initSalePolicys这两个属性
     * 3、根据可能已记录的“被排除的优惠政策”信息，进行excludedPolicyMapping部分信息的还原
     * 4、根据SalePolicyRecord对象中的salePolicyStepCodes属性，进行每一个商品明细步进的还原
     * */
    
    // 1、========
    SalePolicyRecord historySalePolicyRecord = this.salePolicyRecordService.findByBillCode(relevanceCode, tenantCode, customerCode);
    if(historySalePolicyRecord == null) {
      return null;
    }
    Set<SalePolicyRecordProduct> salePolicyRecordProducts = historySalePolicyRecord.getSalePolicyRecordProducts();
    if(CollectionUtils.isEmpty(salePolicyRecordProducts)) {
      return null;
    }
    // 找到INIT性质的步进，后续构建SalePolicyConProduct对象需要
    Map<String, SalePolicyRecordProduct> salePolicyRecordInitProductMapping = salePolicyRecordProducts.stream()
        .filter(item -> StringUtils.equals(item.getStepType(), StepType.INIT.toString()))
        .collect(Collectors.toMap(SalePolicyRecordProduct::getProductCode, item -> item));
    // 可能存在的某个（或多个）商品指定执行的优惠政策信息，在这里记录，有序会使用
    Set<SalePolicyRecordProductSelectedPolicy> salePolicyRecordProductSelectedPolicys = historySalePolicyRecord.getSalePolicyRecordProductSelectedPolicys();
    Map<String, SalePolicyRecordProductSelectedPolicy> salePolicyRecordProductSelectedPolicyMapping = Maps.newLinkedHashMap();
    if(!CollectionUtils.isEmpty(salePolicyRecordProductSelectedPolicys)) {
      salePolicyRecordProductSelectedPolicyMapping = salePolicyRecordProductSelectedPolicys.stream().collect(Collectors.toMap(SalePolicyRecordProductSelectedPolicy::getProductCode, item -> item));
    }
    
    // 2、========
    // 构建初始化的initPolicyProducts信息
    String initProductCodes = historySalePolicyRecord.getProductCodes();
    if(StringUtils.isBlank(initProductCodes)) {
      log.warn("重建优惠政策执行历史上下文时，发现错误的initProductCodes信息，请检查对应的流水记录!!");
      return null;
    }
    String[] initProductCodeArray = StringUtils.split(initProductCodes , ",");
    Set<SalePolicyConProduct> initPolicyProducts = Sets.newLinkedHashSet();
    for (String initProductCode : initProductCodeArray) {
      SalePolicyRecordProduct salePolicyRecordProduct = salePolicyRecordInitProductMapping.get(initProductCode);
      // 如果条件成立，说明数据存在问题
      if(salePolicyRecordProduct == null) {
        log.warn("重建优惠政策执行历史上下文时，发现错误的salePolicyRecordProduct信息，请检查对应的流水记录!!");
        return null;
      }
      SalePolicyConProduct defaultPolicyConProduct = new SalePolicyConProduct();
      defaultPolicyConProduct.setBillItemCode(salePolicyRecordProduct.getBillItemCode());
      defaultPolicyConProduct.setProductCode(salePolicyRecordProduct.getProductCode());
      defaultPolicyConProduct.setProductName(salePolicyRecordProduct.getProductName());
      defaultPolicyConProduct.setQuantity(new BigDecimal(salePolicyRecordProduct.getInitNumbers()));
      // 如果这个商品在历史执行时，曾指定了特定执行的优惠政策，则在这里进行还原
      SalePolicyRecordProductSelectedPolicy currentSalePolicyRecordProductSelectedPolicy = null;
      if((currentSalePolicyRecordProductSelectedPolicy = salePolicyRecordProductSelectedPolicyMapping.get(initProductCode)) != null) {
        String selectedPolicyCodes = currentSalePolicyRecordProductSelectedPolicy.getSelectedPolicyCodes();
        defaultPolicyConProduct.setSelectedPolicyCodes(StringUtils.split(selectedPolicyCodes , ","));
      } else {
        defaultPolicyConProduct.setSelectedPolicyCodes(null);
      }
      defaultPolicyConProduct.setSubtotalAmount(salePolicyRecordProduct.getLastSubtotal());
      defaultPolicyConProduct.setUnitPrice(salePolicyRecordProduct.getLastPrices());
      initPolicyProducts.add(defaultPolicyConProduct);
    }
    // 构建初始化的initSalePolicys信息
    String salePolicyCodes = historySalePolicyRecord.getSalePolicyCodes();
    if(StringUtils.isBlank(salePolicyCodes)) {
      log.warn("重建优惠政策执行历史上下文时，发现错误的salePolicyCodes信息，请检查对应的流水记录!!");
      return null;
    }
    String[] initSalePolicyCodeArray = StringUtils.split(salePolicyCodes , ",");
    Set<SalePolicyVo> initSalePolicys = Sets.newLinkedHashSet();
    for (String initSalePolicyCode : initSalePolicyCodeArray) {
      SalePolicyVo initSalePolicy = this.salePolicyVoService.findDetailsByCode(initSalePolicyCode);
      if(initSalePolicy == null) {
        log.warn("重建优惠政策执行历史上下文时，发现错误的initSalePolicyCode信息，请检查对应的流水记录(很可能是数据错误)!!");
        return null;
      }
      initSalePolicys.add(initSalePolicy);
    }
    // 映射初始化的优惠政策信息，后续要使用
    Map<String , SalePolicyVo> initSalePolicyMapping = initSalePolicys.stream().collect(Collectors.toMap(SalePolicyVo::getSalePolicyCode , item->item));
    // 寻找初始化的initTotalAmount、initTotalNumber、kneading
    BigDecimal initTotalAmount = historySalePolicyRecord.getInitTotalAmount();
    Integer initTotalNumber = historySalePolicyRecord.getInitTotalNumber();
    boolean kneading = historySalePolicyRecord.getKneading();
    DefaultPolicyExecuteContext policyExecuteContext = new DefaultPolicyExecuteContext(initPolicyProducts, initSalePolicys, initTotalAmount, initTotalNumber, tenantCode, customerCode, kneading , true , relevanceCode);
    
    // 3、======
    Set<SalePolicyRecordExcludePolicy> salePolicyRecordExcludePolicys = historySalePolicyRecord.getSalePolicyRecordExcludePolicys();
    if(!CollectionUtils.isEmpty(salePolicyRecordExcludePolicys)) {
      for (SalePolicyRecordExcludePolicy salePolicyRecordExcludePolicy : salePolicyRecordExcludePolicys) {
        String salePolicyCode = salePolicyRecordExcludePolicy.getPolicyCode();
        String excludedMsg = salePolicyRecordExcludePolicy.getExcludedMsg();
        SalePolicyVo currentSalePolicyVo = initSalePolicyMapping.get(salePolicyCode);
        if(currentSalePolicyVo.getWholePolicy() == null || StringUtils.isBlank(excludedMsg)) {
          log.warn("重建优惠政策执行历史上下文时，发现错误的salePolicyCode/excludedMsg信息，请检查对应的流水记录!!");
          return null;
        }
        policyExecuteContext.addExcludedPolicy(currentSalePolicyVo, excludedMsg);
      }
    } 
    
    // 4、======
    // 将salePolicyRecordProducts集合按照商品编号进行分组，以便后续过程使用
    Map<String, List<SalePolicyRecordProduct>> salePolicyRecordProductGroupNoSortedMapping = salePolicyRecordProducts.stream().collect(Collectors.groupingBy(SalePolicyRecordProduct::getProductCode));
    // 这是没有排序的，还需要进行排序
    Map<String, List<SalePolicyRecordProduct>> salePolicyRecordProductGroupMapping = Maps.newLinkedHashMap();
    for (String productCode : salePolicyRecordProductGroupNoSortedMapping.keySet()) {
      List<SalePolicyRecordProduct> salePolicyRecordProductNoSorts = salePolicyRecordProductGroupNoSortedMapping.get(productCode);
      List<SalePolicyRecordProduct> salePolicyRecordProductSorts = salePolicyRecordProductNoSorts.stream().sorted((source , target) -> source.getSorted() - target.getSorted()).collect(Collectors.toList());
      salePolicyRecordProductGroupMapping.put(productCode, salePolicyRecordProductSorts);
    }
    Set<SalePolicyRecordStepPolicy> salePolicyRecordStepPolicySets = historySalePolicyRecord.getSalePolicyRecordStepPolicys();
    // 注意：要排除INTI性质的步进，还要按照sorted进行排序
    List<SalePolicyRecordStepPolicy> salePolicyRecordStepPolicys = salePolicyRecordStepPolicySets.stream()
        .filter(item -> !StringUtils.equals(item.getStepType(), StepType.INIT.toString()))
        .sorted((source , target) -> source.getSorted() - target.getSorted())
        .collect(Collectors.toList());
    if(CollectionUtils.isEmpty(salePolicyRecordStepPolicys)) {
      log.warn("重建优惠政策执行历史上下文时，未发现正确的优惠政策执行信息!!");
      return null;
    }
    // 现在根据历史上记录的优惠政策执行步进，还原上下文
    for (SalePolicyRecordStepPolicy salePolicyRecordStepPolicy : salePolicyRecordStepPolicys) {
      String salePolicyCode = salePolicyRecordStepPolicy.getSalePolicyCode();
      String executeCode = salePolicyRecordStepPolicy.getExecuteCode();
      Set<ProductPolicyStepResult> productStepResults = Sets.newLinkedHashSet();
      for (String productCode : salePolicyRecordProductGroupMapping.keySet()) {
        // 从历史流水记录中，找到正确的流水(productCode和executeCode都要匹配)，只可能最多有一条满足salePolicyCode、executeCode和productCode的步进记录
        List<SalePolicyRecordProduct> salePolicyRecordProductSteps = salePolicyRecordProductGroupMapping.get(productCode);
        SalePolicyRecordProduct nowSalePolicyRecordProduct = salePolicyRecordProductSteps.stream()
            .filter(item -> StringUtils.equals(executeCode, item.getExecuteCode()) && StringUtils.equals(salePolicyCode, item.getSalePolicyCode()))
            .findFirst()
            .orElse(null);
        if(nowSalePolicyRecordProduct == null) {
          continue;
        }
        int index = salePolicyRecordProductSteps.indexOf(nowSalePolicyRecordProduct);
        // 如果条件成立，说明找到了这个商品和指定优惠政策匹配的步进（这个index一定不为0）
        // 原因是index == 0 的，都是INIT性质的步进，不会有getSalePolicyCode()方法不会有返回值
        ProductPolicyStepResult productPolicyStepResult = new ProductPolicyStepResult();
        productPolicyStepResult.setProductCode(productCode);
        productPolicyStepResult.setSalePolicyCode(salePolicyCode);
        productPolicyStepResult.setExecuteCode(executeCode);
        StepType stepType = StepType.valueOf(nowSalePolicyRecordProduct.getStepType());
        if(stepType == null) {
          log.warn("重建优惠政策执行历史上下文时，发现错误的stepType信息，请检查对应的流水记录!!");
          return null;
        }
        productPolicyStepResult.setStepType(stepType);
        productPolicyStepResult.setInitNumbers(nowSalePolicyRecordProduct.getInitNumbers());
        productPolicyStepResult.setInitPrices(nowSalePolicyRecordProduct.getInitPrices());
        productPolicyStepResult.setInitSubtotal(nowSalePolicyRecordProduct.getInitSubtotal());
        // 重建pre部分的信息，pre部分的信息就是本商品上一个步进的last信息
        SalePolicyRecordProduct previousSalePolicyRecordProduct = salePolicyRecordProductSteps.get(index - 1);
        // 可能的赠品pre信息还需要构建
        Set<SalePolicyRecordGift> previousSalePolicyRecordGifts = previousSalePolicyRecordProduct.getSalePolicyRecordGifts();
        List<GiftResultInfo> preGifts = Lists.newArrayList();
        if(!CollectionUtils.isEmpty(previousSalePolicyRecordGifts)) {
          for (SalePolicyRecordGift previousSalePolicyRecordGift : previousSalePolicyRecordGifts) {
            GiftResultInfo preGift = new GiftResultInfo();
            preGift.setProductCode(previousSalePolicyRecordGift.getGiftCode());
            preGift.setProductName(previousSalePolicyRecordGift.getGiftName());
            preGift.setQuantity(previousSalePolicyRecordGift.getQuantity());
            preGift.setSubtotalAmount(previousSalePolicyRecordGift.getSubtotalAmount());
            preGifts.add(preGift);
          }
        } 
        productPolicyStepResult.setPreGifts(preGifts);
        productPolicyStepResult.setPrePrices(previousSalePolicyRecordProduct.getLastPrices());
        productPolicyStepResult.setPreSubtotal(previousSalePolicyRecordProduct.getLastSubtotal());
        productPolicyStepResult.setPreSurplusTotalAmount(previousSalePolicyRecordProduct.getLastSurplusTotalAmount());
        productPolicyStepResult.setPreSurplusTotalNumber(previousSalePolicyRecordProduct.getLastSurplusTotalNumber());
        // 重建last部分的信息
        // 可能的赠品last信息还需要构建
        Set<SalePolicyRecordGift> lastSalePolicyRecordGifts = nowSalePolicyRecordProduct.getSalePolicyRecordGifts();
        List<GiftResultInfo> lastGifts = Lists.newArrayList();
        if(!CollectionUtils.isEmpty(lastSalePolicyRecordGifts)) {
          for (SalePolicyRecordGift lastSalePolicyRecordGift : lastSalePolicyRecordGifts) {
            GiftResultInfo lastGift = new GiftResultInfo();
            lastGift.setProductCode(lastSalePolicyRecordGift.getGiftCode());
            lastGift.setProductName(lastSalePolicyRecordGift.getGiftName());
            lastGift.setQuantity(lastSalePolicyRecordGift.getQuantity());
            lastGift.setSubtotalAmount(lastSalePolicyRecordGift.getSubtotalAmount());
            lastGifts.add(lastGift);
          }
        }
        productPolicyStepResult.setLastGifts(lastGifts);
        productPolicyStepResult.setLastPrices(nowSalePolicyRecordProduct.getLastPrices());
        productPolicyStepResult.setLastSubtotal(nowSalePolicyRecordProduct.getLastSubtotal());
        productPolicyStepResult.setLastSurplusTotalAmount(nowSalePolicyRecordProduct.getLastSurplusTotalAmount());
        productPolicyStepResult.setLastSurplusTotalNumber(nowSalePolicyRecordProduct.getLastSurplusTotalNumber());
        productStepResults.add(productPolicyStepResult);
      }
      
      // 将汇总的这个步进的商品明细状态，加入到优惠政策上下文中
      // 注意，只有productStepResults集合存在步进信息时，才说明这个优惠政策被匹配了，需要执行
      if(CollectionUtils.isEmpty(productStepResults)) {
        continue;
      }
      SalePolicyVo currentSalePolicyVo = initSalePolicyMapping.get(salePolicyCode);
      if(currentSalePolicyVo.getWholePolicy() == null) {
        log.warn("重建优惠政策执行历史上下文时，发现错误的wholePolicy信息，请检查对应的流水记录!!");
        return null;
      }
      // 另外，根据当前优惠政策是整单优惠还是商品优惠，调用的addNewMathedPolicy方法也不一样
      if(currentSalePolicyVo.getWholePolicy()) {
        policyExecuteContext.addNewMathedPolicy(currentSalePolicyVo);
      } else {
        String[] matchedProductCodes = productStepResults.stream().map(ProductPolicyStepResult::getProductCode).distinct().toArray(String[]::new);
        policyExecuteContext.addNewMathedPolicy(currentSalePolicyVo, matchedProductCodes);
      }
      policyExecuteContext.addPolicyStepResult(productStepResults);
    }
    return policyExecuteContext;
  }
}
