package com.biz.crm.dms.business.policy.local.service.internal;

import com.biz.crm.dms.business.policy.local.service.task.SalePolicyCustomerScopeMatchedTask;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyMatchedStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyProductInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * SalePolicyMatchedStrategy，业务单据在执行时，其和优惠政策的匹配策略在标品中的默认实现</p>
 * 
 * 标品中的默认匹配策略，只匹配其中客户和商品本身的限制
 * 
 * @author yinwenjie
 */
@Component
public class SalePolicyMatchedStrategyImpl implements SalePolicyMatchedStrategy {

  @Autowired(required = false)
  private ApplicationContext applicationContext;
  /**
   * 用于优惠政策客户范围匹配的执行线程池
   * 和SalePolicyCacheLoadingService服务使用的执行线程池一样
   */
  @Autowired(required = false)
  @Qualifier("policyLoadingExecutor")
  private ThreadPoolExecutor policyLoadingExecutor;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyMatchedStrategyImpl.class);
  
  @Override
  public Set<SalePolicyVo> match(Set<SalePolicyVo> currentSalePolices, String tenantCode, boolean processing, String customerCode, String relevanceCode) {
    Date now = new Date();
    boolean isProcessing = false;
    if(CollectionUtils.isEmpty(currentSalePolices)) {
      return null;
    }
    
    /*
     * 匹配过程为：
     * 1、首先找到缓存中那些时间范围有效的，状态有效的优惠政策，形成futureResultMapping，K-V对应关系，以便后续过程使用
     * 2、一个一个优惠政策任务进行指定用户和优惠政策的匹配
     * */
    
    // 1、=======
    Map<SalePolicyVo , Future<Boolean>> futureResultMapping = Maps.newLinkedHashMap();
    for (SalePolicyVo currentSalePolicy : currentSalePolices) {
      // 确认优惠政策是在有效期内
      if(now.after(currentSalePolicy.getValidStartTime()) && now.before(currentSalePolicy.getValidEndTime())) {
        isProcessing = true;
      }
      if(isProcessing != processing) {
        continue;
      }
      
      // 以下只关心客户集合是否匹配
      SalePolicyCustomerScopeMatchedTask task = this.applicationContext.getBean(SalePolicyCustomerScopeMatchedTask.class , currentSalePolicy , tenantCode , customerCode);
      Future<Boolean> future = policyLoadingExecutor.submit(task);
      futureResultMapping.put(currentSalePolicy , future);
    }
    
    // 2、=======
    Set<SalePolicyVo> results = Sets.newLinkedHashSet();
    for (Map.Entry<SalePolicyVo,Future<Boolean>> futureEntry : futureResultMapping.entrySet()) {
      try {
        SalePolicyVo currentSalePolicy = futureEntry.getKey();
        Future<Boolean> currentFutureValue = futureEntry.getValue();
        Boolean currentFuture = currentFutureValue.get();
        if(currentFuture != null && currentFuture) {
          results.add(currentSalePolicy);
        }
      } catch (InterruptedException | ExecutionException e) {
        LOGGER.error(e.getMessage() , e);
      }
    }
    return results;
  }

  @Override
  public Set<SalePolicyVo> match(Set<SalePolicyVo> currentSalePolices, String tenantCode, boolean processing, String customerCode, String relevanceCode, String... productCodes) {
    Set<SalePolicyVo> salePolicys = this.match(currentSalePolices, tenantCode, processing, customerCode, relevanceCode);
    if(CollectionUtils.isEmpty(salePolicys)) {
      return null;
    }
    
    // 现在判定产品维度（注意，如果是整单优惠，则认为匹配指定的商品）
    Set<SalePolicyVo> results = Sets.newLinkedHashSet();
    for (SalePolicyVo currentSalePolicy : salePolicys) {
      if(currentSalePolicy.getWholePolicy()) {
        results.add(currentSalePolicy);
        continue;
      }
      
      Set<AbstractSalePolicyProductInfo> salePolicyProductInfos = currentSalePolicy.getSalePolicyProductInfos();
      if(CollectionUtils.isEmpty(salePolicyProductInfos)) {
        continue;
      }
      // 进行优惠政策中商品的匹配
      for (AbstractSalePolicyProductInfo salePolicyProductInfo : salePolicyProductInfos) {
        if(StringUtils.containsAny(salePolicyProductInfo.getProductCode(), productCodes)) {
          results.add(currentSalePolicy);
          break;
        }
      }
    }
    return results;
  }
}
