package com.biz.crm.dms.business.policy.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyTemplete;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyTempleteExecuteStrategy;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyTempleteExecuteStrategyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyTempleteRepository;
import com.biz.crm.dms.business.policy.local.service.SalePolicyTempleteVoService;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicStrategy;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyTempleteLogDto;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyTempletePageDto;
import com.biz.crm.dms.business.policy.sdk.enums.ProductSelectionMethod;
import com.biz.crm.dms.business.policy.sdk.event.SalePolicyTempleteLogEventListener;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyTempleteExecuteStrategyVo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyTempleteVo;
import com.bizunited.nebula.common.service.CopyObjectCallback;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

@Service
public class SalePolicyTempleteVoServiceImpl implements SalePolicyTempleteVoService , CopyObjectCallback<SalePolicyTemplete, SalePolicyTempleteVo> {

  @Autowired(required = false)
  private SalePolicyTempleteRepository salePolicyTempleteRepository;
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private SalePolicyTempleteExecuteStrategyRepository salePolicyTempleteExecuteStrategyRepository;
  @Autowired(required = false)
  private SalePolicyRepository salePolicyRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyTempleteVoServiceImpl.class);
  
  @Override
  @Transactional
  public SalePolicyTempleteVo create(SalePolicyTempleteVo salePolicyTemplete) {
    Validate.notNull(salePolicyTemplete , "添加优惠政策模板时，需要确认传入优惠政策模板类对象!!");
    String tenantCode = TenantUtils.getTenantCode();
    salePolicyTemplete.setTenantCode(tenantCode);
    salePolicyTemplete.setId(null);
    String type = salePolicyTemplete.getType();
    Validate.notBlank(type , "添加优惠政策模板时，模板类型信息必须传入!!");
    String templeteCode = salePolicyTemplete.getTempleteCode();
    // 如果没有设定优惠政策模板的业务编号，则在这里生成一个
    if(StringUtils.isBlank(templeteCode)) {
      templeteCode = StringUtils.join(tenantCode , new SimpleDateFormat("yyyyMMddHHmmssS").format(new Date()));
      salePolicyTemplete.setTempleteCode(templeteCode);
    }
    SalePolicyTemplete existTemplete = this.salePolicyTempleteRepository.findByTempleteCode(templeteCode, tenantCode);
    Validate.isTrue(existTemplete == null , "添加优惠政策模板时，发现模板业务编号已经使用，请检查!!");
    
    // 其它的选择信息的判定
    this.validate(salePolicyTemplete , "添加优惠政策模板时");
    salePolicyTemplete.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    
    // ======= 开始进行保存
    SalePolicyTemplete salePolicyTempleteEntity = this.nebulaToolkitService.copyObjectByBlankList(salePolicyTemplete, SalePolicyTemplete.class, LinkedHashSet.class, ArrayList.class);
    salePolicyTempleteEntity.setBindingCharacteristicStrategyCodes(StringUtils.join(salePolicyTemplete.getBindingCharacteristicStrategyCodes() , ","));
    salePolicyTempleteEntity.setBindingProductSelectionMethods(StringUtils.join(salePolicyTemplete.getBindingProductSelectionMethods() , ","));
    salePolicyTempleteEntity.setBindingLimitStrategyCodes(StringUtils.join(salePolicyTemplete.getBindingLimitStrategyCodes() , ","));
    salePolicyTempleteEntity.setBindingScopeTypes(StringUtils.join(salePolicyTemplete.getBindingScopeTypes() , ","));
    this.salePolicyTempleteRepository.save(salePolicyTempleteEntity);
    salePolicyTemplete.setId(salePolicyTempleteEntity.getId());
    
    // 开始保存关联的执行策略信息
    Set<SalePolicyTempleteExecuteStrategyVo> salePolicyTempleteExecuteStrategys = salePolicyTemplete.getBindingExecuteStraties();
    for (SalePolicyTempleteExecuteStrategyVo salePolicyTempleteExecuteStrategyVo : salePolicyTempleteExecuteStrategys) {
      Set<String> bindingExecuteLadders = salePolicyTempleteExecuteStrategyVo.getBindingExecuteLadders(); 
      String bindingExecuteStrategyCode = salePolicyTempleteExecuteStrategyVo.getBindingExecuteStrategyCode();
      // 创建entity，并且保存
      SalePolicyTempleteExecuteStrategy salePolicyTempleteExecuteStrategy = new SalePolicyTempleteExecuteStrategy();
      salePolicyTempleteExecuteStrategy.setTempleteId(salePolicyTempleteEntity.getId());
      salePolicyTempleteExecuteStrategy.setBindingExecuteStrategyCode(bindingExecuteStrategyCode);
      salePolicyTempleteExecuteStrategy.setBindingExecuteLadders(StringUtils.join(bindingExecuteLadders, ","));
      this.salePolicyTempleteExecuteStrategyRepository.save(salePolicyTempleteExecuteStrategy);
      salePolicyTempleteExecuteStrategyVo.setId(salePolicyTempleteExecuteStrategy.getId());
    }
    // 记录操作日志
    this.recordLogOnCreate(salePolicyTemplete);
    return salePolicyTemplete;
  }

  /**
   * 记录创建日志
   *
   * @param salePolicyTemplete 销售策略模板
   */
  private void recordLogOnCreate(SalePolicyTempleteVo salePolicyTemplete) {
    SalePolicyTempleteLogDto logDto = new SalePolicyTempleteLogDto();
    logDto.setOriginal(null);
    logDto.setNewest(salePolicyTemplete);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyTempleteLogEventListener, SalePolicyTempleteLogDto> onCreate = SalePolicyTempleteLogEventListener::onCreate;
    this.nebulaNetEventClient.publish(logDto, SalePolicyTempleteLogEventListener.class, onCreate);
  }

  /**
   * 在进行优惠政策模板创建或者修改时，首先需要进行各种信息的验证
   * @param salePolicyTemplete
   */
  private void validate(SalePolicyTempleteVo salePolicyTemplete , String onTime) {
    String type = salePolicyTemplete.getType();
    // 是否支持关联预算信息
    Boolean supportBudget = salePolicyTemplete.getSupportBudget();
    Validate.notNull(supportBudget , onTime + "，该优惠模板是否支持关联预算，必须查询，请检查!!");
    SalePolicyTempleteRegister currentSalePolicyTempleteRegister = this.findSalePolicyTempleteRegister(type);
    Validate.notNull(currentSalePolicyTempleteRegister , onTime + "，未发现正确的优惠政策注册器（%s），请检查!" , type);
    // 绑定的可使用的执行特性
    Set<String> bindingCharacteristicStrategyCodes = salePolicyTemplete.getBindingCharacteristicStrategyCodes();
    if(!CollectionUtils.isEmpty(bindingCharacteristicStrategyCodes)) {
      for (String characteristicStrategyCode : bindingCharacteristicStrategyCodes) {
        Validate.notNull(this.findCorrectCharacteristicStrategy(currentSalePolicyTempleteRegister, characteristicStrategyCode) , onTime + "，未找到指定的特性执行规则信息（%s），请检查!!" , characteristicStrategyCode);
      }
    }
    // 商品选择方式
    Boolean supportProduct = currentSalePolicyTempleteRegister.supportProduct();
    Validate.notNull(supportProduct , onTime + "，指定的优惠政策注册器(%s)，并没有指定supportProduct()方法的返回值，请检查!!" , type);
    Set<String> productSelectionMethods = salePolicyTemplete.getBindingProductSelectionMethods();
    Validate.isTrue(!CollectionUtils.isEmpty(productSelectionMethods) , "添加优惠政策模板时，必须指定商品选择方式，请检查!!");
    if(!supportProduct) {
      Validate.isTrue(productSelectionMethods.size() == 1 , onTime + "，当前优惠政策模板注册器，设定为“不支持商品选择方式”，所以优惠政策模板的商品选择方式只能是“NOSELECT”");
      Validate.isTrue(StringUtils.equals(productSelectionMethods.iterator().next(), ProductSelectionMethod.NOSELECT.getCode()) , "当前优惠政策模板注册器，设定为“不支持商品选择方式”，所以优惠政策模板的商品选择方式只能是“NOSELECT”");
    }
    // 绑定的可使用的执行政策信息
    Set<SalePolicyTempleteExecuteStrategyVo> salePolicyTempleteExecuteStrategys = salePolicyTemplete.getBindingExecuteStraties();
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyTempleteExecuteStrategys) , onTime + "，当前优惠政策模板注册器，必须设定绑定的政策执行信息，请检查!!");
    for (SalePolicyTempleteExecuteStrategyVo salePolicyTempleteExecuteStrategyVo : salePolicyTempleteExecuteStrategys) {
      String executeStrategyCode = salePolicyTempleteExecuteStrategyVo.getBindingExecuteStrategyCode();
      Validate.notNull(this.findCorrectExecuteStrategy(currentSalePolicyTempleteRegister, executeStrategyCode) , onTime + "，未找到指定优惠政策执行策略（%s），请检查!!" , executeStrategyCode);
      Set<String> bindingExecuteLadders = salePolicyTempleteExecuteStrategyVo.getBindingExecuteLadders(); 
      Validate.isTrue(!CollectionUtils.isEmpty(bindingExecuteLadders) , onTime + "，至少有一个执行策略信息没有指定使用的阶梯策略");
    }
    // 绑定的可使用的限量执行规则
    Set<String> bindingLimitStrategyCodes = salePolicyTemplete.getBindingLimitStrategyCodes();
    if(!CollectionUtils.isEmpty(bindingLimitStrategyCodes)) {
      for (String limitStrategyCode : bindingLimitStrategyCodes) {
        Validate.notNull(this.findCorrectLimitStrategy(currentSalePolicyTempleteRegister, limitStrategyCode) , onTime + "，未找到指定的限量政策信息（%s），请检查!!" , limitStrategyCode);
      }
    }
    // 绑定的客户选择方式
    Set<String> bindingScopeTypes = salePolicyTemplete.getBindingScopeTypes();
    Validate.isTrue(!CollectionUtils.isEmpty(bindingScopeTypes) , onTime + "，当前优惠政策模板注册器，必须设定绑定的客户范围选择方式信息，请检查!!");
    for (String bindingScopeType : bindingScopeTypes) {
      Validate.notNull(this.findCorrectCustomerScopeStrategy(currentSalePolicyTempleteRegister, bindingScopeType) , onTime + "，未找到指定的客户范围选择方式（%s），请检查!!" , bindingScopeType);
    }
  }

  /**
   * 在指定的salePolicyTempleteRegister优惠政策模板注册器中，找到正确的执行特性策略
   * @return 如果没有找到，则返回null
   */
  private CharacteristicStrategy<? extends AbstractCharacteristicInfo> findCorrectCharacteristicStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String characteristicStrategyCode) {
    Collection<Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>>> characteristicStrategyClasses = salePolicyTempleteRegister.getCharacteristicStrategyClasses();
    if(CollectionUtils.isEmpty(characteristicStrategyClasses)) {
      return null;
    }
    for (Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>> characteristicStrategyClass : characteristicStrategyClasses) {
      try {
        CharacteristicStrategy<? extends AbstractCharacteristicInfo> characteristicStrategy = applicationContext.getBean(characteristicStrategyClass);
        if(StringUtils.equals(characteristicStrategy.characteristicCode(), characteristicStrategyCode)) {
          return characteristicStrategy;
        }
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        return null;
      }
    }
    return null;
  }
  
  /**
   * 在指定的salePolicyTempleteRegister优惠政策模板注册器中，找到正确的优惠政策模板执行策略
   * @return 如果没有找到，则返回null
   */
  private SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> findCorrectExecuteStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String executeStrategyCode) {
    Collection<Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>>> executeStrategyClasses = salePolicyTempleteRegister.getExecuteStrategyClasses();
    for (Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> executeStrategyClass : executeStrategyClasses) {
      try {
        SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy = applicationContext.getBean(executeStrategyClass);
        if(StringUtils.equals(salePolicyExecuteStrategy.getExecuteStrategyCode(), executeStrategyCode)) {
          return salePolicyExecuteStrategy;
        }
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        return null;
      }
    }
    return null;
  }
    
  /**
   * 该私有方法用于基于特定的优惠政策模板查询所有已绑定的限量政策
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> findAllBindingLimitStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>>> bindableLimitStrategyClasses = salePolicyTempleteRegister.getBindableLimitStrategyClasses();
    if(CollectionUtils.isEmpty(bindableLimitStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些限价政策可以进行绑定
    Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyClass : bindableLimitStrategyClasses) {
      try {
        SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = applicationContext.getBean(bindableLimitStrategyClass);
        bindableLimitStrategySet.add(currentSalePolicyLimitStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableLimitStrategySet;
  }
  
  /**
   * 该私有方法用于查询符合要求的限价政策（基于特定的优惠政策模板）
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> findCorrectLimitStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String limitStrategyCode) {
    Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategies = this.findAllBindingLimitStrategy(salePolicyTempleteRegister);
    Validate.isTrue(!CollectionUtils.isEmpty(bindableLimitStrategies) , "未发现当前优惠政策模板有至少一个能绑定的限价政策，请检查!!");
    HashMap<String, SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy : bindableLimitStrategies) {
      bindableLimitStrategyMapping.put(salePolicyLimitStrategy.getLimitStrategyCode(), salePolicyLimitStrategy);
    }
    
    // 寻找正确的策略处理器，并进行保存处理
    Validate.notBlank(limitStrategyCode , "至少一个将要保存的现价策略没有指定code信息，请检查!!");
    SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = bindableLimitStrategyMapping.get(limitStrategyCode);
    Validate.notNull(currentSalePolicyLimitStrategy , "未匹配到业务编号为:%s 的限量政策信息，可能是该政策不存在或者改优惠政策模板不允许关联这个限量政策，请检查!!" , limitStrategyCode);
    return currentSalePolicyLimitStrategy;
  }
  
  /**
   * 该私有方法为寻找匹配的人员范围选择策略
   * @return
   */
  private SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> findCorrectCustomerScopeStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String customerScopeType) {
    Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerScopeStrategies = this.findAllBindCustomerScopeStrategy(salePolicyTempleteRegister);
    HashMap<String, SalePolicyCustomerScopeStrategy<?>> bindableCustomerScopeStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeStrategy : salePolicyCustomerScopeStrategies) {
      bindableCustomerScopeStrategyMapping.put(salePolicyCustomerScopeStrategy.getScopeType(), salePolicyCustomerScopeStrategy);
    }
    // 寻找正确的客户范围处理策略，并进行保存
    Validate.notBlank(customerScopeType , "该优惠政策模板保存时，并没有设定客户范围的选择类型(或者根本没有传入)，请检查!!");
    SalePolicyCustomerScopeStrategy<?> currentCustomerScopeStrategy = bindableCustomerScopeStrategyMapping.get(customerScopeType);
    Validate.notNull(currentCustomerScopeStrategy , "该优惠政策模板保存时，并没有发现指定类型的客户范围选择策略（%s），请检查!!");
    return currentCustomerScopeStrategy;
  }
  
  /**
   * 该私有方法为寻找匹配的人员范围选择策略
   * @return
   */
  private Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> findAllBindCustomerScopeStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>>> salePolicyCustomerScopeStrategyClasses = salePolicyTempleteRegister.getCustomerScopeStrategyClasses();
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyCustomerScopeStrategyClasses) , "该优惠政策模板并没有注册任何客户返回选择方式，请检查!!");
    Set<SalePolicyCustomerScopeStrategy<?>> bindableCustomerScopeStrategies = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyCustomerScopeStrategy<?>> salePolicyCustomerScopeStrategyClass : salePolicyCustomerScopeStrategyClasses) {
      try {
        SalePolicyCustomerScopeStrategy<?> currentSalePolicyCustomerScopeStrategy = applicationContext.getBean(salePolicyCustomerScopeStrategyClass);
        String currentScopeType = currentSalePolicyCustomerScopeStrategy.getScopeType();
        Validate.notBlank(currentScopeType , "发现至少一个客户范围策略没有指定code信息(%s)，请检查!!" , salePolicyCustomerScopeStrategyClass.getName());
        bindableCustomerScopeStrategies.add(currentSalePolicyCustomerScopeStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableCustomerScopeStrategies;
  }

  /**
   * 通过指定的type找到对应的优惠政策模板模板信息，如果没有找到则返回null
   */
  private SalePolicyTempleteRegister findSalePolicyTempleteRegister(String type) {
    for (SalePolicyTempleteRegister salePolicyTempleteRegister : salePolicyTempleteRegisters) {
      if(StringUtils.equals(salePolicyTempleteRegister.getType(), type)) {
        return salePolicyTempleteRegister;
      }
    }
    return null;
  }
  
  @Override
  @Transactional
  public SalePolicyTempleteVo update(SalePolicyTempleteVo salePolicyTemplete) {
    Validate.notNull(salePolicyTemplete , "修改优惠政策模板时，需要确认传入优惠政策模板类对象!!");
    String tenantCode = TenantUtils.getTenantCode();
    salePolicyTemplete.setTenantCode(tenantCode);
    String id = salePolicyTemplete.getId();
    Validate.notBlank(id , "修改优惠政策模板时，需要传入优惠政策模板的id信息，请检查!!");
    SalePolicyTemplete currentSalePolicyTemplete = this.salePolicyTempleteRepository.findById(id);
    Validate.notNull(currentSalePolicyTemplete , "修改优惠政策模板时，未找到指定的优惠政策模板信息，请检查!!");
    SalePolicyTempleteVo currentSalePolicyTempleteVo = this.nebulaToolkitService.copyObjectByBlankList(currentSalePolicyTemplete, SalePolicyTempleteVo.class, HashSet.class, LinkedList.class);
    // type和templeteCode不允许修改
    salePolicyTemplete.setType(currentSalePolicyTemplete.getType());
    salePolicyTemplete.setTempleteCode(currentSalePolicyTemplete.getTempleteCode());
    // 其它的选择信息的判定
    this.validate(salePolicyTemplete , "添加优惠政策模板时");
    
    // ======= 开始进行修改。注意，并不是优惠政策模板上的所有信息都允许修改
    currentSalePolicyTemplete.setBindingCharacteristicStrategyCodes(StringUtils.join(salePolicyTemplete.getBindingCharacteristicStrategyCodes() , ","));
    currentSalePolicyTemplete.setBindingProductSelectionMethods(StringUtils.join(salePolicyTemplete.getBindingProductSelectionMethods() , ","));
    currentSalePolicyTemplete.setBindingLimitStrategyCodes(StringUtils.join(salePolicyTemplete.getBindingLimitStrategyCodes() , ","));
    currentSalePolicyTemplete.setBindingScopeTypes(StringUtils.join(salePolicyTemplete.getBindingScopeTypes() , ","));
    currentSalePolicyTemplete.setSupportBudget(salePolicyTemplete.getSupportBudget());
    currentSalePolicyTemplete.setTempleteName(salePolicyTemplete.getTempleteName());
    this.salePolicyTempleteRepository.saveOrUpdate(currentSalePolicyTemplete);
    
    // ======= 接着是修改关联信息，修改关联信息实际上是先删除之前的，然后再重新添加
    this.salePolicyTempleteExecuteStrategyRepository.deleteByTempleteId(id);
    // 开始重新添加关联的执行策略信息
    Set<SalePolicyTempleteExecuteStrategyVo> salePolicyTempleteExecuteStrategys = salePolicyTemplete.getBindingExecuteStraties();
    for (SalePolicyTempleteExecuteStrategyVo salePolicyTempleteExecuteStrategyVo : salePolicyTempleteExecuteStrategys) {
      Set<String> bindingExecuteLadders = salePolicyTempleteExecuteStrategyVo.getBindingExecuteLadders(); 
      String bindingExecuteStrategyCode = salePolicyTempleteExecuteStrategyVo.getBindingExecuteStrategyCode();
      // 创建entity，并且保存
      SalePolicyTempleteExecuteStrategy salePolicyTempleteExecuteStrategy = new SalePolicyTempleteExecuteStrategy();
      salePolicyTempleteExecuteStrategy.setTempleteId(id);
      salePolicyTempleteExecuteStrategy.setBindingExecuteStrategyCode(bindingExecuteStrategyCode);
      salePolicyTempleteExecuteStrategy.setBindingExecuteLadders(StringUtils.join(bindingExecuteLadders, ","));
      this.salePolicyTempleteExecuteStrategyRepository.save(salePolicyTempleteExecuteStrategy);
      salePolicyTempleteExecuteStrategyVo.setId(salePolicyTempleteExecuteStrategy.getId());
    }
    // 记录操作日志
    this.recordLogOnUpdate(currentSalePolicyTempleteVo, salePolicyTemplete);
    return salePolicyTemplete;
  }

  /**
   * 记录更新日志
   *
   * @param salePolicyTemplete 销售策略模板
   */
  private void recordLogOnUpdate(SalePolicyTempleteVo currentSalePolicyTemplete, SalePolicyTempleteVo salePolicyTemplete) {
    SalePolicyTempleteLogDto logDto = new SalePolicyTempleteLogDto();
    logDto.setOriginal(currentSalePolicyTemplete);
    logDto.setNewest(salePolicyTemplete);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyTempleteLogEventListener, SalePolicyTempleteLogDto> onUpdate = SalePolicyTempleteLogEventListener::onUpdate;
    this.nebulaNetEventClient.publish(logDto, SalePolicyTempleteLogEventListener.class, onUpdate);
  }

  @Override
  public boolean validate(Class<?> sourceClass, Class<?> targetClass) {
    return sourceClass == SalePolicyTemplete.class && targetClass == SalePolicyTempleteVo.class;
  }

  @Override
  @Transactional
  public void invalid(String salePolicyTempleteCode) {
    Validate.notBlank(salePolicyTempleteCode , "禁用优惠政策模板时，必须传入相关的模板code");
    String tenantCode = TenantUtils.getTenantCode();
    SalePolicyTemplete currentSalePolicyTemplete = this.salePolicyTempleteRepository.findByTempleteCode(salePolicyTempleteCode, tenantCode);
    Validate.notNull(currentSalePolicyTemplete , "禁用优惠政策模板时，未找到对应的模板code");
    
    currentSalePolicyTemplete.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
    this.salePolicyTempleteRepository.saveOrUpdate(currentSalePolicyTemplete);
    this.recordLogOnDisable(currentSalePolicyTemplete);
  }

  /**
   * 禁用日志记录
   *
   * @param currentSalePolicyTemplete 目前销售政策模板
   */
  private void recordLogOnDisable(SalePolicyTemplete currentSalePolicyTemplete) {
    SalePolicyTempleteVo voDb = new SalePolicyTempleteVo();
    voDb.setId(currentSalePolicyTemplete.getId());
    voDb.setEnableStatus(EnableStatusEnum.ENABLE.getCode());

    SalePolicyTempleteVo vo = new SalePolicyTempleteVo();
    vo.setId(currentSalePolicyTemplete.getId());
    vo.setEnableStatus(EnableStatusEnum.DISABLE.getCode());

    SalePolicyTempleteLogDto logDto = new SalePolicyTempleteLogDto();
    logDto.setOriginal(voDb);
    logDto.setNewest(vo);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyTempleteLogEventListener, SalePolicyTempleteLogDto> OnDisable = SalePolicyTempleteLogEventListener::OnDisable;
    this.nebulaNetEventClient.publish(logDto, SalePolicyTempleteLogEventListener.class, OnDisable);
  }

  @Override
  @Transactional
  public void effective(String salePolicyTempleteCode) {
    Validate.notBlank(salePolicyTempleteCode , "启用优惠政策模板时，必须传入相关的模板code");
    String tenantCode = TenantUtils.getTenantCode();
    SalePolicyTemplete currentSalePolicyTemplete = this.salePolicyTempleteRepository.findByTempleteCode(salePolicyTempleteCode, tenantCode);
    Validate.notNull(currentSalePolicyTemplete , "启用优惠政策模板时，未找到对应的模板code");
    
    currentSalePolicyTemplete.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.salePolicyTempleteRepository.saveOrUpdate(currentSalePolicyTemplete);

    this.recordLogOnEnable(currentSalePolicyTemplete);
  }

  /**
   * 记录启用日志
   *
   * @param currentSalePolicyTemplete 目前销售政策模板
   */
  private void recordLogOnEnable(SalePolicyTemplete currentSalePolicyTemplete) {
    SalePolicyTempleteVo voDb = new SalePolicyTempleteVo();
    voDb.setId(currentSalePolicyTemplete.getId());
    voDb.setEnableStatus(EnableStatusEnum.DISABLE.getCode());

    SalePolicyTempleteVo vo = new SalePolicyTempleteVo();
    vo.setId(currentSalePolicyTemplete.getId());
    vo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());

    SalePolicyTempleteLogDto logDto = new SalePolicyTempleteLogDto();
    logDto.setOriginal(voDb);
    logDto.setNewest(vo);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyTempleteLogEventListener, SalePolicyTempleteLogDto> OnEnable = SalePolicyTempleteLogEventListener::OnEnable;
    this.nebulaNetEventClient.publish(logDto, SalePolicyTempleteLogEventListener.class, OnEnable);
  }

  @Override
  @Transactional
  public void deleteByTempleteCode(String salePolicyTempleteCode) {
    Validate.notBlank(salePolicyTempleteCode , "删除优惠政策模板时，必须传入优惠政策业务编号!");
    String tenantCode = TenantUtils.getTenantCode();
    // 如果优惠政策模板已关联优惠政策，则不允许被删除
    List<SalePolicy> salePolicies = this.salePolicyRepository.findByTempleteCodeAndTenantCode(salePolicyTempleteCode , tenantCode);
    Validate.isTrue(CollectionUtils.isEmpty(salePolicies) , "删除优惠政策模板时，发现该优惠政策模板已经被使用，不允许删除");
    SalePolicyTemplete exsitSalePolicyTemplete = this.salePolicyTempleteRepository.findByTempleteCode(salePolicyTempleteCode, tenantCode);
    Validate.notNull(exsitSalePolicyTemplete , "删除优惠政策模板时，未找到指定的模板信息，请检查!!");
    // 先删除设定的模板执行策略明细
    this.salePolicyTempleteExecuteStrategyRepository.deleteByTempleteId(exsitSalePolicyTemplete.getId());
    this.salePolicyTempleteRepository.deleteByTempleteCode(salePolicyTempleteCode, tenantCode);
    // 记录操作日志
    this.recordLogOnDelete(exsitSalePolicyTemplete);
  }

  /**
   * 记录删除日志
   *
   * @param salePolicyTemplete 存在销售策略模板
   */
  private void recordLogOnDelete(SalePolicyTemplete salePolicyTemplete) {
    SalePolicyTempleteVo voDb = new SalePolicyTempleteVo();
    voDb.setId(salePolicyTemplete.getId());
    SalePolicyTempleteLogDto logDto = new SalePolicyTempleteLogDto();
    logDto.setOriginal(voDb);
    logDto.setNewest(null);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyTempleteLogEventListener, SalePolicyTempleteLogDto> OnDelete = SalePolicyTempleteLogEventListener::OnDelete;
    this.nebulaNetEventClient.publish(logDto, SalePolicyTempleteLogEventListener.class, OnDelete);
  }

  @Override
  public void callback(Class<SalePolicyTemplete> sourceClass,
      List<SalePolicyTemplete> sourceObjects, Class<SalePolicyTempleteVo> targetClass,
      List<SalePolicyTempleteVo> targetObjects) {
    Iterator<SalePolicyTempleteVo> targetObjectIterator = targetObjects.iterator();
    for (SalePolicyTemplete sourceSalePolicyTemplete : sourceObjects) {
      SalePolicyTempleteVo targetSalePolicyTemplete = targetObjectIterator.next();
      // 已绑定的执行特性判定信息
      String bindingCharacteristicStrategyCodes = sourceSalePolicyTemplete.getBindingCharacteristicStrategyCodes();
      if(StringUtils.isNotBlank(bindingCharacteristicStrategyCodes)) {
        targetSalePolicyTemplete.setBindingCharacteristicStrategyCodes(Sets.newHashSet(StringUtils.split(bindingCharacteristicStrategyCodes, ",")));
      }
      // 已绑定的商品选择方式
      String bindingProductSelectionMethods = sourceSalePolicyTemplete.getBindingProductSelectionMethods();
      if(StringUtils.isNotBlank(bindingProductSelectionMethods)) {
        targetSalePolicyTemplete.setBindingProductSelectionMethods(Sets.newHashSet(StringUtils.split(bindingProductSelectionMethods, ",")));
      }
      // 已绑定的限量执行规则
      String bindingLimitStrategyCodes = sourceSalePolicyTemplete.getBindingLimitStrategyCodes();
      if(StringUtils.isNotBlank(bindingLimitStrategyCodes)) {
        targetSalePolicyTemplete.setBindingLimitStrategyCodes(Sets.newHashSet(StringUtils.split(bindingLimitStrategyCodes, ",")));
      }
      // 已绑定的客户选择方式规则
      String bindingScopeTypes = sourceSalePolicyTemplete.getBindingScopeTypes();
      if(StringUtils.isNotBlank(bindingScopeTypes)) {
        targetSalePolicyTemplete.setBindingScopeTypes(Sets.newHashSet(StringUtils.split(bindingScopeTypes, ",")));
      }
      // 已经绑定的执行政策信息
      Set<SalePolicyTempleteExecuteStrategy> bindingExecuteStrategys = sourceSalePolicyTemplete.getBindingExecuteStraties();
      if(!CollectionUtils.isEmpty(bindingExecuteStrategys)) {
        Set<SalePolicyTempleteExecuteStrategyVo> bindingExecuteStrategyVos = Sets.newLinkedHashSet();
        for (SalePolicyTempleteExecuteStrategy salePolicyTempleteExecuteStrategy : bindingExecuteStrategys) {
          SalePolicyTempleteExecuteStrategyVo salePolicyTempleteExecuteStrategyVo = new SalePolicyTempleteExecuteStrategyVo();
          salePolicyTempleteExecuteStrategyVo.setId(salePolicyTempleteExecuteStrategy.getId());
          salePolicyTempleteExecuteStrategyVo.setBindingExecuteLadders(Sets.newHashSet(StringUtils.split(salePolicyTempleteExecuteStrategy.getBindingExecuteLadders(), ",")));
          salePolicyTempleteExecuteStrategyVo.setBindingExecuteStrategyCode(salePolicyTempleteExecuteStrategy.getBindingExecuteStrategyCode());
          bindingExecuteStrategyVos.add(salePolicyTempleteExecuteStrategyVo);
        }
        targetSalePolicyTemplete.setBindingExecuteStraties(bindingExecuteStrategyVos);
      }
      // 开始进行模板注册器相关的数据补充
      String templeteRegisterType = sourceSalePolicyTemplete.getType();
      SalePolicyTempleteRegister salePolicyTempleteRegister = this.findSalePolicyTempleteRegister(templeteRegisterType);
      if(salePolicyTempleteRegister != null) {
        targetSalePolicyTemplete.setTypeDesc(salePolicyTempleteRegister.getTypeDesc());
      }
    }
  }

  @Override
  public SalePolicyTempleteVo findByTempleteCode(String templeteCode, String tenantCode) {
    if(StringUtils.isAnyBlank(templeteCode , tenantCode)) {
      return null;
    }
    SalePolicyTemplete salePolicyTemplete = this.salePolicyTempleteRepository.findByTempleteCode(templeteCode, tenantCode);
    if(salePolicyTemplete == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(salePolicyTemplete, SalePolicyTempleteVo.class, LinkedHashSet.class, ArrayList.class);
  }

  @Override
  public Set<SalePolicyTempleteVo> findByTenantCode(String tenantCode , EnableStatusEnum enableStatus) {
    if(StringUtils.isBlank(tenantCode)) {
      return null;
    }
    Set<SalePolicyTemplete> salePolicyTempletes = this.salePolicyTempleteRepository.findByTenantCode(tenantCode , enableStatus);
    if(CollectionUtils.isEmpty(salePolicyTempletes)) {
      return null;
    }
    // 进行VO装换
    List<SalePolicyTempleteVo> salePolicyTempleteVos = this.covertEntityToVo(Lists.newArrayList(salePolicyTempletes));
    return Sets.newLinkedHashSet(salePolicyTempleteVos);
  }
  
  @Override
  public Page<SalePolicyTempleteVo> findByConditions(Pageable pageable, SalePolicyTempletePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new SalePolicyTempletePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    
    // 校验后进行查询和转换
    Page<SalePolicyTemplete> entityPage = this.salePolicyTempleteRepository.findByConditions(pageable, dto);
    List<SalePolicyTemplete> entities = entityPage.getRecords();
    Page<SalePolicyTempleteVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    pageResult.setRecords(covertEntityToVo(entities));
    return pageResult;
  }
  
  /**
   * 进行Entity到VO的转换工作
   */
  private List<SalePolicyTempleteVo> covertEntityToVo(List<SalePolicyTemplete> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    List<SalePolicyTempleteVo> salePolicyTempleteVos = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, SalePolicyTemplete.class , SalePolicyTempleteVo.class, LinkedHashSet.class, ArrayList.class));
    return salePolicyTempleteVos;
  }
}
