package com.biz.crm.dms.business.policy.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyRecord;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyTemplete;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRecordRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRepository;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyTempleteRepository;
import com.biz.crm.dms.business.policy.local.service.task.SalePolicyCacheLoadingService;
import com.biz.crm.dms.business.policy.sdk.characteristic.CharacteristicStrategy;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyLogDto;
import com.biz.crm.dms.business.policy.sdk.dto.SalePolicyPageDto;
import com.biz.crm.dms.business.policy.sdk.event.SalePolicyEventListener;
import com.biz.crm.dms.business.policy.sdk.event.SalePolicyLogEventListener;
import com.biz.crm.dms.business.policy.sdk.register.SalePolicyTempleteRegister;
import com.biz.crm.dms.business.policy.sdk.service.SalePolicyVoService;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCustomerScopeStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyLimitStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyMatchedStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyStickupListener;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;

import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyCustomerScopeInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyLimitInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyProductInfo;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyThreshold;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 优惠政策基本信息的服务层接口实现
 * @author yinwenjie
 */
@Service
public class SalePolicyVoServiceImpl implements SalePolicyVoService {
  @Autowired(required = false)
  private List<SalePolicyTempleteRegister> salePolicyTempleteRegisters;
  @Autowired(required = false)
  private List<SalePolicyEventListener> salePolicyEventListeners;
  @Autowired(required = false)
  private SalePolicyMatchedStrategy salePolicyMatchedStrategy;
  @Autowired(required = false)
  private RedisMutexService redisMutexService;
  @Autowired(required = false)
  private SalePolicyRepository salePolicyRepository;
  @Autowired(required = false)
  private SalePolicyTempleteRepository salePolicyTempleteRepository;
  @Autowired(required = false)
  private SalePolicyRecordRepository salePolicyRecordRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  @Autowired(required = false)
  private SalePolicyCacheLoadingService salePolicyCacheLoadingService;
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyVoServiceImpl.class);
  /**
   * 优惠政策预生成的redis-map结构的名字
   */
  private static final String SALE_POLICY_GEN_PREFIX = "SALE_POLICY_GEN_PREFIX";
  
  @Override
  public String preSave() {
    String prefix = UUID.randomUUID().toString();
    // 7天后过期
    this.redisMutexService.setMCode(SALE_POLICY_GEN_PREFIX, prefix, prefix, TimeUnit.MILLISECONDS.convert(7 ,  TimeUnit.DAYS));
    return prefix;
  }
  
  @Override
  @Transactional
  public SalePolicyVo create(JSONObject salePolicyJson) {
    Validate.notNull(salePolicyJson , "创建优惠政策时，必须传入优惠政策信息（JSON形式）。");
    SalePolicyVo currentSalePolicyVo = this.convertSalePolicyVo(salePolicyJson, "创建时");
    this.createHandle(currentSalePolicyVo);
    // 换新缓存
    String tenantCode = TenantUtils.getTenantCode();
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
    return currentSalePolicyVo;
  }

  @Override
  @Transactional
  public SalePolicyVo create(SalePolicyVo salePolicy) {
    Validate.notNull(salePolicy , "创建优惠政策时，必须传入优惠政策信息（对象形式）。");
    SalePolicyVo currentSalePolicyVo = this.createHandle(salePolicy);
    // 换新缓存
    String tenantCode = TenantUtils.getTenantCode();
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
    return currentSalePolicyVo;
  }

  private SalePolicyVo createHandle(SalePolicyVo salePolicy) {
    Validate.notNull(salePolicy , "创建时，必须传入要添加的优惠政策信息（包括关联信息），请参见文档!!");
    /*
     * 这是真正的添加方法，注意：
     * 该方法只处理基本信息，其它关联信息的保存交给适配成功的SalePolicyRegister注册器，过程为：
     * 1、验证和保存基本信息
     * 2、将保存可能的商品信息、可能的预算信息、政策逻辑等扩展信息的过程交给正确的SalePolicyRegister
     * 3、完成保存后，进行添加事件的通知
     * */
    
    // 优惠政策创建操作需要应该考虑客户多次重复点击的情况出现
    String prefix = salePolicy.getPrefix();
    Validate.notBlank(prefix , "错误的优惠政策预添加标记，请检查!!");
    Validate.isTrue(StringUtils.isNotBlank(this.redisMutexService.getMCode(SALE_POLICY_GEN_PREFIX, prefix)) , "没有发现优惠政策的预添加标记，可能是因为重复操作的原因，请使用preSave功能进行添加!!");
    boolean isLock = false;
    try {
      if(isLock = this.redisMutexService.tryLock(prefix, TimeUnit.MILLISECONDS, 1)) {
        this.redisMutexService.setMCode(SALE_POLICY_GEN_PREFIX, prefix, prefix, 1l);
      } else {
        throw new IllegalArgumentException("请不要重复进行优惠政策添加!!");
      }
    } finally {
      if(isLock) {
        this.redisMutexService.unlock(prefix);
      }
    }
    
    // 1、======
    String type = salePolicy.getType(); 
    String tenantCode = TenantUtils.getTenantCode();
    Validate.notBlank(type , "创建时，错误的优惠政策类型信息，请检查!!"); 
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = this.findSalePolicyTempleteRegister(type);
    String templeteCode = salePolicy.getTempleteCode();
    Validate.notBlank(templeteCode , "创建时，必须传入优惠政策模板业务编号信息，请检查!!");
    SalePolicyTemplete existSalePolicyTemplete = this.salePolicyTempleteRepository.findByTempleteCode(templeteCode, tenantCode);
    Validate.notNull(existSalePolicyTemplete , "创建时，未找到指定的优惠政策模板信息(%s)，请检查" , templeteCode);
    Validate.isTrue(StringUtils.equals(existSalePolicyTemplete.getType(), type) , "创建时，优惠政策设定的优惠政策类型和优惠政策模板的类型不一致，请检查!!");
    Boolean supportProduct = matchedSalePolicyTempleteRegister.supportProduct();
    Validate.notNull(supportProduct , "当前优惠政策匹配的注册器(%s)，并未指定当前优惠政策是针对整单的还是针对商品的，请检查并完善!!" , type);
    salePolicy.setWholePolicy(!supportProduct); 
    salePolicy.setTenantCode(tenantCode);
    salePolicy.setId(null);
    this.saveValidation(salePolicy, false);
    // 正式进行基本信息的保存
    SalePolicy salePolicyEntity = this.nebulaToolkitService.copyObjectByWhiteList(salePolicy, SalePolicy.class, LinkedHashSet.class, ArrayList.class);
    this.salePolicyRepository.save(salePolicyEntity);
    salePolicy.setId(salePolicyEntity.getId());
    
    // 2、=======
    this.handleExtendsInfos(false , matchedSalePolicyTempleteRegister, salePolicy, null , supportProduct);
    
    // 3、=======
    if(!CollectionUtils.isEmpty(salePolicyEventListeners)) {
      for (SalePolicyEventListener salePolicyEventListener : salePolicyEventListeners) {
        salePolicyEventListener.onCreated(salePolicy);
      }
    }
    // 记录操作日志
    this.recordLogOnCreate(salePolicy);
    return salePolicy;
  }

  /**
   * 记录创建日志
   *
   * @param salePolicyVo 销售策略模板
   */
  private void recordLogOnCreate(SalePolicyVo salePolicyVo) {
    SalePolicyLogDto logDto = new SalePolicyLogDto();
    logDto.setOriginal(null);
    logDto.setNewest(salePolicyVo);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyLogEventListener, SalePolicyLogDto> onCreate = SalePolicyLogEventListener::onCreate;
    this.nebulaNetEventClient.publish(logDto, SalePolicyLogEventListener.class, onCreate);
  }
  
  /**
   * 该私有方法针对优惠政策的扩展信息进行数据运维（优惠政策创建场景）：门槛数据、商品本品数据、优惠执行策略信息、客户范围信息、限量控制信息
   * @param salePolicy
   * @param salePolicyStickupListener
   */
  @SuppressWarnings("unchecked")
  private <T extends AbstractSalePolicyThreshold , P extends AbstractSalePolicyProductInfo ,R extends AbstractSalePolicyExecutorInfo> void handleExtendsInfos(boolean update , SalePolicyTempleteRegister salePolicyTempleteRegister , SalePolicyVo salePolicy  , SalePolicyVo oldSalePolicy , boolean supportProduct) {
    String salePolicyCode = salePolicy.getSalePolicyCode();
    String tenantCode = salePolicy.getTenantCode();
    SalePolicyStickupListener<T, P> salePolicyStickupListener = (SalePolicyStickupListener<T, P>)salePolicyTempleteRegister.getStickupListener();
    // 优惠政策特性执行（可能有）
    Collection<Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>>> characteristicStrategyClasses = salePolicyTempleteRegister.getCharacteristicStrategyClasses();
    // 注意，绑定在这个模板下的所有可用优惠政策特性执行策略，都要工作。
    if(!CollectionUtils.isEmpty(characteristicStrategyClasses)) {
      Set<CharacteristicStrategy<? extends AbstractCharacteristicInfo>> characteristicStraties = this.findAllBindingCharacteristicStrategy(salePolicyTempleteRegister);
      for (CharacteristicStrategy<? extends AbstractCharacteristicInfo> characteristicStrategy : characteristicStraties) {
        characteristicStrategy.onSaveCharacteristicInfo(update, salePolicy, oldSalePolicy);
      }
    }
    // 优惠政策门槛（一定有）
    T salePolicyProductThreshold = (T)salePolicy.getSalePolicyProductThreshold();
    Validate.notNull(salePolicyProductThreshold , "维护优惠政策时，未发现任何优惠政策门槛信息，如果当前添加的优惠政策没有任何门槛，也应该设定一个门槛值为0的门槛（请参见SalePolicyThresholdVo模型）");
    salePolicyProductThreshold.setSalePolicyCode(salePolicyCode);
    salePolicyStickupListener.onSaveSalePolicyProductThreshold(update, salePolicy , oldSalePolicy , salePolicyProductThreshold);
    // 优惠政策涉及的执行政策（一定有）
    Set<R> salePolicyExecutorInfos = (Set<R>)salePolicy.getSalePolicyExecutorInfos();
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyExecutorInfos) , "维护优惠政策时，至少需要一组优惠政策规则执行详情（executorInfo），请检查!!");
    salePolicyExecutorInfos.stream().forEach(item -> {
      item.setSalePolicyCode(salePolicyCode);
      item.setTenantCode(tenantCode);
    });
    Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> salePolicyExecuteStrategies = this.findAllBindingExecuteStrategy(salePolicyTempleteRegister);
    for (SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy : salePolicyExecuteStrategies) {
      salePolicyExecuteStrategy.onSaveSalePolicyExecutorInfo(update, salePolicy, oldSalePolicy);
    }
    // 优惠政策商品（可能）
    Set<P> salePolicyProductInfos = (Set<P>)salePolicy.getSalePolicyProductInfos();
    if(!supportProduct) {
      Validate.isTrue(CollectionUtils.isEmpty(salePolicyProductInfos) , "由于当前优惠政策不是针对商品本品的，所以当前优惠政策不能传入商品信息!!");
    } else {
      Validate.isTrue(!CollectionUtils.isEmpty(salePolicyProductInfos) , "由于当前优惠政策是针对商品本品进行的，所以当前优惠政策必须传入商品信息!!");
      salePolicyProductInfos.stream().forEach(item -> {
        item.setSalePolicyCode(salePolicyCode);
        item.setTenantCode(tenantCode);
      });
      salePolicyStickupListener.onSaveSalePolicyProductInfo(update , salePolicy, oldSalePolicy , salePolicyProductInfos);
    }
    // TODO 优惠政策涉及的预算情况（可能有）
    // =======
    // 优惠政策涉及的限量控制（可能有）
    // bindableLimitStrategyClasses这是该优惠政策可以绑定的限量控制策略（并不是已绑定的）
    Collection<Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>>> bindableLimitStrategyClasses = salePolicyTempleteRegister.getBindableLimitStrategyClasses();
    if(!CollectionUtils.isEmpty(bindableLimitStrategyClasses)) {
      Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> salePolicyLimitStrategies = this.findAllBindingLimitStrategy(salePolicyTempleteRegister);
      for (SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy : salePolicyLimitStrategies) {
        salePolicyLimitStrategy.onSaveSalePolicyLimitInfos(update, salePolicy, oldSalePolicy);
      }
    }
    // 优惠政策涉及的客户返回（一定）
    Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> currentCustomerScopeStrategies = this.findAllBindCustomerScopeStrategy(salePolicyTempleteRegister);
    for (SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeStrategy : currentCustomerScopeStrategies) {
      salePolicyCustomerScopeStrategy.onSaveSalePolicyCustomerInfo(update, salePolicy, oldSalePolicy);
    }
  }
  
  /**
   * 在创建一个新的SalePolicy模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   * @param update 本次验证是创建性质的验证还是修改性质的验证？true：表示是修改性质的验证；false表示是创建性质的验证
   */
  private void saveValidation(SalePolicyVo salePolicy , boolean update) {
    String onTimeValue;
    Date now = new Date();
    String tenantCode = TenantUtils.getTenantCode();
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication , "未发现登录信息，请检查!!");
    String currentAccount =authentication.getName();
    if(!update) {
      onTimeValue = "添加优惠政策时";
      salePolicy.setId(null);
      salePolicy.setCreateAccount(currentAccount);
      salePolicy.setCreateTime(now);
      salePolicy.setModifyAccount(currentAccount);
      salePolicy.setModifyTime(now);
      salePolicy.setTenantCode(tenantCode);
      Validate.isTrue(StringUtils.isBlank(salePolicy.getId()), onTimeValue + "，当期信息的数据编号（主键）不能有值！");
      String salePolicyCode = salePolicy.getSalePolicyCode();
      // 如果没有设定优惠政策的业务编号，则在这里生成一个
      if(StringUtils.isBlank(salePolicyCode)) {
        salePolicyCode = StringUtils.join(tenantCode , new SimpleDateFormat("yyyyMMddHHmmssS").format(now));
        salePolicy.setSalePolicyCode(salePolicyCode);
      }
      Validate.notBlank(salePolicyCode , onTimeValue + "，必须传入/设置优惠政策业务编号信息，请检查！！");
      SalePolicy exsitSalePolicy = this.salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
      Validate.isTrue(exsitSalePolicy == null , onTimeValue + "，当前使用的优惠政策业务编号已经被使用，请检查!!");
      Boolean effective = salePolicy.getEffective();
      if(effective == null) {
        salePolicy.setEffective(true);
      }
    } else {
      onTimeValue = "修改优惠政策时";
      salePolicy.setModifyAccount(currentAccount);
      salePolicy.setModifyTime(now);
      String salePolicyCode = salePolicy.getSalePolicyCode();
      Validate.notBlank(salePolicyCode ,  onTimeValue + "，原始优惠政策的业务编号信息必须传入，请检查!!");
      Validate.isTrue(!StringUtils.isBlank(salePolicy.getId()), onTimeValue + "，当期信息的数据编号（主键）必须有值！");
      SalePolicy exsitSalePolicy = this.salePolicyRepository.findBySalePolicyCodeAndTenantCodeWithoutId(salePolicyCode, tenantCode, salePolicy.getId());
      Validate.isTrue(exsitSalePolicy == null , onTimeValue + "，传入的优惠政策业务编号已经被使用，请检查更换!!");
      Boolean effective = salePolicy.getEffective();
      Validate.notNull(effective , onTimeValue + "，优惠政策的有效信息必须传入，请检查!!");
    }
    Validate.notNull(salePolicy , onTimeValue + "，信息对象必须传入!!");

    // ======== 首先验证基本信息
    Validate.notBlank(salePolicy.getSalePolicyName(), onTimeValue + "，优惠政策名称不能为空！");
    // 有效时间判定
    Date validStartTime = salePolicy.getValidStartTime();
    Validate.notNull(validStartTime , onTimeValue + "，优惠起始时间不能为空！");
    Date validEndTime = salePolicy.getValidEndTime();
    Validate.notNull(validEndTime , onTimeValue + "，优惠截止时间不能为空！");
    long validStartTimeValue = validStartTime.getTime();
    long validEndTimeValue = validEndTime.getTime();
    Validate.isTrue(validStartTimeValue < validEndTimeValue , onTimeValue + "，优惠起始时间必须小于优惠截止时间！");
    Validate.isTrue(now.getTime() < validStartTimeValue , onTimeValue + "，优惠开始时间必须在当前时间之后");
    // 中文描述信息
    Validate.notBlank(salePolicy.getDescription(), onTimeValue + "，优惠政策描述信息不能为空！");
    // 是否有效，添加时都设定为“有效”
    salePolicy.setEffective(true);
    Validate.notNull(salePolicy.getOverlap(), onTimeValue + "，是否允许政策叠加不能为空！");
    String label = salePolicy.getLabel();
    Validate.notBlank(label , onTimeValue + "，优惠政策简称不能为空！");
    
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(salePolicy.getDescription() == null || salePolicy.getDescription().length() < 400 , onTimeValue + "优惠政策中文描述信息,在进行添加时填入值超过了限定长度(200个字符)，请检查!");
    Validate.isTrue(salePolicy.getSalePolicyCode() == null || salePolicy.getSalePolicyCode().length() < 128 , onTimeValue + "促销编码,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(salePolicy.getSalePolicyName() == null || salePolicy.getSalePolicyName().length() < 128 , onTimeValue + "促销名称,在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(salePolicy.getDescription() == null || salePolicy.getDescription().length() < 512 , onTimeValue + "中文描述信息,在进行添加时填入值超过了限定长度(512)，请检查!");
  }

  @Override
  @Transactional
  public SalePolicyVo update(JSONObject salePolicyJson) {
    Validate.notNull(salePolicyJson , "修改优惠政策时，必须传入优惠政策信息（JSON形式）。");
    SalePolicyVo currentSalePolicyVo = this.convertSalePolicyVo(salePolicyJson, "修改时");
    this.updateHandle(currentSalePolicyVo);
    // 刷新缓存
    String tenantCode = TenantUtils.getTenantCode();
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
    return currentSalePolicyVo;
  }

  @Override
  @Transactional
  public SalePolicyVo update(SalePolicyVo salePolicy) {
    Validate.notNull(salePolicy , "修改优惠政策时，必须传入优惠政策信息（对象形式）。");
    this.updateHandle(salePolicy);
    // 刷新缓存
    String tenantCode = TenantUtils.getTenantCode();
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
    return salePolicy;
  }
  
  /**
   * @return 注意，这里返回的是没有修改前的优惠政策信息
   */
  private SalePolicyVo updateHandle(SalePolicyVo salePolicy) {
    /*
     * 对特定的优惠政策进行修改的过程，包括：
     * 1、对基本信息进行修改（包括进行验证）
     * 2、而对各种关联资源进行修改的操作，建议各种已注册的实现，采用先删除关联再进行重建的方式进行
     * （还是将修改可能的商品信息、可能的预算信息、政策逻辑等扩展信息的过程交给正确的SalePolicyRegister）
     * 3、完成保存后，进行添加事件的通知
     * */
    // 优惠政策创建操作需要应该考虑客户多次重复点击的情况出现
    String prefix = salePolicy.getPrefix();
    Validate.notBlank(prefix , "错误的优惠政策预修改标记，请检查!!");
    Validate.isTrue(StringUtils.isNotBlank(this.redisMutexService.getMCode(SALE_POLICY_GEN_PREFIX, prefix)) , "没有发现优惠政策的预添加标记，可能是因为重复操作的原因，请使用preSave功能进行添加!!");
    boolean isLock = false;
    try {
      if(isLock = this.redisMutexService.tryLock(prefix, TimeUnit.MILLISECONDS, 1)) {
        this.redisMutexService.setMCode(SALE_POLICY_GEN_PREFIX, prefix, prefix, 1l);
      } else {
        throw new IllegalArgumentException("请不要重复进行优惠政策添加!!");
      }
    } finally {
      if(isLock) {
        this.redisMutexService.unlock(prefix);
      }
    }
    Date now = new Date();
    
    // 1、======
    Validate.notNull(salePolicy , "修改时，必须传入要修改的优惠政策信息（包括关联信息），请参见文档!!");
    String salePolicyCode = salePolicy.getSalePolicyCode();
    Validate.notBlank(salePolicyCode , "修改时，未发现将要修改的优惠政策有正确的业务编号，请检查!!");
    String salePolicyId = salePolicy.getId();
    Validate.notBlank(salePolicyId , "修改时，未发现将要修改的优惠政策有正确的id技术编号，请检查!!");
    SalePolicy currentSalePolicy = this.salePolicyRepository.findBySalePolicyId(salePolicyId);
    Validate.notNull(currentSalePolicy , "修改时，未发现将要修改的优惠政策信息，请检查!!");
    SalePolicyVo currentSalePolicyVo = this.nebulaToolkitService.copyObjectByBlankList(currentSalePolicy, SalePolicyVo.class, HashSet.class, LinkedList.class);
    // 从缓存中取得修改前的优惠政策信息（一定有）
    String oldSalePolicyCode = currentSalePolicy.getSalePolicyCode();
    SalePolicyVo oldSalePolicyVo = this.findDetailsByCode(oldSalePolicyCode);
    Validate.notNull(oldSalePolicyVo , "修改时，未找到指定优惠政策的历史信息，请检查!!");
    boolean effective = currentSalePolicy.getEffective();
    Date validStartTime = currentSalePolicy.getValidStartTime();
    if(!effective || now.getTime() > validStartTime.getTime()) {
      throw new IllegalArgumentException("修改时，已失效/已结束的优惠政策 , 不允许进行修改;  请检查!!");
    }
    // 不允许传入错误的优惠政策模板编号(优惠政策对应的优惠政策模板编号不允许修改)
    Validate.isTrue(StringUtils.equals(currentSalePolicy.getTempleteCode(), oldSalePolicyVo.getTempleteCode()) , "修改时，传入了错误的优惠政策编号信息，请检查!!");
    // 促销类型不允许修改
    String policyType =  salePolicy.getType();
    String tenantCode = TenantUtils.getTenantCode();
    Validate.notNull(policyType , "优惠策略类型必须传入，请检查!!");
    Validate.isTrue(StringUtils.equals(policyType, currentSalePolicy.getType()) , "修改时，优惠政策类型不允许修改，请检查!!");
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = this.findSalePolicyTempleteRegister(policyType); 
    Validate.notNull(salePolicy , "修改时，必须传入要添加的优惠政策信息（包括关联信息），请参见文档!!");
    Boolean supportProduct = matchedSalePolicyTempleteRegister.supportProduct();
    salePolicy.setWholePolicy(!supportProduct); 
    salePolicy.setTenantCode(tenantCode);
    // 但请注意，针对基本信息来说，并不是所有基本信息都允许修改
    currentSalePolicy.setSalePolicyCode(salePolicyCode);
    currentSalePolicy.setDescription(salePolicy.getDescription());
    currentSalePolicy.setLabel(salePolicy.getLabel());
    currentSalePolicy.setOverlap(salePolicy.getOverlap());
    currentSalePolicy.setSalePolicyName(salePolicy.getSalePolicyName());
    currentSalePolicy.setValidEndTime(salePolicy.getValidEndTime());
    currentSalePolicy.setValidStartTime(salePolicy.getValidStartTime());
    currentSalePolicy.setPromotionalGoods(salePolicy.getPromotionalGoods());
    this.salePolicyRepository.saveOrUpdate(currentSalePolicy);
    
    // 2、======
    this.handleExtendsInfos(true, matchedSalePolicyTempleteRegister, salePolicy, oldSalePolicyVo, supportProduct);
    
    // 3、======
    if(!CollectionUtils.isEmpty(salePolicyEventListeners)) {
      for (SalePolicyEventListener salePolicyEventListener : salePolicyEventListeners) {
        salePolicyEventListener.onUpdate(salePolicy);
      }
    }
    // 记录操作日志
    this.recordLogOnUpdate(currentSalePolicyVo, salePolicy);
    return oldSalePolicyVo;
  }

  /**
   * 记录更新日志
   *
   * @param currentSalePolicy 目前销售政策
   * @param salePolicy        销售政策
   */
  private void recordLogOnUpdate(SalePolicyVo currentSalePolicy, SalePolicyVo salePolicy) {
    SalePolicyLogDto logDto = new SalePolicyLogDto();
    logDto.setOriginal(currentSalePolicy);
    logDto.setNewest(salePolicy);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyLogEventListener, SalePolicyLogDto> onUpdate = SalePolicyLogEventListener::onUpdate;
    this.nebulaNetEventClient.publish(logDto, SalePolicyLogEventListener.class, onUpdate);
  }

  /**
   * 将不确定格式的json信息转换为SalePolicyVo对象——借助正确的SalePolicyRegister的帮助
   * @param json
   * @param onTime 是“创建时”，还是“修改时”
   * @return
   */
  private SalePolicyVo convertSalePolicyVo(JSONObject json , String onTime) {
    Validate.notNull(json , onTime + "，错误的json格式信息");
    String type = json.getString("type");
    Validate.notBlank(type , onTime + "，错误的优惠政策类型信息，请检查!!");
    SalePolicyTempleteRegister matchedSalePolicyRegister = this.findSalePolicyTempleteRegister(type);
    Boolean supportProduct = matchedSalePolicyRegister.supportProduct();
    Validate.notNull(supportProduct , onTime + "，错误的商品支撑信息，请检查!!");
    SalePolicyStickupListener<? extends AbstractSalePolicyThreshold, ? extends AbstractSalePolicyProductInfo> stickupListener = null;
    stickupListener = matchedSalePolicyRegister.getStickupListener();
    Validate.notNull(stickupListener , onTime + "，未发现匹配的优惠政策具体注册器存在数据运维监听SalePolicyStickupListener，请检查!!");
    
    SalePolicyVo salePolicyVo = JSON.parseObject(json.toJSONString(), SalePolicyVo.class);
    // 选中按可能的执行特性信息信息转换 characteristicInfos
    JSONArray characteristicInfoArray = json.getJSONArray("characteristicInfos");
    if(!CollectionUtils.isEmpty(characteristicInfoArray)) {
      Set<AbstractCharacteristicInfo> characteristicInfos = Sets.newLinkedHashSet();
      for(int index = 0 ; index < characteristicInfoArray.size() ; index++) {
        JSONObject characteristicInfoJsonObject = characteristicInfoArray.getJSONObject(index);
        String characteristicCode = characteristicInfoJsonObject.getString("characteristicCode");
        Validate.notBlank(characteristicCode , "在进行优惠政策转换时，发现至少有一个特性执行信息（characteristicCode属性）没有对应的特性执行策略，请检查!!");
        CharacteristicStrategy<? extends AbstractCharacteristicInfo> currentCharacteristicStrategy = this.findCorrectCharacteristicStrategy(matchedSalePolicyRegister, characteristicCode);
        Class<? extends AbstractCharacteristicInfo> currentCharacteristicInfoClass = currentCharacteristicStrategy.getCharacteristicInfoClass();
        AbstractCharacteristicInfo characteristicInfo = JSONObject.toJavaObject(characteristicInfoJsonObject, currentCharacteristicInfoClass);
        characteristicInfos.add(characteristicInfo);
      }
      salePolicyVo.setCharacteristicInfos(characteristicInfos);
    }
    // 商品信息转换(商品优惠场景下一定有)
    if(supportProduct) {
      Class<? extends AbstractSalePolicyProductInfo> currentSalePolicyProductInfoClass = stickupListener.getSalePolicyProductInfoClass();
      Validate.notNull(currentSalePolicyProductInfoClass , "发现类型(%s)为的优惠政策模板注册，没有指定具体的SalePolicyProductInfo类信息，请检查!!" , type);
      JSONArray salePolicyProductInfoArray = json.getJSONArray("salePolicyProductInfos");
      Validate.isTrue(!CollectionUtils.isEmpty(salePolicyProductInfoArray) , "在进行优惠政策转换时，未设定任何一个商品信息，请检查!!");
      List<? extends AbstractSalePolicyProductInfo> salePolicyProductInfos = JSONArray.parseArray(salePolicyProductInfoArray.toJSONString(), currentSalePolicyProductInfoClass);
      Validate.isTrue(!CollectionUtils.isEmpty(salePolicyProductInfos) , "发现类型(%s)为的优惠政策模板注册，没有正确转换商品信息，请检查" , type);
      salePolicyVo.setSalePolicyProductInfos(Sets.newLinkedHashSet(salePolicyProductInfos));
    }
    // 寻找客户范围策略的class转换，每组客户范围依次转换
    JSONObject salePolicyCustomerInfoMaps = json.getJSONObject("salePolicyCustomerInfos");
    Validate.notNull(salePolicyCustomerInfoMaps , onTime + "，未发现任何客户范围选择信息，请检查!!");
    Set<String> scopeTypes = salePolicyCustomerInfoMaps.keySet();
    Map<String , Set<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerInfoObjectMaps = Maps.newLinkedHashMap();
    for (String customerScopeType : scopeTypes) {
      SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> currentCustomerScopeStrategy = this.findCorrectCustomerScopeStrategy(matchedSalePolicyRegister, customerScopeType);
      Class<? extends AbstractSalePolicyCustomerScopeInfo> currentSalePolicyCustomerInfoClass = currentCustomerScopeStrategy.getSalePolicyCustomerInfoClass();
      Validate.notNull(currentSalePolicyCustomerInfoClass , "发现类型(%s)为的优惠政策模板注册，没有指定具体的SalePolicyCustomerInfo类信息，请检查!!" , customerScopeType);
      JSONArray customerInfoArray = salePolicyCustomerInfoMaps.getJSONArray(customerScopeType);
      if(!CollectionUtils.isEmpty(customerInfoArray)) {
        List<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerInfos = JSONArray.parseArray(customerInfoArray.toJSONString(), currentSalePolicyCustomerInfoClass);
        salePolicyCustomerInfoObjectMaps.put(customerScopeType, Sets.newLinkedHashSet(salePolicyCustomerInfos));
      }
    }
    salePolicyVo.setCustomerScopeMapping(salePolicyCustomerInfoObjectMaps);
    // 转换的门槛信息（一定有）
    Class<? extends AbstractSalePolicyThreshold> currentSalePolicyProductThresholdClass = stickupListener.getSalePolicyProductThresholdClass();
    Validate.notNull(currentSalePolicyProductThresholdClass , "发现类型(%s)为的优惠政策门槛，没有指定具体的AbstractSalePolicyProductThreshold类信息，请检查!!" , type);
    JSONObject salePolicyProductThresholdObject = json.getJSONObject("salePolicyProductThreshold");
    if(salePolicyProductThresholdObject != null) {
      AbstractSalePolicyThreshold salePolicyThreshold = JSONObject.toJavaObject(salePolicyProductThresholdObject, currentSalePolicyProductThresholdClass);
      salePolicyVo.setSalePolicyProductThreshold(salePolicyThreshold);
    }
    // 寻找限量策略的class转换(进行抽象集合的转换)
    JSONArray salePolicyLimitInfoArray = json.getJSONArray("salePolicyLimitInfos");
    if(!CollectionUtils.isEmpty(salePolicyLimitInfoArray)) {
      Set<AbstractSalePolicyLimitInfo> salePolicyLimitInfos = Sets.newLinkedHashSet();
      for(int index = 0 ; index < salePolicyLimitInfoArray.size() ; index++) {
        JSONObject limitInfoJsonObject = salePolicyLimitInfoArray.getJSONObject(index);
        String limitStrategyCode = limitInfoJsonObject.getString("limitStrategyCode");
        Validate.notBlank(limitStrategyCode , "在进行优惠政策转换时，发现至少有一个限价政策没有设置编号信息（limitStrategyCode属性），请检查!!");
        SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = this.findCorrectLimitStrategy(matchedSalePolicyRegister, limitStrategyCode);
        Class<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitInfoClass = currentSalePolicyLimitStrategy.getSalePolicyLimitInfoClass();
        AbstractSalePolicyLimitInfo salePolicyLimitInfo = JSONObject.toJavaObject(limitInfoJsonObject, currentSalePolicyLimitInfoClass);
        salePolicyLimitInfos.add(salePolicyLimitInfo);
      }
      salePolicyVo.setSalePolicyLimitInfos(salePolicyLimitInfos);
    }
    // 执行规则(一定有，进行抽象对象的转换)
    JSONArray salePolicyExecutorInfoArray = json.getJSONArray("salePolicyExecutorInfos");
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyExecutorInfoArray) , "在进行优惠政策转换时，未设定任何一种优惠执行信息，请检查!!");
    Set<AbstractSalePolicyExecutorInfo> salePolicyExecutorInfos = Sets.newLinkedHashSet();
    for(int index = 0 ; index < salePolicyExecutorInfoArray.size() ; index++) {
      JSONObject salePolicyExecutorInfoObject = salePolicyExecutorInfoArray.getJSONObject(index);
      String executeStrategyCode = salePolicyExecutorInfoObject.getString("executeStrategyCode");
      Validate.notBlank(executeStrategyCode , "在进行优惠政策转换时，未发现指定的优惠政策执行策略编号，请检查!!");
      SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy = this.findCorrectExecuteStrategy(matchedSalePolicyRegister, executeStrategyCode);
      Class<? extends AbstractSalePolicyExecutorInfo> salePolicyExecutorInfoClass = salePolicyExecuteStrategy.getSalePolicyExecutorInfoClass();
      AbstractSalePolicyExecutorInfo salePolicyExecutorInfo = JSONObject.toJavaObject(salePolicyExecutorInfoObject, salePolicyExecutorInfoClass);
      salePolicyExecutorInfos.add(salePolicyExecutorInfo);
      // 验证其中的“阶梯循环规则”是否被支持（一定有）
      String cycleRuleCode = salePolicyExecutorInfo.getCycleRuleCode();
      Validate.notBlank(cycleRuleCode , "进行优惠政策维护时，必须传入优惠政策使用的“阶梯循环规则”(cycleRuleCode)，请检查!!");
      Collection<Class<? extends SalePolicyCycleRuleStrategy>> cycleRuleStrategyClasses = salePolicyExecuteStrategy.getCycleRuleStrategyClasses();
      Validate.isTrue(!CollectionUtils.isEmpty(cycleRuleStrategyClasses) , "发现类型(%s)为的优惠政策模板注册，未设定任何一个“阶梯循环规则”（SalePolicyCycleRuleStrategy），请检查!!" , type);
      boolean matchedSalePolicyCycleRuleStrategy = false;
      for (Class<? extends SalePolicyCycleRuleStrategy> cycleRuleStrategyClass : cycleRuleStrategyClasses) {
        SalePolicyCycleRuleStrategy salePolicyCycleRuleStrategy = this.applicationContext.getBean(cycleRuleStrategyClass);
        if(StringUtils.equals(salePolicyCycleRuleStrategy.getCycleRuleCode(), cycleRuleCode)) {
          matchedSalePolicyCycleRuleStrategy = true;
          break;
        }
      }
      Validate.isTrue(matchedSalePolicyCycleRuleStrategy , "未发现指定的“阶梯循环规则”（%s） , 请检查!!" , cycleRuleCode);
    }
    salePolicyVo.setSalePolicyExecutorInfos(salePolicyExecutorInfos);
    return salePolicyVo;
  }
  
  private Set<CharacteristicStrategy<? extends AbstractCharacteristicInfo>> findAllBindingCharacteristicStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>>> characteristicStrategyClasses = salePolicyTempleteRegister.getCharacteristicStrategyClasses();
    if(CollectionUtils.isEmpty(characteristicStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些特性执行策略可以进行绑定
    Set<CharacteristicStrategy<? extends AbstractCharacteristicInfo>> bindableCharacteristicStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>> characteristicStrategyClass : characteristicStrategyClasses) {
      try {
        CharacteristicStrategy<? extends AbstractCharacteristicInfo> currentCharacteristicStrategy = applicationContext.getBean(characteristicStrategyClass);
        bindableCharacteristicStrategySet.add(currentCharacteristicStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableCharacteristicStrategySet;
  }
  
  /**
   * 该私有方法用于基于特定的优惠政策模板查询所有已绑定的优惠执行策略
   * @param salePolicyTempleteRegister 
   * @return
   */
  private Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> findAllBindingExecuteStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>>> bindableExecuteStrategyClasses = salePolicyTempleteRegister.getExecuteStrategyClasses();
    if(CollectionUtils.isEmpty(bindableExecuteStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些限价政策可以进行绑定
    Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategyClass : bindableExecuteStrategyClasses) {
      try {
        SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> currentSalePolicyExecuteStrategy = applicationContext.getBean(bindableExecuteStrategyClass);
        bindableExecuteStrategySet.add(currentSalePolicyExecuteStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableExecuteStrategySet;
  }
  
  /**
   * 在指定的salePolicyTempleteRegister优惠政策模板注册器中，找到正确的执行特性策略
   * @return 如果没有找到，则返回null
   */
  private CharacteristicStrategy<? extends AbstractCharacteristicInfo> findCorrectCharacteristicStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String characteristicStrategyCode) {
    Collection<Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>>> characteristicStrategyClasses = salePolicyTempleteRegister.getCharacteristicStrategyClasses();
    if(CollectionUtils.isEmpty(characteristicStrategyClasses)) {
      return null;
    }
    for (Class<? extends CharacteristicStrategy<? extends AbstractCharacteristicInfo>> characteristicStrategyClass : characteristicStrategyClasses) {
      try {
        CharacteristicStrategy<? extends AbstractCharacteristicInfo> characteristicStrategy = applicationContext.getBean(characteristicStrategyClass);
        if(StringUtils.equals(characteristicStrategy.characteristicCode(), characteristicStrategyCode)) {
          return characteristicStrategy;
        }
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        return null;
      }
    }
    return null;
  }
  
  /**
   * 该私有方法用于查询符合要求的限价政策（基于特定的优惠政策）
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> findCorrectExecuteStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String executeStrategyCode) {
    Set<SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategies = this.findAllBindingExecuteStrategy(salePolicyTempleteRegister);
    Validate.isTrue(!CollectionUtils.isEmpty(bindableExecuteStrategies) , "未发现当前优惠政策有至少一个能绑定的优惠政策执行策略，请检查!!");
    HashMap<String, SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo>> bindableExecuteStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> salePolicyExecuteStrategy : bindableExecuteStrategies) {
      bindableExecuteStrategyMapping.put(salePolicyExecuteStrategy.getExecuteStrategyCode(), salePolicyExecuteStrategy);
    }
    
    // 寻找正确的策略处理器，并进行保存处理
    Validate.notBlank(executeStrategyCode , "至少一个将要保存的优惠执行策略没有指定executorCode信息，请检查!!");
    SalePolicyExecuteStrategy<? extends AbstractSalePolicyExecutorInfo> currentSalePolicyExecuteStrategy = bindableExecuteStrategyMapping.get(executeStrategyCode);
    Validate.notNull(currentSalePolicyExecuteStrategy , "未匹配到业务编号为:%s 的优惠执行策略信息，可能是该政策不存在或者该优惠政策不允许关联这个执行政策，请检查!!" , executeStrategyCode);
    return currentSalePolicyExecuteStrategy;
  }
  
  /**
   * 该私有方法用于基于特定的优惠政策查询所有已绑定的限量政策
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> findAllBindingLimitStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>>> bindableLimitStrategyClasses = salePolicyTempleteRegister.getBindableLimitStrategyClasses();
    if(CollectionUtils.isEmpty(bindableLimitStrategyClasses)) {
      return null;
    }
    
    // 确认有哪些限价政策可以进行绑定
    Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategySet = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyClass : bindableLimitStrategyClasses) {
      try {
        SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = applicationContext.getBean(bindableLimitStrategyClass);
        bindableLimitStrategySet.add(currentSalePolicyLimitStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableLimitStrategySet;
  }
  
  /**
   * 该私有方法用于查询符合要求的限价政策（基于特定的优惠政策）
   * @param matchedSalePolicyRegister
   * @param limitStrategyCode
   * @return
   */
  private SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> findCorrectLimitStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String limitStrategyCode) {
    Set<SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategies = this.findAllBindingLimitStrategy(salePolicyTempleteRegister);
    Validate.isTrue(!CollectionUtils.isEmpty(bindableLimitStrategies) , "未发现当前优惠政策有至少一个能绑定的限价政策，请检查!!");
    HashMap<String, SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo>> bindableLimitStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> salePolicyLimitStrategy : bindableLimitStrategies) {
      bindableLimitStrategyMapping.put(salePolicyLimitStrategy.getLimitStrategyCode(), salePolicyLimitStrategy);
    }
    
    // 寻找正确的策略处理器，并进行保存处理
    Validate.notBlank(limitStrategyCode , "至少一个将要保存的现价策略没有指定code信息，请检查!!");
    SalePolicyLimitStrategy<? extends AbstractSalePolicyLimitInfo> currentSalePolicyLimitStrategy = bindableLimitStrategyMapping.get(limitStrategyCode);
    Validate.notNull(currentSalePolicyLimitStrategy , "未匹配到业务编号为:%s 的限量政策信息，可能是该政策不存在或者改优惠政策不允许关联这个限量政策，请检查!!" , limitStrategyCode);
    return currentSalePolicyLimitStrategy;
  }
  
  /**
   * 该私有方法为寻找匹配的人员范围选择策略
   * @return
   */
  private SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> findCorrectCustomerScopeStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister , String customerScopeType) {
    Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> salePolicyCustomerScopeStrategies = this.findAllBindCustomerScopeStrategy(salePolicyTempleteRegister);
    HashMap<String, SalePolicyCustomerScopeStrategy<?>> bindableCustomerScopeStrategyMapping = Maps.newLinkedHashMap();
    for (SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo> salePolicyCustomerScopeStrategy : salePolicyCustomerScopeStrategies) {
      bindableCustomerScopeStrategyMapping.put(salePolicyCustomerScopeStrategy.getScopeType(), salePolicyCustomerScopeStrategy);
    }
    // 寻找正确的客户范围处理策略，并进行保存
    Validate.notBlank(customerScopeType , "该优惠政策保存时，并没有设定客户范围的选择类型(或者根本没有传入)，请检查!!");
    SalePolicyCustomerScopeStrategy<?> currentCustomerScopeStrategy = bindableCustomerScopeStrategyMapping.get(customerScopeType);
    Validate.notNull(currentCustomerScopeStrategy , "该优惠政策保存时，并没有发现指定类型的客户范围选择策略（%s），请检查!!");
    return currentCustomerScopeStrategy;
  }
  
  /**
   * 该私有方法为寻找匹配的人员范围选择策略
   * @return
   */
  private Set<SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>> findAllBindCustomerScopeStrategy(SalePolicyTempleteRegister salePolicyTempleteRegister) {
    Collection<Class<? extends SalePolicyCustomerScopeStrategy<? extends AbstractSalePolicyCustomerScopeInfo>>> salePolicyCustomerScopeStrategyClasses = salePolicyTempleteRegister.getCustomerScopeStrategyClasses();
    Validate.isTrue(!CollectionUtils.isEmpty(salePolicyCustomerScopeStrategyClasses) , "该优惠政策并没有注册任何客户返回选择方式，请检查!!");
    Set<SalePolicyCustomerScopeStrategy<?>> bindableCustomerScopeStrategies = Sets.newLinkedHashSet();
    for (Class<? extends SalePolicyCustomerScopeStrategy<?>> salePolicyCustomerScopeStrategyClass : salePolicyCustomerScopeStrategyClasses) {
      try {
        SalePolicyCustomerScopeStrategy<?> currentSalePolicyCustomerScopeStrategy = applicationContext.getBean(salePolicyCustomerScopeStrategyClass);
        String currentScopeType = currentSalePolicyCustomerScopeStrategy.getScopeType();
        Validate.notBlank(currentScopeType , "发现至少一个客户范围策略没有指定code信息(%s)，请检查!!" , salePolicyCustomerScopeStrategyClass.getName());
        bindableCustomerScopeStrategies.add(currentSalePolicyCustomerScopeStrategy);
      } catch(RuntimeException e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e.getMessage() , e);
      }
    }
    return bindableCustomerScopeStrategies;
  }

  /**
   * 该私有方法，根据type获得正确的SalePolicyRegister营销策略注册器
   * @param cacheResults
   * @return
   */
  private SalePolicyTempleteRegister findSalePolicyTempleteRegister(String type) {
    Validate.notBlank("错误的优惠政策类型，请检查传入值");
    Validate.isTrue(!CollectionUtils.isEmpty(this.salePolicyTempleteRegisters) , "错误的SalePolicyTempleteRegister（集合）信息，请检查!!");
    SalePolicyTempleteRegister matchedSalePolicyTempleteRegister = null;
    for (SalePolicyTempleteRegister item : salePolicyTempleteRegisters) {
      if(StringUtils.equals(type, item.getType())) {
        matchedSalePolicyTempleteRegister = item;
        break;
      }
    }
    Validate.notNull(matchedSalePolicyTempleteRegister , "未匹配任何SalePolicyTempleteRegister信息，请检查!!");
    return matchedSalePolicyTempleteRegister;
  }
  
  @Override
  @Transactional
  public void invalid(String[] salePolicyCodes) {
    Validate.isTrue(salePolicyCodes != null && salePolicyCodes.length > 0 , "进行优惠政策失效操作时，必须传入优惠政策业务编码，请检查!!");
    String tenantCode = TenantUtils.getTenantCode();
    for (String salePolicyCode : salePolicyCodes) {
      SalePolicy currentSalePolicy = this.salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
      Validate.notNull(currentSalePolicy , "进行优惠政策失效操作时，指定的优惠政策必须存在(%s)" , salePolicyCode);
      SalePolicyVo currentSalePolicyVo = this.findDetailsByCode(currentSalePolicy.getSalePolicyCode());
      
      // 先进行作废操作，再进行缓存通知(主要是effective属性)
      currentSalePolicy.setEffective(false);
      currentSalePolicy.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
      this.salePolicyRepository.saveOrUpdate(currentSalePolicy);
      if(!CollectionUtils.isEmpty(salePolicyEventListeners)) {
        for (SalePolicyEventListener salePolicyEventListener : salePolicyEventListeners) {
          salePolicyEventListener.onUpdate(currentSalePolicyVo);
        }
      }
      // 记录操作日志
      this.recordLogOnDisable(currentSalePolicy);
    }

    // 进行缓存清理
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
  }

  /**
   * 禁用日志记录
   *
   * @param currentSalePolicy 目前销售政策
   */
  private void recordLogOnDisable(SalePolicy currentSalePolicy) {
    SalePolicyVo voDb = new SalePolicyVo();
    voDb.setId(currentSalePolicy.getId());
    voDb.setEffective(true);

    SalePolicyVo vo = new SalePolicyVo();
    vo.setId(currentSalePolicy.getId());
    vo.setEffective(false);

    SalePolicyLogDto logDto = new SalePolicyLogDto();
    logDto.setOriginal(voDb);
    logDto.setNewest(vo);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyLogEventListener, SalePolicyLogDto> OnDisable = SalePolicyLogEventListener::OnDisable;
    this.nebulaNetEventClient.publish(logDto, SalePolicyLogEventListener.class, OnDisable);
  }

  @Override
  @Transactional
  public void effective(String[] salePolicyCodes) {
    Validate.isTrue(salePolicyCodes != null && salePolicyCodes.length > 0 , "进行优惠政策启用操作时，必须传入优惠政策业务编码，请检查!!");
    String tenantCode = TenantUtils.getTenantCode();
    
    for (String salePolicyCode : salePolicyCodes) {
      SalePolicy currentSalePolicy = this.salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
      Validate.notNull(currentSalePolicy , "进行优惠政策启用操作时，指定的优惠政策必须存在（%s）" , salePolicyCode);
      SalePolicyVo currentSalePolicyVo = this.findDetailsByCode(currentSalePolicy.getSalePolicyCode());
      
      // 先进行启用操作，再进行缓存通知(主要是effective属性)
      currentSalePolicy.setEffective(true);
      currentSalePolicy.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      this.salePolicyRepository.saveOrUpdate(currentSalePolicy);
      if(!CollectionUtils.isEmpty(salePolicyEventListeners)) {
        for (SalePolicyEventListener salePolicyEventListener : salePolicyEventListeners) {
          salePolicyEventListener.onUpdate(currentSalePolicyVo);
        }
      }
      // 记录操作日志
      this.recordLogOnEnable(currentSalePolicy);
    }
    // 进行缓存清理
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
  }

  /**
   * 记录启用日志
   *
   * @param currentSalePolicy 目前销售政策
   */
  private void recordLogOnEnable(SalePolicy currentSalePolicy) {
    SalePolicyVo voDb = new SalePolicyVo();
    voDb.setId(currentSalePolicy.getId());
    voDb.setEffective(false);

    SalePolicyVo vo = new SalePolicyVo();
    vo.setId(currentSalePolicy.getId());
    vo.setEffective(true);

    SalePolicyLogDto logDto = new SalePolicyLogDto();
    logDto.setOriginal(voDb);
    logDto.setNewest(vo);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyLogEventListener, SalePolicyLogDto> OnEnable = SalePolicyLogEventListener::OnEnable;
    this.nebulaNetEventClient.publish(logDto, SalePolicyLogEventListener.class, OnEnable);
  }

  @Override
  @Transactional
  public void deleteBySalePolicyCode(String salePolicyCode) {
    Validate.notBlank(salePolicyCode , "进行优惠政策删除时，必须传入优惠政策业务编码，请检查!!");
    SalePolicyVo currntSalePolicy = this.findDetailsByCode(salePolicyCode);
    Validate.notNull(currntSalePolicy , "进行优惠政策删除时，未发现指定的优惠政策信息，请检查!!");
    String tenantCode = TenantUtils.getTenantCode();
    
    Set<SalePolicyRecord> salePolicyRecords = this.salePolicyRecordRepository.findByTenantCodeAndSalePolicyCode(tenantCode, salePolicyCode);
    Validate.isTrue(CollectionUtils.isEmpty(salePolicyRecords) , "进行优惠政策删除时，发现该优惠政策已经产生了执行流水，不能不删除！");
    this.salePolicyRepository.deleteByIds(Sets.newHashSet(currntSalePolicy.getId()));
    // 记录操作日志
    this.recordLogOnDelete(currntSalePolicy);
    // 进行缓存清理
    this.salePolicyCacheLoadingService.notifyCacheRefresh(tenantCode);
  }

  /**
   * 记录删除日志
   *
   * @param currntSalePolicy currnt销售政策
   */
  private void recordLogOnDelete(SalePolicyVo currntSalePolicy) {
    SalePolicyVo voDb = new SalePolicyVo();
    voDb.setId(currntSalePolicy.getId());
    SalePolicyLogDto logDto = new SalePolicyLogDto();
    logDto.setOriginal(voDb);
    logDto.setNewest(null);
    // 订单创建事件
    SerializableBiConsumer<SalePolicyLogEventListener, SalePolicyLogDto> OnDelete = SalePolicyLogEventListener::OnDelete;
    this.nebulaNetEventClient.publish(logDto, SalePolicyLogEventListener.class, OnDelete);
  }

  // ============= 以下是和缓存以及缓存查询有关
  
  @Override
  public SalePolicyVo findDetailsByCode(String salePolicyCode) {
    String tenantCode = TenantUtils.getTenantCode();
    if(StringUtils.isBlank(salePolicyCode)) {
      return null;
    }
    
    // ====== 试图从缓存加载
    Set<SalePolicyVo> results = this.salePolicyCacheLoadingService.findByTenantCode(tenantCode, Sets.newHashSet(salePolicyCode));
    // 注意，如果缓存里面没有，可能并不是真的没有
    // 目前已经发现缓存加载时，由于远程调用超时的原因，导致可能有10%的几率，不能完全加载优惠政策信息
    // 这个时候，就从数据层直接进行查询，如果数据层存在这个优惠政策信息，那么就清理缓存，并重新要求进行缓存加载(不过只进行本地缓存加载即可)
    if(CollectionUtils.isEmpty(results)) {
      SalePolicy repositorySalePolicy = this.salePolicyRepository.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode);
      if(repositorySalePolicy != null) {
        this.salePolicyCacheLoadingService.reloadingCache(tenantCode);
        return findDetailsByCode(salePolicyCode);
      } else {
        return null;
      }
    }
    
    // 全深度拷贝，避免调用者对缓存数据进行修改
    SalePolicyVo result = results.iterator().next();
    try {
      return this.single(result);
    } catch(CloneNotSupportedException e) {
      LOGGER.warn(e.getMessage(), e);
      return null;
    }
  }

  @Override
  public Set<SalePolicyVo> findDetailsByProcessing(boolean processing) {
    String tenantCode = TenantUtils.getTenantCode();
    Set<SalePolicyVo> resultSalePolicyVos = Sets.newLinkedHashSet();
    
    // ====== 试图从缓存加载
    Date now = new Date();
    Set<SalePolicyVo> salePolicies = this.salePolicyCacheLoadingService.findByTenantCode(tenantCode, null);
    if(CollectionUtils.isEmpty(salePolicies)) {
      return resultSalePolicyVos;
    }
    for (SalePolicyVo salePolicyVoItem : salePolicies) {
      /*
       * 进行中的优惠活动一定满足两个条件：
       * 1、这个优惠政策本身是有效的
       * 2、当前时间刚好处于这个优惠政策的活动时间范围内
       * */
      boolean isProcessing = false;
      if(salePolicyVoItem.getEffective() 
          && now.after(salePolicyVoItem.getValidStartTime()) && now.before(salePolicyVoItem.getValidEndTime())) {
        isProcessing = true;
      }
      // 确认是否要加入到结果集合
      if(processing == isProcessing) {
        resultSalePolicyVos.add(salePolicyVoItem);
      }
    }
    
    // 深度copy后返回
    try {
      return this.multiple(resultSalePolicyVos);
    } catch(CloneNotSupportedException e) {
      LOGGER.warn(e.getMessage(), e);
      return null;
    }
  }

  @Override
  public Set<SalePolicyVo> findDetailsByProcessingAndCustomerCode(String tenantCode, boolean processing, String customerCode, String relevanceCode) {
    if(StringUtils.isAnyBlank(tenantCode , customerCode)) {
      return null;
    }
    
    // ====== 试图从缓存加载
    Set<SalePolicyVo> salePolicies = this.salePolicyCacheLoadingService.findByTenantCode(tenantCode, null);
    if(CollectionUtils.isEmpty(salePolicies)) {
      return null;
    }
    Set<SalePolicyVo> filterResultSalePolicyVos = Sets.newLinkedHashSet();
    for (SalePolicyVo salePolicyVoItem : salePolicies) {
      if(!salePolicyVoItem.getEffective()) {
        continue;
      }
      filterResultSalePolicyVos.add(salePolicyVoItem);
    }
    // 开始进行用户适用范围的匹配
    Set<SalePolicyVo> resultSalePolicyVos = this.salePolicyMatchedStrategy.match(filterResultSalePolicyVos, tenantCode, processing, customerCode, relevanceCode);
    
    // 深度copy后返回
    try {
      return this.multiple(resultSalePolicyVos);
    } catch(CloneNotSupportedException e) {
      LOGGER.warn(e.getMessage(), e);
      return null;
    }
  }

  @Override
  public Set<SalePolicyVo> findDetailsByProcessingAndCustomerCodesAndProductCodes(String tenantCode , boolean processing , String customerCode, String relevanceCode , String... productCodes) {
    if(StringUtils.isAnyBlank(tenantCode , customerCode) || productCodes == null || productCodes.length == 0) {
      return null;
    }
    
    // ====== 试图从缓存加载
    Set<SalePolicyVo> salePolicies = this.salePolicyCacheLoadingService.findByTenantCode(tenantCode, null);
    if(CollectionUtils.isEmpty(salePolicies)) {
      return null;
    }
    Set<SalePolicyVo> filterResultSalePolicyVos = Sets.newLinkedHashSet();
    for (SalePolicyVo salePolicyVoItem : salePolicies) {
      if(!salePolicyVoItem.getEffective()) {
        continue;
      }
      filterResultSalePolicyVos.add(salePolicyVoItem);
    }
    // 开始进行用户适用范围的匹配
    Set<SalePolicyVo> resultSalePolicyVos = this.salePolicyMatchedStrategy.match(filterResultSalePolicyVos, tenantCode, processing, customerCode, relevanceCode , productCodes);
    
    // 深度copy后返回
    try {
      return this.multiple(resultSalePolicyVos);
    } catch(CloneNotSupportedException e) {
      LOGGER.warn(e.getMessage(), e);
      return null;
    }
  }
  
  // ========= 以下查询过程直接来源于数据层
  
  @Override
  public Page<SalePolicyVo> findByConditions(Pageable pageable, SalePolicyPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new SalePolicyPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    
    // 校验后进行查询和转换
    Page<SalePolicy> entityPage = this.salePolicyRepository.findByConditions(pageable, dto);
    List<SalePolicy> entities = entityPage.getRecords();
    Page<SalePolicyVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    pageResult.setRecords(covertEntityToVo(entities));
    return pageResult;
  }
  
  /**
   * 进行Entity到VO的转换工作
   */
  private List<SalePolicyVo> covertEntityToVo(List<SalePolicy> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    // 这里要注意一个问题，由于直接来源于数据层，所以关于模板类型、模板注册类型等关联展示信息，需要进行补充
    List<String> templeteCodes = entities.stream().map(SalePolicy::getTempleteCode).distinct().collect(Collectors.toList());
    List<SalePolicyTemplete> salePolicyTempletes = this.salePolicyTempleteRepository.findByTempleteCodes(templeteCodes, TenantUtils.getTenantCode());
    // 这种情况不会出现
    if(CollectionUtils.isEmpty(salePolicyTempletes)) {
      LOGGER.warn("在进行优惠政策模板信息填充时，发现错误数据，处理过程终止!!");
      return Lists.newArrayList();
    }
    Map<String , SalePolicyTemplete> salePolicyTempleteCodeMapping = salePolicyTempletes.stream().collect(Collectors.toMap(SalePolicyTemplete::getTempleteCode, item -> item));
    // 从entity转换为vo
    List<SalePolicyVo> salePolicyVos = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, SalePolicy.class , SalePolicyVo.class, LinkedHashSet.class, ArrayList.class));
    if(CollectionUtils.isEmpty(salePolicyVos)) {
      return Lists.newArrayList();
    }
    
    // 开始进行模板相关的数据补充
    for (SalePolicyVo salePolicyItem : salePolicyVos) {
      String currentSalePolicyTempleteCode = salePolicyItem.getTempleteCode();
      SalePolicyTemplete currentSalePolicyTemplete = salePolicyTempleteCodeMapping.get(currentSalePolicyTempleteCode);
      // 脏数据时可能出现这个情况
      if(currentSalePolicyTemplete == null) {
        continue;
      }
      String templeteRegisterType = currentSalePolicyTemplete.getType();
      salePolicyItem.setTempleteName(currentSalePolicyTemplete.getTempleteName());
      SalePolicyTempleteRegister salePolicyTempleteRegister = this.findSalePolicyTempleteRegister(templeteRegisterType);
      if(salePolicyTempleteRegister != null) {
        salePolicyItem.setTempleteRegisterType(templeteRegisterType);
        salePolicyItem.setTempleteRegisterTypeDesc(salePolicyTempleteRegister.getTypeDesc());
      }
    }
    
    // 深度copy后返回
    try {
      return this.multiple(salePolicyVos);
    } catch(CloneNotSupportedException e) {
      LOGGER.warn(e.getMessage(), e);
      return null;
    }
  }
  
  /**
   * 该私有方法对单个优惠政策完整信息进行深度克隆，以保证调用者不经意的写操作行为对原始缓存数据造成影响
   * @param targetSalePolicyVo 
   * @return
   * @throws CloneNotSupportedException
   */
  private SalePolicyVo single(SalePolicyVo targetSalePolicyVo) throws CloneNotSupportedException {
    // 全深度clone
    byte[] contextBytes = null;
    ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
    try(ObjectOutputStream out = new ObjectOutputStream(byteStream);) {
      out.writeObject(targetSalePolicyVo);
      contextBytes = byteStream.toByteArray();
    } catch(IOException e) {
      throw new CloneNotSupportedException(e.getMessage());
    }
    
    // 生成一个新的对象SalePolicyVo对象
    SalePolicyVo cloneSalePolicyVo = null;
    try(ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(contextBytes));) {
      cloneSalePolicyVo = (SalePolicyVo)in.readObject();
    } catch(IOException | ClassNotFoundException e) {
      throw new CloneNotSupportedException(e.getMessage());
    }
    return cloneSalePolicyVo;
  }
  
  /**
   * 该私有方法对多个优惠政策完整信息集合进行深度克隆，以保证调用者不经意的写操作行为对原始缓存数据造成影响
   * @param targetSalePolicyVo 
   * @return
   * @throws CloneNotSupportedException
   */
  @SuppressWarnings("unchecked")
  private <T extends Collection<SalePolicyVo>> T multiple(Collection<SalePolicyVo> targetSalePolicyVos) throws CloneNotSupportedException {
    // 全深度clone
    byte[] contextBytes = null;
    ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
    try(ObjectOutputStream out = new ObjectOutputStream(byteStream);) {
      out.writeObject(targetSalePolicyVos);
      contextBytes = byteStream.toByteArray();
    } catch(IOException e) {
      throw new CloneNotSupportedException(e.getMessage());
    }
    
    // 重新组装新的集合
    T resultsCollection = null;
    try(ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(contextBytes));) {
      resultsCollection = (T)in.readObject();
    } catch(IOException | ClassNotFoundException e) {
      throw new CloneNotSupportedException(e.getMessage());
    }
    return resultsCollection;
  }
}