package com.biz.crm.dms.business.policy.local.service.task;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.commons.lang3.StringUtils;
import org.redisson.Redisson;
import org.redisson.api.RTopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.entity.SalePolicy;
import com.biz.crm.dms.business.policy.local.repository.SalePolicyRepository;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * 优惠政策的缓存加载服务，在这里将具体控制一次从缓存查询数据的请求的具体处理逻辑：</br>
 * 1、是直接从已加载好的缓存中取出优惠政策数据</br>
 * 2、还是重新做一次全新的缓存信息</p>
 * 
 * 当然调用者也可以直接要求缓存加载服务清理掉指定租户的所有优惠政策缓存信息
 * @author yinwenjie
 *
 */
@Component
public class SalePolicyCacheLoadingService {
  /**
   * 缓存变化订阅的固定键值(基于tenantCode信息的全局变化通知)
   */
  public static final String SALE_POLICY_NOTIFY = "_SALE_POLICY_NOTIFY";
  /**
   * 优惠政策的缓存按照功能唯一业务编号进行存储（注意，没有进行最终排序）</br>
   * 使用软连接保证内存不够时的清理</br>
   * 第一级Key：二级租户信息（默认为default）</br>
   * 第二级Key：优惠政策在二级租户下的唯一业务编码</br>
   * value：当前二级租户下缓存的有效优惠政策信息（结束时间在当前时间之前的优惠政策不会加载）</p>
   * 注意，特定tenantCode下的的Map为null或者为empty的意义是不一样的，只有为null的情况，才表示缓存没有加载
   */
  private static Map<String , Map<String , SalePolicyVo>> salePolicysCacheMapping = Maps.newConcurrentMap();
  @Autowired(required = false)
  private Redisson redisson;
  /**
   * 易变的属性标记当前全功能缓存是否正在进行刷新，以及是那个线程在进行刷新
   * K：二级租户信息
   * V：当前正在主导刷新的调用者线程；以及当前抢占到某个租户缓存刷新权的线程
   */
  private volatile Map<String , AtomicReference<Thread>> flashingThreadMapping = Maps.newConcurrentMap();
  /**
   * 保证在缓存刷新过程中，保证在多个线程同时要求进行缓存加载时，也可以有序处理加载要求</br>
   * 实际上由CAS的限制，这里的lock有点多余，但为了便于理解，还是加上</br>
   * K：二级租户信息
   * V：对应的可重入锁
   */
  private Map<String , ReentrantLock> salePolicyloadLockMapping = Maps.newConcurrentMap();
  
  /// ======= 以下变量和业务处理过程有关
  
  @Autowired(required = false)
  private SalePolicyRepository salePolicyRepository;
  @Autowired(required = false)
  private ApplicationContext applicationContext;
  @Autowired(required = false)
  @Qualifier("policyLoadingExecutor")
  private ThreadPoolExecutor policyLoadingExecutor;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(SalePolicyCacheLoadingService.class);
  
  // ======= 以下是和清理以及通知清理相关的方法
  
  /**
   * 该方法负责基于redis/MQ或一些其它的事件推送机制，通知全局节点某个租户下的优惠政策已经发生了变化
   * @param tenantCode 二级租户信息
   */
  public void notifyCacheRefresh(String tenantCode) {
    RTopic topic = redisson.getTopic(SalePolicyCacheLoadingService.SALE_POLICY_NOTIFY);
    topic.publish(StringUtils.join(tenantCode));
  }
  
  /**
   * 重加载优惠政策
   * @param tenantCode 二级租户信息
   */
  public void reloadingCache(String tenantCode) {
    if(StringUtils.isBlank(tenantCode)) {
      return;
    }
    /*
     * 处理过程为：
     * 1、锁准备阶段，保证每一个二级租户都有对应的可重入锁，已保证后续的安全性操作
     * 每一个租户的缓存加载只能一次一次的做
     * 2、只有获得了锁，才能开始后续操作
     * 接着，清理当前租户对应的缓存信息、变更其刷新状态
     * 并调用刷新过程
     * */
    
    // 1、=====
    ReentrantLock tenantReentrantLock = null;
    while((tenantReentrantLock = this.salePolicyloadLockMapping.get(tenantCode)) == null) {
      synchronized (SalePolicyCacheLoadingService.class) {
        while((tenantReentrantLock = this.salePolicyloadLockMapping.get(tenantCode)) == null) {
          tenantReentrantLock = new ReentrantLock();
          this.salePolicyloadLockMapping.put(tenantCode, tenantReentrantLock);
        }
      }
    }
    AtomicReference<Thread> tenantLeadingThreadReference = null;
    while((tenantLeadingThreadReference = this.flashingThreadMapping.get(tenantCode)) == null) {
      synchronized (SalePolicyCacheLoadingService.class) {
        while((tenantLeadingThreadReference = this.flashingThreadMapping.get(tenantCode)) == null) {
          tenantLeadingThreadReference = new AtomicReference<>();
          this.flashingThreadMapping.put(tenantCode, tenantLeadingThreadReference);
        }
      }
    }
    
    // 2、======
    Thread leadingThread = Thread.currentThread();
    boolean isFlashingThread = false;
    if(!(isFlashingThread = tenantLeadingThreadReference.compareAndSet(null, leadingThread))) {
      Thread.yield();
    }
    // 如果没有抢占到加载任务，则一直自旋，直到刷新过程结束
    if(!isFlashingThread) {
      while(tenantLeadingThreadReference.get() != null) {
        Thread.yield();
      } 
      return;
    }
    
    // 3、=====
    tenantReentrantLock.lock();
    try {
      // 由于要满足“优先查询”特性的要求，所以需要直接引入一个新的Map
      Map<String , SalePolicyVo> newSalePolicysCacheForTenantMapping = Maps.newConcurrentMap();
      salePolicysCacheMapping.put(tenantCode, newSalePolicysCacheForTenantMapping);
      // 设定各种标记后，然后正式开始进行缓存的加载
      this.cacheByTenantCodeFromRepository(tenantCode, newSalePolicysCacheForTenantMapping);
    } finally {
      tenantLeadingThreadReference.compareAndSet(leadingThread, null);
      tenantReentrantLock.unlock();
    }
  }
  
  /**
   * 该方法用于从数据持久层查询指定二级租户下目前所有执行中的和将要执行的优惠政策到缓存中</br>
   * 注意，该私有方法不负责获取操作权
   * @param newSalePolicysCacheForTenantMapping 这是本次加载过程得到的结果存储的目标Map集合
   */
  private void cacheByTenantCodeFromRepository(String tenantCode , Map<String , SalePolicyVo> newSalePolicysCacheForTenantMapping) {
    List<SalePolicy> allSalePolicies = this.salePolicyRepository.findByTenantCode(tenantCode);
    if(CollectionUtils.isEmpty(allSalePolicies)) {
      return;
    }
    
    // 开始一个一个查询（每一个优惠政策的查询，都需要较长时间，因为各种数据都需要完善）
    // 所以优惠政策的查询放到线程池中，以多个任务的方式快速执行
    Set<Future<SalePolicyVo>> loadingSet = Sets.newLinkedHashSet();
    for (SalePolicy salePolicy : allSalePolicies) {
      SalePolicyLoadingTask policyLoadingTask = this.applicationContext.getBean(SalePolicyLoadingTask.class , salePolicy , tenantCode);
      Future<SalePolicyVo> future = this.policyLoadingExecutor.submit(policyLoadingTask);
      loadingSet.add(future);
    }
    
    // 只有所有的都加载完成了，才能进行后续处理
    for (Future<SalePolicyVo> futureItem : loadingSet) {
      try {
        SalePolicyVo result = futureItem.get();
        if(result != null) {
          newSalePolicysCacheForTenantMapping.put(result.getSalePolicyCode(), result);
        }
      } catch(ExecutionException | InterruptedException e) {
        LOGGER.error(e.getMessage() , e);
        continue;
      }
    }
  }
  
  /**
   * 该方法用于从缓存中查询指定租户的优惠政策信息。
   * @param tenantCode
   * @param prioritySalePolicyCodes 该方法可以指定优先查询的优惠政策编号，当这些优惠政策被全部从缓存获取到以后
   * 无论这时缓存处于哪种状态，该方法都将被返回
   * @return
   */
  public Set<SalePolicyVo> findByTenantCode(String tenantCode , Set<String> prioritySalePolicyCodes) {
    if(StringUtils.isBlank(tenantCode)) {
      return null;
    }
    
    /*
     * 根据目前的缓存设计，该方法被调用时，当前缓存可能处于以下几种状态
     * 1、当前缓存还重没有被加载
     * 这种情况下salePolicyloadLockMapping相关信息为null、flashingThreadMapping相关信息为null
     * 2、当前缓存正在加载
     * 这种情况salePolicyloadLockMapping不为null、flashingThreadMapping信息不为null
     * 3、当前缓存同步完成
     * 这种情况salePolicyloadLockMapping不为null、flashingThreadMapping信息为null
     * 
     * 那么处理过程如下：
     * 1、这种情况下，直接调用cacheByTenantCodeFromRepository方法，要求其进行装载，并在完成后，继续处理后续动作
     * 但实际上这种情况不常见，因为init的时候，缓存就会装载
     * 2、这种情况下，如果没有prioritySalePolicyCodes信息，那么说明需要阻塞到缓存加载完成，才能进行后续处理
     * 如果有prioritySalePolicyCodes信息，那么只需要在salePolicyloadLockMapping的map集合中找到相关的优惠政策，就可以直接返回了
     * 3、这种情况下，直接从salePolicyloadLockMapping集合中找到相关优惠政策，就可以了，没有找到的话说明真的没有
     * */
    Set<SalePolicyVo> result = Sets.newLinkedHashSet();
    for(;;) {
      // 1、=======
      if(this.salePolicyloadLockMapping.get(tenantCode) == null || flashingThreadMapping.get(tenantCode) == null) {
        this.reloadingCache(tenantCode);
      }
      
      // 2、=======
      AtomicReference<Thread> tenantLeadingThreadReference = null;
      if(this.salePolicyloadLockMapping.get(tenantCode) != null 
          && (tenantLeadingThreadReference = flashingThreadMapping.get(tenantCode)) != null
          && tenantLeadingThreadReference.get() != null) {
        // 如果条件成立，那么需要阻塞到缓存加载完成
        if(CollectionUtils.isEmpty(prioritySalePolicyCodes)) {
          while(tenantLeadingThreadReference.get() != null) {
            Thread.yield();
          } 
        } else {
          // 只要有一个没有被查找到，就从头来
          Map<String, SalePolicyVo> cache = salePolicysCacheMapping.get(tenantCode);
          if(cache == null) {
            continue;
          }
          for (String prioritySalePolicyCode : prioritySalePolicyCodes) {
            SalePolicyVo currentSalePolicy = cache.get(prioritySalePolicyCode);
            if(currentSalePolicy == null) {
              break;
            } else {
              result.add(currentSalePolicy);
              // 如果条件成立，说明需要优先查询的优惠政策，在缓存加载完以前就已经被全部查询到了
              // 这时不用再等缓存过程结束，就可以返回结果了
              if(result.size() == prioritySalePolicyCodes.size()) {
                return result;
              }
            }
          }
        }
      }
      
      // 3、=======
      // 如果该条件成立，说明缓存已经加载完毕，没必要再考虑优先查询的问题了，直接查询缓存即可
      if(this.salePolicyloadLockMapping.get(tenantCode) != null 
          && (tenantLeadingThreadReference = flashingThreadMapping.get(tenantCode)) != null
          && tenantLeadingThreadReference.get() == null) {
        Map<String, SalePolicyVo> cache = salePolicysCacheMapping.get(tenantCode);
        if(CollectionUtils.isEmpty(prioritySalePolicyCodes)) {
          result = Sets.newLinkedHashSet(cache.values());
        } else {
          result = Sets.newLinkedHashSet();
          for (String prioritySalePolicyCode : prioritySalePolicyCodes) {
            SalePolicyVo currentSalePolicyVo = cache.get(prioritySalePolicyCode);
            if(currentSalePolicyVo != null) {
              result.add(currentSalePolicyVo);
            }
          }
        }
        return result;
      }
    }
  }
}
