package com.biz.crm.dms.business.policy.local.sharestrategy;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.local.context.DefaultPolicyExecuteContext;
import com.biz.crm.dms.business.policy.local.context.UnitpriceCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.ProductPolicyStepResult;
import com.biz.crm.dms.business.policy.sdk.context.StepType;
import com.biz.crm.dms.business.policy.sdk.context.UnitpriceCycleStepResult;
import com.biz.crm.dms.business.policy.sdk.context.UnitpriceProduct;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteShareStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.google.common.collect.Sets;

/**
 * 专门服务于诸如 商品-单价满减、商品-单价特价这样 的分摊逻辑（按本品金额进行分摊）。</p>
 * 
 * 该分摊逻辑，从2022-5-10创建，是优惠政策分摊逻辑的第二次需求变更。变更的核心点在于：</p>
 * 
 * 采用这种分摊逻辑进行计算后，各本品未参与优惠执行的剩余数量为0；</br>
 * 各本品未参与优惠执行的剩余金额为 = 各本品执行前的未参与优惠金额 - (各自单价优惠金额 * 本品数量)
 * 
 * 
 * 注意：直接减单价的优惠结束后，分摊时是否揉价的设定就不再起作用了，因为单价已经全部在本品上被剪掉了
 * 
 * @author yinwenjie
 */
@Component
public class UnitpriceShareStrategyWithAmountAndNumber implements SalePolicyExecuteShareStrategy {

  @Override
  public void share(String customer , SalePolicyVo salePolicy ,  AbstractPolicyExecuteContext abstractPolicyExecuteContext , AbstractCycleExecuteContext abstractCycleExecuteContext , Set<String> productCodes) {
    DefaultPolicyExecuteContext policyExecuteContext = (DefaultPolicyExecuteContext)abstractPolicyExecuteContext;
    UnitpriceCycleExecuteContext cycleExecuteContext = (UnitpriceCycleExecuteContext)abstractCycleExecuteContext;
    UnitpriceCycleStepResult cycleLastStepResult = cycleExecuteContext.findLastStepResult();
    
    /*
     * 因为按照单价进行优惠的情况，实际上已经将优惠总价分摊到了本品上。
     * 所以只需要基于cycleExecuteContext中的本品最新单价和小计价格，完成优惠政策上下文中的优惠步进创建，操作步骤如下：
     * 1、基于优惠本品状态，直接记录本品最新的单价和小计价格
     * 2、按照各个本品在数量中的比例，将总的优惠金额分摊到每个本品上，并记录到policyExecuteContext中
     * */
    
    // 1、=======
    Set<UnitpriceProduct> lastProducts = cycleLastStepResult.getLastProducts();
    Map<String , UnitpriceProduct> lastProductMapping = lastProducts.stream().collect(Collectors.toMap(UnitpriceProduct::getProductCode , item-> item));
    Set<ProductPolicyStepResult> newProductPolicyStepResults = Sets.newLinkedHashSet();
    for (String productCode : productCodes) {
      ProductPolicyStepResult productPolicyStepResult = policyExecuteContext.findLastProductStepResultByProductCode(productCode);
      // 找到对应的优惠本品状态
      UnitpriceProduct fullminusUnitpriceProduct = lastProductMapping.get(productCode);
      // 为当前商品添加一个新的优惠步进
      ProductPolicyStepResult newProductPolicyStepResult = new ProductPolicyStepResult();
      newProductPolicyStepResult.setStepType(salePolicy.getWholePolicy()?StepType.POLICY:StepType.PRODUCT);
      newProductPolicyStepResult.setSalePolicyCode(salePolicy.getSalePolicyCode());
      newProductPolicyStepResult.setExecuteCode(abstractCycleExecuteContext.getExecuteCode());
      newProductPolicyStepResult.setProductCode(productCode);
      Integer initNumbers = productPolicyStepResult.getInitNumbers();
      newProductPolicyStepResult.setInitNumbers(initNumbers);
      newProductPolicyStepResult.setInitPrices(productPolicyStepResult.getInitPrices());
      newProductPolicyStepResult.setInitSubtotal(productPolicyStepResult.getInitSubtotal());
      newProductPolicyStepResult.setPreGifts(productPolicyStepResult.getLastGifts());
      BigDecimal prePrices = productPolicyStepResult.getLastPrices();
      newProductPolicyStepResult.setPrePrices(productPolicyStepResult.getLastPrices());
      newProductPolicyStepResult.setPreSubtotal(productPolicyStepResult.getLastSubtotal());
      BigDecimal preSurplusTotalAmount = productPolicyStepResult.getLastSurplusTotalAmount();
      newProductPolicyStepResult.setPreSurplusTotalAmount(preSurplusTotalAmount);
      newProductPolicyStepResult.setPreSurplusTotalNumber(productPolicyStepResult.getLastSurplusTotalNumber());
      newProductPolicyStepResult.setPreGiftEnjoyedTotalNumber(productPolicyStepResult.getLastGiftEnjoyedTotalNumber());
      newProductPolicyStepResult.setPreGiftEnjoyedTotalAmount(productPolicyStepResult.getLastGiftEnjoyedTotalAmount());
      // 因为不是买赠优惠，所以赠品数量、已享受优惠的赠品数量和金额不会改变
      newProductPolicyStepResult.setLastGifts(productPolicyStepResult.getLastGifts());
      newProductPolicyStepResult.setLastGiftEnjoyedTotalNumber(productPolicyStepResult.getLastGiftEnjoyedTotalNumber());
      newProductPolicyStepResult.setLastGiftEnjoyedTotalAmount(productPolicyStepResult.getLastGiftEnjoyedTotalAmount());
      // 按照要求，执行单价满减后，该本品未进行优惠的数量，直接记为 0
      newProductPolicyStepResult.setLastSurplusTotalNumber(0);
      
      /*
       * 本质上要完成以下本品信息的步进变化：
       * a、记录单价减少后最新的单价信息（注意，单价可能为0）
       * b、记录单价减少后最新的本品小计信息（注意，小计价格可能为0）
       * c、计算这个本品还有多少小计价格，没有参与任何优惠
       * 还没有参与优惠的本品小计价格 = 商品之前没有参与优惠的小计价格 - (本品数量 * 单价差额)
       * 注意，如果计算结果小于0了，则记为0
       * */
      
      // a、===
      BigDecimal lastPrices = fullminusUnitpriceProduct.getPrices().setScale(4, RoundingMode.HALF_UP);
      newProductPolicyStepResult.setLastPrices(lastPrices);
      // b、===
      BigDecimal newLastSubtotal = new BigDecimal(fullminusUnitpriceProduct.getNumbers()).multiply(lastPrices);
      newProductPolicyStepResult.setLastSubtotal(newLastSubtotal);
      // d、===
      // 所以diffPrices就是可能出现的单价差，注意，单价差可能为0或者负数，如果为负数，则要设置成0
      BigDecimal diffPrices = prePrices.subtract(lastPrices);
      if(diffPrices.compareTo(BigDecimal.ZERO) != 1) {
        diffPrices = BigDecimal.ZERO;
      }
      BigDecimal productCycleLastRealSubtotalAmount = preSurplusTotalAmount.subtract(new BigDecimal(initNumbers).multiply(diffPrices).setScale(4, RoundingMode.HALF_UP));
      if(productCycleLastRealSubtotalAmount.compareTo(BigDecimal.ZERO) != 1) {
        newProductPolicyStepResult.setLastSurplusTotalAmount(BigDecimal.ZERO);
      } else {
        newProductPolicyStepResult.setLastSurplusTotalAmount(productCycleLastRealSubtotalAmount);
      }
      newProductPolicyStepResults.add(newProductPolicyStepResult);
    }
    policyExecuteContext.addPolicyStepResult(newProductPolicyStepResults);
  }
}
