package com.biz.crm.dms.business.policy.sdk.characteristic;

import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractCharacteristicInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;

/**
 * 优惠政策在执行时，会有很多执行特性，例如：只允许客户首单使用；不允许叠加使用；只允许客户使用一次；等等</p>
 * 
 * 在这种情况下，预筛选出来的优惠政策，需要执行的优惠政策，需要进入该策略的某一个（或多个）实现，判定其是否满足某一个执行特性。</p>
 * 
 * 根据判定结果，可分为：</br>
 * GRANTED：经过判定，本优惠政策确实可以执行，且不会影响被筛选的其它优惠政策执行</br>
 * DENIED：经过判定，本优惠政策不能执行，需要排除；但是不会影响被筛选的其它优惠政策执行</br>
 * MONOPOLY：经过判定，本优惠政策可以进入执行过程，但是被筛选的其它优惠政策均不能再执行</br>
 * 
 * </p>
 * 
 * 请注意，执行特性策略的适用场景是那些需要进行特定业务逻辑判定才能进行优惠执行的优惠政策，但是呢，这些优惠政策如果没有判定通过也不会引起全局性质的终止执行
 * 
 * @author yinwenjie
 */
public interface CharacteristicStrategy <C extends AbstractCharacteristicInfo> {
  /**
   * 该特性筛选策略在全系统中的唯一编号
   */
  public String characteristicCode();
  /**
   * 该特性筛选策略的中文描述信息，例如：只允许客户正确使用一次
   */
  public String characteristicName();
  /**
   * 该方法将返回这个具体特性执行策略的页面显示内容表达式。
   */
  public String getExpression();
  /**
   * 返回具体限量政策使用的具体对象描述信息
   */
  public Class<C> getCharacteristicInfoClass();
  /**
   * 当有一个具体优惠政策需要构造具体特性执行描述的时候，该方法会被触发（注意，这里需要返回的是当前优惠政策已经绑定的特性执行信息，而不是可以绑定的特性执行信息）
   * @param salePolicyCode 当前构造的具体优惠政策业务编号
   * @return 
   */
  public C onRequestCharacteristicInfo(String salePolicyCode);
  /**
   * 当需要具体的优惠政策进行绑定的一个或多个特性执行信息的维护时（可能是新增也可能是修改），该方法会被触发
   * @param update 表示当前优惠政策信息是被创建还是被修改，true表示是修改；其它值表示是创建
   * @param currentSalePolicy 当前正在进行维护的优惠政策基本信息（且这个基本信息已经维护成功）
   * @param oldSalePolicy 如果update为true，那么oldSalePolicy描述了修改前的优惠政策详细信息（包括所有的关联信息）
   */
  public void onSaveCharacteristicInfo(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy);
  /**
   * 做出决定，即在具体的优惠政策执行/预执行时，根据特定的特性执行策略做出决定：到底客户能不能参与这个优惠
   * @param executeContext 优惠政策执行上下文
   * @param salePolicyVo 当前需要进行执行特性决定的优惠政策详细信息
   * @param customerCode 当前执行优惠政策的客户（如经销商）信息，业务编号
   * @param formal 当前优惠政策是否是正式执行（例如，如果当前是预执行，则该入参为false）
   * @param relevanceCode 如果formal为true，那么该入参必然有值；该值表示和这个优惠政策执行过程关联的外部业务单据号（例如订单编号）
   * @return GRANTED、DENIED、MONOPOLY;
   */
  public CharacteristicVoter make(AbstractPolicyExecuteContext executeContext , SalePolicyVo salePolicyVo , String customerCode , boolean formal , String relevanceCode);
  /**
   * 当make方法做出了排除自身或者排除其他优惠政策的决定后，该方法会被触发。</br>
   * 以便向调用者返回优惠政策不能参与执行的具体原因，例如：被优惠政策“XXXX”的“不可累计”特性所影响
   * @return
   */
  public String errorMsg(SalePolicyVo salePolicyVo);
}
