package com.biz.crm.dms.business.policy.sdk.context;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Deque;
import java.util.Map;
import java.util.Set;

/**
 * 这是一个标识接口，实现该接口的模型定义都属于优惠政策的执行上下文，例如在policy.local模块中就有一个该标识接口的实现
 * TODO policy模块中，优惠政策部分的模型要做整理
 * @author yinwenjie
 *
 */
public interface AbstractPolicyExecuteContext extends Cloneable , Serializable {
  /**
   * 查询指定本品目前已匹配的优惠政策（编码）信息。所有这个商品理论上可以参与的优惠政策都在这里返回
   * @param productCode 指定的商品业务编号
   * @return 如果当前本品没有匹配任何优惠政策，则返回null
   */
  public Set<String> findPolicyCodesByProductCode(String productCode);
  /**
   * 根据一个商品信息，查询得到这个商品信息在本次优惠执行过程中的每一个优惠步骤</br>
   * 需要注意的是，调用者不应该改变返回值集合中的数据，否则执行上下文的结果正确性会受到影响
   * @param productCode 指定的商品业务编号
   * @return
   */
  public Deque<ProductPolicyStepResult> findPolicyStepResultsByProductCode(String productCode);
  /**
   * 查询最后一次有效执行的优惠政策
   * @return
   */
  public PolicyStepResult findLastStepResult();
  /**
   * 按照商品本品业务编号查询这个商品在上下文中最新的优惠状态信息 
   * @param productCode 指定的商品业务编号
   * @return 如果没有找到，则返回null
   */
  public ProductPolicyStepResult findLastProductStepResultByProductCode(String productCode);
  /**
   * 按照指定的商品编码查询，这个商品应该参与但是由于门槛未到等错误原因，最终未能参与的优惠政策（人为排除的优惠政策不在此列）
   * @param productCode 指定的商品编码信息
   * @return
   */
  public Map<String , String> findExcludedPolicyByProductCode(String productCode);
  /**
   * 根据指定的优惠政策（业务编号）查询这个业务编号在执行后，一共优惠的多少金额的信息。
   * 注意，不是“有多少金额享受了优惠”，而是实际优惠了多少金额
   * @param salePolicyCode 优惠政策（业务编号）
   * @return BigDecimal
   */
  public BigDecimal findEnjoyedTotalAmount(String salePolicyCode);
  /**
   * 根据指定的优惠政策（业务编号）查询这个业务编号在执行后，一共优惠的多少数量的信息.
   * 注意，由于本品数量不会减少，所以，实际上该方法和“有多少数量享受了优惠”的意义是相同的
   * @param salePolicyCode 优惠政策（业务编号）
   * @return Integer
   */
  public Integer findEnjoyedTotalNumber(String salePolicyCode);
  /**
   * 根据指定的优惠政策（业务编号）查询这个业务编号在执行后，一共优惠的多少金额的赠品信息。
   * 注意，不是“有多少金额享受了优惠”，而是实际优惠了多少金额
   * @param salePolicyCode 优惠政策（业务编号）
   * @return BigDecimal
   */
  public BigDecimal findGiftEnjoyedTotalAmount(String salePolicyCode);
  /**
   * 根据指定的优惠政策（业务编号）查询这个业务编号在执行后，一共优惠的多少数量的赠品信息.
   * 注意，由于本品数量不会减少，所以，实际上该方法和“有多少数量享受了优惠”的意义是相同的
   * @param salePolicyCode 优惠政策（业务编号）
   * @return Integer
   */
  public Integer findGiftEnjoyedTotalNumber(String salePolicyCode);
  /**
   * 该功能负责在完成计算后（无论计算结果如何），在正式返回给调用者之前；
   * 需要对上下文中的记录信息进行精简。</p>
   * 请注意，该方法应该在整个营销过程完成后，最后再进行调用
   */
  public AbstractPolicyExecuteContext streamlining();
}