package com.biz.crm.dms.business.policy.standard.fullminus.executestrategy;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.local.cyclestrategy.AccumulationCycleRuleStrategy;
import com.biz.crm.dms.business.policy.local.cyclestrategy.MultipleCycleRuleStrategy;
import com.biz.crm.dms.business.policy.local.cyclestrategy.SingleCycleRuleStrategy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutor;
import com.biz.crm.dms.business.policy.local.service.SalePolicyExecutorService;
import com.biz.crm.dms.business.policy.local.sharestrategy.UnitpriceShareStrategyWithAmountAndNumber;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorLadderVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorVo;
import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.local.context.UnitpriceCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.UnitpriceCycleStepResult;
import com.biz.crm.dms.business.policy.sdk.context.UnitpriceProduct;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteShareStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

/**
 * 在商品满减优惠方式中使用的，满数量享单价满减的具体优惠执行策略
 * @author yinwenjie
 */
public class ProductFullminusUnitpriceCountExecuteStrategy extends AbstractFullminusUnitpriceExecuteStrategy {

  @Autowired(required = false)
  private SalePolicyExecutorService salePolicyExecutorService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  
  @Override
  public String getExecuteStrategyCode() {
    return "productFullminusUnitpriceCountExecuteStrategy";
  }

  @Override
  public String getExecuteStrategyDesc() {
    return "满数量享单价满减";
  }

  @Override
  public String getExpression() {
    return "本单中本品数量购买满{integer:input-integer:fullCountValue}销售单位，各本品单价最多减{decimal:input-random:priceValue}元";
  }
  
  @Override
  public Class<SalePolicyExecutorVo> getSalePolicyExecutorInfoClass() {
    return SalePolicyExecutorVo.class;
  }
  
  @Override
  public String getCycleRuleExample(String cycleRuleCode) {
    if(StringUtils.equals(cycleRuleCode, SingleCycleRuleStrategy.SINGLE)) {
      return "例如：满1000销售单位各本品单价最多减3元，满500销售单位各本品单价最多减1元。那么本品销售数量为1300单位时，则总价最多可优惠3900元 ";
    } else if(StringUtils.equals(cycleRuleCode, MultipleCycleRuleStrategy.MULTIPLE)) { 
      return "例如：满1000销售单位各本品单价最多减3元，满500销售单位各本品单价最多减1元。那么本品销售数量为1300单位时，则总价最多可优惠3900元（因享受1000销售单位区间满减后，剩余未享受优惠的数量已不足500）";
    } else if(StringUtils.equals(cycleRuleCode, AccumulationCycleRuleStrategy.ACCUMULATION)) { 
      return "例如：满1000销售单位各本品单价最多减3元，满500销售单位各本品单价最多减1元。那么本品销售数量为2600单位时，则总价最多可优惠7800 + 7800 + 2600 = 18200元";
    }
    return "";
  }
  
  @Override
  public Class<? extends SalePolicyExecuteShareStrategy> getSalePolicyExecuteShareStrategy() {
    return UnitpriceShareStrategyWithAmountAndNumber.class;
  }
  
  @Override
  public Collection<Class<? extends SalePolicyCycleRuleStrategy>> getCycleRuleStrategyClasses() {
    // 标品中的三种阶梯循环方式，都支持
    return Lists.newArrayList(SingleCycleRuleStrategy.class , MultipleCycleRuleStrategy.class , AccumulationCycleRuleStrategy.class);
  }
  
  @Override
  public List<SalePolicyExecutorVo> onRequestSalePolicyExecutorInfo(String tenantCode , String salePolicyCode) {
    List<SalePolicyExecutor> salePolicyExecutors = this.salePolicyExecutorService.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode , getExecuteStrategyCode());
    if(CollectionUtils.isEmpty(salePolicyExecutors)) {
      return null;
    }
    // 转换成特定的VO（集合）
    Collection<SalePolicyExecutorVo> salePolicyExecutorVos = this.nebulaToolkitService.copyCollectionByWhiteList(salePolicyExecutors, SalePolicyExecutor.class, SalePolicyExecutorVo.class, LinkedHashSet.class, ArrayList.class, "salePolicyExecutorLadders" , "salePolicyExecutorLadders.executorLadderVars");
    return Lists.newArrayList(salePolicyExecutorVos);
  }

  @Override
  @Transactional
  public void onSaveSalePolicyExecutorInfo(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy) {
    this.handleSaveSalePolicyExecutorInfo(update, currentSalePolicy, oldSalePolicy, currentSalePolicy.getSalePolicyExecutorInfos().stream().map(item -> (SalePolicyExecutorVo)item).collect(Collectors.toSet()));
    super.validatetExpressionParams(currentSalePolicy);
  }
  
  @Override
  public AbstractCycleExecuteContext buildCycleExecuteContext(AbstractPolicyExecuteContext abstractExecuteContext, String executorCode ,  Set<SalePolicyConProduct> initPolicyProducts, String customerCode) {
    return super.buildFullminusUnitpriceCycleExecuteContext(executorCode , abstractExecuteContext, initPolicyProducts, customerCode);
  }
  
  public boolean execute(AbstractCycleExecuteContext abstractCycleExecuteContext , int index, int times , Map<String, Object> varParams , SalePolicyExecutorLadderVo salePolicyExecutorLadder) {
    // ======== 以下正式开始进行计算
    UnitpriceCycleExecuteContext cycleExecuteContext = (UnitpriceCycleExecuteContext)abstractCycleExecuteContext;
    UnitpriceCycleStepResult lastCycleStepResult = cycleExecuteContext.findLastStepResult();
    // 该变量是指，截止上次优惠后，还有多少本品小计数量没有参加优惠
    Integer preSurplusSubtotalNumber = lastCycleStepResult.getLastSurplusSubtotalNumber();
    // 这里一定是有值的
    Object fullCountValueObject = varParams.get("fullCountValue");
    Object priceValueObject = varParams.get("priceValue");
    if(fullCountValueObject == null || priceValueObject == null) {
      return false;
    }
    Integer fullCountValue = (Integer)fullCountValueObject;
    BigDecimal priceValue = (BigDecimal)priceValueObject;
    // 如果条件成立，说明不满足优惠条件
    if(fullCountValue > preSurplusSubtotalNumber) {
      return false;
    }
    
    // ======== 如果处理逻辑走到了这里，说明符合该阶梯的处理前提，开始正式进行优惠处理
    /*
     * 按单价优惠就找出参与这个优惠的所有本品，并且每个本品减少单价后，再进行计算   处理过程为：
     * 1、取得当前阶梯循环上下文中，上一个阶梯中的所有本品信息，
     * 并进行本品的单价扣减（并记录汇总的优惠金额）
     * 2、接着记录当前阶梯计算后的优惠状态
     * */ 
    
    // 1、=======
    BigDecimal preSubtotalAmount = lastCycleStepResult.getLastSubtotalAmount();
    BigDecimal preSurplusSubtotalAmount = lastCycleStepResult.getLastSurplusSubtotalAmount();
    Set<UnitpriceProduct> preProducts = lastCycleStepResult.getLastProducts();
    UnitpriceCycleStepResult newFullminusUnitpriceCycleStepResult = new UnitpriceCycleStepResult(index, times, preSubtotalAmount, preSurplusSubtotalAmount, preSurplusSubtotalNumber , preProducts);
    // 这个变量记录了各个本品在减去单价后，汇总的优惠金额
    BigDecimal newCycleAmountResult = BigDecimal.ZERO;
    Set<UnitpriceProduct> lastProducts = Sets.newLinkedHashSet();
    for (UnitpriceProduct preFullminusUnitpriceProduct : preProducts) {
      String productCode = preFullminusUnitpriceProduct.getProductCode();
      BigDecimal prePrices = preFullminusUnitpriceProduct.getPrices();
      Integer productNumber = preFullminusUnitpriceProduct.getNumbers();
      // 重新计算单价
      BigDecimal lastPrices = prePrices.subtract(priceValue).setScale(4, RoundingMode.HALF_UP);
      // 如果条件成立，说明计算后本品单价小于零，那么就取单价为0
      if(lastPrices.compareTo(BigDecimal.ZERO) == -1) {
        lastPrices = BigDecimal.ZERO;
        newCycleAmountResult = newCycleAmountResult.add(BigDecimal.ZERO);
      } else {
        newCycleAmountResult = newCycleAmountResult.add(priceValue.multiply(new BigDecimal(productNumber)).setScale(4, RoundingMode.HALF_UP));
      }
      // 记录这个本品优惠后的状态
      UnitpriceProduct lastFullminusUnitpriceProduct = new UnitpriceProduct(productCode, productNumber, lastPrices);
      lastProducts.add(lastFullminusUnitpriceProduct);
    }
    
    // 2、=======
    // 计算优惠后的金额： 原总计金额 - 小计折扣金额
    BigDecimal result = preSubtotalAmount.subtract(newCycleAmountResult).setScale(4, RoundingMode.HALF_UP);
    newFullminusUnitpriceCycleStepResult.setLastSubtotalAmount(result);
    // 本次计算结果作为下次计算的起始依据
    newFullminusUnitpriceCycleStepResult.setLastSurplusSubtotalAmount(result);
    newFullminusUnitpriceCycleStepResult.setLastSurplusSubtotalNumber(preSurplusSubtotalNumber - fullCountValue);
    newFullminusUnitpriceCycleStepResult.setLastProducts(lastProducts);
    cycleExecuteContext.addStepResult(newFullminusUnitpriceCycleStepResult);
    return true;
  }
}
