package com.biz.crm.dms.business.policy.standard.given.executestrategy;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.biz.crm.dms.business.policy.sdk.context.CycleStepResult;
import com.biz.crm.dms.business.policy.sdk.context.GiftResultInfo;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutor;
import com.biz.crm.dms.business.policy.local.executestrategy.AbstractStandardExecuteStrategy;
import com.biz.crm.dms.business.policy.local.utils.GiftBuildUtils;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorLadderGiftVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorLadderVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorVo;
import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;

import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.AbstractSalePolicyExecutorInfo;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.biz.crm.dms.business.policy.standard.given.cyclestrategy.GivenAccumulationCycleRuleStrategy;
import com.biz.crm.dms.business.policy.standard.given.cyclestrategy.GivenMultipleCycleRuleStrategy;
import com.biz.crm.dms.business.policy.standard.given.cyclestrategy.GivenSingleCycleRuleStrategy;
import com.biz.crm.mdm.business.price.sdk.dto.FindPriceDto;
import com.biz.crm.mdm.business.price.sdk.enums.FindPriceUserTypeEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

public abstract class AbstractGivenExecuteStrategy extends AbstractStandardExecuteStrategy implements SalePolicyExecuteStrategy<SalePolicyExecutorVo> {
  
  @Autowired(required = false)
  private PriceModelVoService priceModelVoService;
  
  /**
   * 这个由抽象层提供的执行器保存的接口，主要就是用于保存标品中赠品性质的执行策略
   * @param update
   * @param requestGiftType 本次进行赠品执行策略维护时，要求额赠品计数类型。
   * @param currentSalePolicy
   * @param oldSalePolicy
   */
  protected void onSaveGivenExecutorInfo(boolean update , Integer requestGiftType, SalePolicyVo currentSalePolicy, SalePolicyVo oldSalePolicy) {
     /* 
      * 1、判定新的优惠政策中，每一个执行策略都传入了至少一个赠品信息（至于赠品信息本身的数值准确性，交由salePolicyExecutorLadderService内部去做）
      * 2、如果是修改操作，则需要删除之前的执行策略信息和关联信息（包括关联的赠品信息）
      * 3、然后全部按照“新增”操作去做。
      * */
    // 1、====== 
    Set<AbstractSalePolicyExecutorInfo> salePolicyExecutorInfos = currentSalePolicy.getSalePolicyExecutorInfos();
    for (AbstractSalePolicyExecutorInfo abstractSalePolicyExecutorInfo : salePolicyExecutorInfos) {
      SalePolicyExecutorVo salePolicyExecutor = (SalePolicyExecutorVo)abstractSalePolicyExecutorInfo;
      // 如果条件成立，说明将要运维的优惠策略执行器，并不是本执行策略负责的。所以忽略判定过程即可
      if(!StringUtils.equals(salePolicyExecutor.getExecuteStrategyCode(), this.getExecuteStrategyCode())) {
        continue;
      }
      Set<SalePolicyExecutorLadderVo> salePolicyExecutorLadders = salePolicyExecutor.getSalePolicyExecutorLadders();
      Validate.isTrue(!CollectionUtils.isEmpty(salePolicyExecutorLadders) , "在进行优惠政策维护时，每一个优惠政策执行策略都至少需要一个阶梯设定");
      for (SalePolicyExecutorLadderVo salePolicyExecutorLadder : salePolicyExecutorLadders) {
        Set<SalePolicyExecutorLadderGiftVo> executorLadderGifts = salePolicyExecutorLadder.getExecutorLadderGifts();
        Validate.isTrue(!CollectionUtils.isEmpty(executorLadderGifts) , "在进行优惠政策维护时，每一个优惠政策阶梯都需要至少设定一个赠品信息，请检查!!");
        for (SalePolicyExecutorLadderGiftVo executorLadderGift : executorLadderGifts) {
          Validate.isTrue(executorLadderGift.getGiftType() != null && executorLadderGift.getGiftType() == requestGiftType.intValue() , "按照优惠政策执行性质，这里的赠品赠送形式只能传递匹配的值（赠送数量）");
        }
      }
    }
    
    // 2和3都调用父级现成的方法即可
    this.handleSaveSalePolicyExecutorInfo(update, currentSalePolicy, oldSalePolicy, currentSalePolicy.getSalePolicyExecutorInfos().stream().map(item -> (SalePolicyExecutorVo)item).collect(Collectors.toSet()));
    super.validatetExpressionParams(currentSalePolicy);
  }
  
  /**
   * 该方法，依据当前阶梯步进的最新情况，和当前阶梯设定的赠品信息，进行新的竞品状态的计算（累加后的最新赠品值）
   * @param lastCycleStepResult
   * @param salePolicyExecutorLadderGifts
   * @return
   */
  protected Set<GiftResultInfo> buildGiftResult(CycleStepResult lastCycleStepResult , String customerCode , SalePolicyExecutorLadderVo salePolicyExecutorLadder) {
    Set<GiftResultInfo> lastGifts = lastCycleStepResult.getLastGifts();
    if(lastGifts == null) {
      lastGifts = Sets.newLinkedHashSet();
    }
    Set<SalePolicyExecutorLadderGiftVo> salePolicyExecutorLadderGifts = salePolicyExecutorLadder.getExecutorLadderGifts();
    /*
     * 处理过程为：
     * 1、首先根据salePolicyExecutorLadderGifts得到即将加入到上下文的新得到的赠品信息
     * 2、调用工具类GiftBuildUtils.buildGiftResult，形成新的集合
     */
    // 赠品信息要做批量查询
    Set<String> giftCodes = salePolicyExecutorLadderGifts.stream().map(SalePolicyExecutorLadderGiftVo::getGiftCode).collect(Collectors.toSet());
    // 批量查询效率高（赠品询价信息）
    FindPriceDto searchPriceDto = new FindPriceDto();
    //priceTypeCode 订死的？
    searchPriceDto.setUserCode(customerCode);
    searchPriceDto.setUserType(FindPriceUserTypeEnum.CUSTOMER.getDictCode());
    // 一批商品同时批量查
    searchPriceDto.setProductCodeSet(giftCodes);
    Map<String, PriceModelVo> giftPriceMapping = this.priceModelVoService.findPrice(searchPriceDto);
    
    // 1、======
    Map<String , GiftResultInfo> newGiftInfoMappings = Maps.newLinkedHashMap();
    Set<String> newGiftInfoCodes = Sets.newLinkedHashSet();    
    for (SalePolicyExecutorLadderGiftVo salePolicyExecutorLadderGift : salePolicyExecutorLadderGifts) {
      String giftCode = salePolicyExecutorLadderGift.getGiftCode();
      String giftName = salePolicyExecutorLadderGift.getGiftName();
      Integer giftValue = salePolicyExecutorLadderGift.getGiftValue();
      Integer giftType = salePolicyExecutorLadderGift.getGiftType();
      // 取得可能的询价单价
      PriceModelVo priceModelVo = null;
      BigDecimal giftPrice = BigDecimal.ZERO;
      if(!CollectionUtils.isEmpty(giftPriceMapping) && (priceModelVo = giftPriceMapping.get(giftCode)) != null
          && priceModelVo.getPrice() != BigDecimal.ZERO) {
        giftPrice = priceModelVo.getPrice();
      }
      
      // 根据赠品计数类型的不同，得到赠品数量的方式也不同
      // 如果条件成立，说明赠品数量的技术方式为数量（否则为金额）
      int number = 0;
      if(giftType == 1) {
        number = giftValue.intValue();
      } else {
        if(giftPrice != BigDecimal.ZERO) {
          number = new BigDecimal(giftValue.intValue()).divide(giftPrice , 0 , RoundingMode.DOWN).intValue();
        }
      }
      GiftResultInfo newGiftResultInfo = new GiftResultInfo();
      newGiftResultInfo.setProductCode(giftCode);
      newGiftResultInfo.setProductName(giftName);
      newGiftResultInfo.setQuantity(number);
      if(number != 0 && giftPrice != BigDecimal.ZERO) {
        BigDecimal subtotalAmount = giftPrice.multiply(new BigDecimal(number)).setScale(4, RoundingMode.HALF_UP);
        newGiftResultInfo.setSubtotalAmount(subtotalAmount);
      } else {
        newGiftResultInfo.setSubtotalAmount(BigDecimal.ZERO);
      }
      BigDecimal subtotalAmount = new BigDecimal(number).multiply(giftPrice);
      newGiftResultInfo.setSubtotalAmount(subtotalAmount);
      newGiftInfoMappings.put(giftCode , newGiftResultInfo);
      newGiftInfoCodes.add(giftCode);
    }
    
    // 2、=======
    Set<GiftResultInfo> resultGiftInfos = GiftBuildUtils.buildGiftResult(lastGifts, Sets.newHashSet(newGiftInfoMappings.values()));
    return resultGiftInfos;
  }
  
  @Override
  public Class<SalePolicyExecutorVo> getSalePolicyExecutorInfoClass() {
    return SalePolicyExecutorVo.class;
  }

  @Override
  public Collection<Class<? extends SalePolicyCycleRuleStrategy>> getCycleRuleStrategyClasses() {
    // 三种阶梯循环的规则都支持
    return Lists.newArrayList(GivenSingleCycleRuleStrategy.class, GivenMultipleCycleRuleStrategy.class , GivenAccumulationCycleRuleStrategy.class);
  }

  @Override
  public List<SalePolicyExecutorVo> onRequestSalePolicyExecutorInfo(String tenantCode, String salePolicyCode) {
    List<SalePolicyExecutor> salePolicyExecutors = super.salePolicyExecutorService.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode , getExecuteStrategyCode());
    if(CollectionUtils.isEmpty(salePolicyExecutors)) {
      return null;
    }
    // 转换成特定的VO（集合）
    Collection<SalePolicyExecutorVo> salePolicyExecutorVos = super.nebulaToolkitService.copyCollectionByWhiteList(salePolicyExecutors, SalePolicyExecutor.class, SalePolicyExecutorVo.class, LinkedHashSet.class, ArrayList.class, "salePolicyExecutorLadders" , "salePolicyExecutorLadders.executorLadderVars" , "salePolicyExecutorLadders.executorLadderGifts");
    return Lists.newArrayList(salePolicyExecutorVos);
  }
  
  @Override
  public AbstractCycleExecuteContext buildCycleExecuteContext(AbstractPolicyExecuteContext executeContext , String executorCode, Set<SalePolicyConProduct> initPolicyProducts, String customerCode) {
    return super.buildDefaultCycleExecuteContext(executorCode , executeContext, initPolicyProducts, customerCode);
  }
}
