package com.biz.crm.dms.business.policy.standard.special.executestrategy;

import com.biz.crm.dms.business.policy.local.context.UnitpriceCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.UnitpriceCycleStepResult;
import com.biz.crm.dms.business.policy.sdk.context.UnitpriceProduct;
import com.biz.crm.dms.business.policy.local.cyclestrategy.SingleCycleRuleStrategy;
import com.biz.crm.dms.business.policy.local.entity.SalePolicyExecutor;
import com.biz.crm.dms.business.policy.local.service.SalePolicyExecutorService;
import com.biz.crm.dms.business.policy.local.sharestrategy.UnitpriceShareStrategyWithAmountAndNumber;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorLadderVo;
import com.biz.crm.dms.business.policy.local.vo.SalePolicyExecutorVo;
import com.biz.crm.dms.business.policy.sdk.context.AbstractCycleExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.AbstractPolicyExecuteContext;
import com.biz.crm.dms.business.policy.sdk.context.SalePolicyConProduct;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyCycleRuleStrategy;
import com.biz.crm.dms.business.policy.sdk.strategy.SalePolicyExecuteShareStrategy;
import com.biz.crm.dms.business.policy.sdk.vo.SalePolicyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 在商品特价优惠方式中使用的，满金额享单价特价的具体优惠执行策略
 * @author songjingen yinwenjie
 */
@Component
public class ProductSpecialUnitpricePriceExecuteStrategy extends AbstractSpecialUnitpriceExecuteStrategy {

  @Autowired(required = false)
  private SalePolicyExecutorService salePolicyExecutorService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  
  @Override
  public String getExecuteStrategyCode() {
    return "productSpecialUnitpricePriceExecuteStrategy";
  }

  @Override
  public String getExecuteStrategyDesc() {
    return "满金额享受单价特价";
  }

  @Override
  public String getExpression() {
    return "本单中本品金额购买满{integer:input-integer:fullPriceValue}元，各本品单价{decimal:input-random:priceValue}元";
  }

  @Override
  public String getCycleRuleExample(String cycleRuleCode) {
    if(StringUtils.equals(cycleRuleCode, SingleCycleRuleStrategy.SINGLE)) {
      return "例如：满10000元，各本品单价按照80元计算；满5000元各本品单价按照100元计算。那么本品销售数量为1200单位，销售金额为126000元时，则总价为1200 * 80 = 96000 ";
    }
    return "";
  }
  
  @Override
  public Class<SalePolicyExecutorVo> getSalePolicyExecutorInfoClass() {
    return SalePolicyExecutorVo.class;
  }

  @Override
  public Class<? extends SalePolicyExecuteShareStrategy> getSalePolicyExecuteShareStrategy() {
    return UnitpriceShareStrategyWithAmountAndNumber.class;
  }

  @Override
  public Collection<Class<? extends SalePolicyCycleRuleStrategy>> getCycleRuleStrategyClasses() {
    // 商品-单价特价 只支持一种循环处理方式，既是单阶梯
    return Lists.newArrayList(SingleCycleRuleStrategy.class);
  }

  @Override
  public List<SalePolicyExecutorVo> onRequestSalePolicyExecutorInfo(String tenantCode , String salePolicyCode) {
    List<SalePolicyExecutor> salePolicyExecutors = this.salePolicyExecutorService.findBySalePolicyCodeAndTenantCode(salePolicyCode, tenantCode , getExecuteStrategyCode());
    if(CollectionUtils.isEmpty(salePolicyExecutors)) {
      return null;
    }
    // 转换成特定的VO（集合）
    Collection<SalePolicyExecutorVo> specialPolicyExecutorVos = this.nebulaToolkitService.copyCollectionByWhiteList(salePolicyExecutors, SalePolicyExecutor.class, SalePolicyExecutorVo.class, LinkedHashSet.class, ArrayList.class, "salePolicyExecutorLadders" , "salePolicyExecutorLadders.executorLadderVars");
    return Lists.newArrayList(specialPolicyExecutorVos);
  }

  @Override
  @Transactional
  public void onSaveSalePolicyExecutorInfo(boolean update , SalePolicyVo currentSalePolicy , SalePolicyVo oldSalePolicy) {
    this.handleSaveSalePolicyExecutorInfo(update, currentSalePolicy, oldSalePolicy, currentSalePolicy.getSalePolicyExecutorInfos().stream().map(item -> (SalePolicyExecutorVo)item).collect(Collectors.toSet()));
    super.validatetExpressionParams(currentSalePolicy);
  }
  
  @Override
  public AbstractCycleExecuteContext buildCycleExecuteContext(AbstractPolicyExecuteContext abstractExecuteContext, String executorCode , Set<SalePolicyConProduct> initPolicyProducts, String customerCode) {
    return super.buildSpecialUnitpriceCycleExecuteContext(executorCode , abstractExecuteContext, initPolicyProducts, customerCode);
  }

  @Override
  protected boolean execute(AbstractCycleExecuteContext abstractCycleExecuteContext, int index, int times, Map<String, Object> varParams , SalePolicyExecutorLadderVo salePolicyExecutorLadder) {
    // ======== 以下正式开始进行计算
    UnitpriceCycleExecuteContext cycleExecuteContext = (UnitpriceCycleExecuteContext)abstractCycleExecuteContext;
    UnitpriceCycleStepResult lastCycleStepResult = cycleExecuteContext.findLastStepResult();
    // 该变量是指，截止上次优惠后，还有多少本品小计金额没有参加优惠
    BigDecimal preSurplusSubtotalAmount = lastCycleStepResult.getLastSurplusSubtotalAmount();
    // 这里一定是有值的
    Object fullPriceValueObject = varParams.get("fullPriceValue");
    Object priceValueObject = varParams.get("priceValue");
    if(fullPriceValueObject == null || priceValueObject == null) {
      return false;
    }
    Integer fullPriceValue = (Integer)fullPriceValueObject;
    BigDecimal priceValue = (BigDecimal)priceValueObject;
    // 如果条件成立，说明不满足优惠条件或特价设定的金额有问题
    if(fullPriceValue > preSurplusSubtotalAmount.floatValue() || priceValue.compareTo(BigDecimal.ZERO) != 1) {
      return false;
    }
    
    // ======== 如果处理逻辑走到了这里，说明符合该阶梯的处理前提，开始正式进行优惠处理
    /*
     * 按单价优惠就找出参与这个优惠的所有本品，并且每个本品减少单价后，再进行计算   处理过程为：
     * 1、取得当前阶梯循环上下文中，上一个阶梯中的所有本品信息，
     * 并进行本品的单价扣减（并记录汇总的优惠金额）
     * 注意，首先要检查每一个物资的单价大于priceValue，这样才能进行特价优惠
     * 2、接着记录当前阶梯计算后的优惠状态
     * */ 
    // 1、=======
    BigDecimal preSubtotalAmount = lastCycleStepResult.getLastSubtotalAmount();
    Integer preSurplusSubtotalNumber = lastCycleStepResult.getLastSurplusSubtotalNumber();
    Set<UnitpriceProduct> preProducts = lastCycleStepResult.getLastProducts();
    UnitpriceCycleStepResult newUnitpriceCycleStepResult = new UnitpriceCycleStepResult(index, times, preSubtotalAmount, preSurplusSubtotalAmount, preSurplusSubtotalNumber , preProducts);
    // 这个变量记录了各个本品在减去单价后，汇总的优惠金额
    BigDecimal newCycleAmountResult = BigDecimal.ZERO;
    Set<UnitpriceProduct> lastProducts = Sets.newLinkedHashSet();
    for (UnitpriceProduct preSpecialUnitpriceProduct : preProducts) {
      String productCode = preSpecialUnitpriceProduct.getProductCode();
      BigDecimal prePrices = preSpecialUnitpriceProduct.getPrices();
      Integer productNumber = preSpecialUnitpriceProduct.getNumbers();
      // 重新计算单价
      BigDecimal lastPrices = null;
      // 如果单价已经被特价小了，就以当前单价为准
      if(prePrices.compareTo(priceValue) != 1) {
        lastPrices = prePrices;
      } else {
        lastPrices = priceValue;
      }
      // 重新计算小计价，并进行小计价格的累加。如果条件成立，说明计算后本品单价小于零，那么就取单价为0
      if(lastPrices.compareTo(BigDecimal.ZERO) == -1) {
        lastPrices = BigDecimal.ZERO;
        newCycleAmountResult = newCycleAmountResult.add(BigDecimal.ZERO); 
      } else {
        // 优惠的金额 = (原单价- 特价单价) * 本品数量
        newCycleAmountResult = newCycleAmountResult.add(prePrices.subtract(lastPrices).multiply(new BigDecimal(productNumber)).setScale(4, RoundingMode.HALF_UP));
      }
      // 记录这个本品优惠后的状态
      UnitpriceProduct lastSpecialUnitpriceProduct = new UnitpriceProduct(productCode, productNumber, lastPrices);
      lastProducts.add(lastSpecialUnitpriceProduct);
    }
    
    // 2、=======
    // 计算优惠后的金额： 原总计金额 - 小计折扣金额
    BigDecimal result = preSubtotalAmount.subtract(newCycleAmountResult).setScale(4, RoundingMode.HALF_UP);
    newUnitpriceCycleStepResult.setLastSubtotalAmount(result);
    // 本次计算结果作为下次计算的起始依据
    newUnitpriceCycleStepResult.setLastSurplusSubtotalAmount(result);
    newUnitpriceCycleStepResult.setLastSurplusSubtotalNumber(preSurplusSubtotalNumber);
    newUnitpriceCycleStepResult.setLastProducts(lastProducts);
    cycleExecuteContext.addStepResult(newUnitpriceCycleStepResult);
    return true;
  }
}
