package com.biz.crm.dms.business.psi.product.local.repository.productstock;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStockDetail;
import com.biz.crm.dms.business.psi.product.local.mapper.productstock.ProductStockDetailMapper;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockDetailPaginationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperation;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.DeliveryStockVo;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductStockDetailVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * <p>
 * 商品库存明细表 服务实现类
 * </p>
 *
 * @author Bao Hongbin
 * @since 2022-01-11
 */
@Component
public class ProductStockDetailRepository extends ServiceImpl<ProductStockDetailMapper, ProductStockDetail> {

  /**
   * 通过订单编号查询指定库存操作的库存明细（如不指定库存操作，则查询全部）
   *
   * @param orderCode  库存明细关联的订单编号
   * @param operation  库存操作
   * @param tenantCode 租户编号
   * @return 商品库存明细集合
   */
  public List<ProductStockDetail> findListByOrderCode(String orderCode, ProductStockOperation operation, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductStockDetail::getTenantCode, tenantCode)
        .eq(ProductStockDetail::getOrderCode, orderCode)
        .eq(operation != null, ProductStockDetail::getProductStockOperation, operation)
        .list();
  }

  /**
   * 占用库存明细多条件分页查询(可适用于列表分页查询和公用分页弹框)
   * @param pageable
   * @param paginationDto
   * @return
   */
  public Page<ProductStockDetailVo> findFreezeStockDetailByConditions(Pageable pageable, ProductStockDetailPaginationDto paginationDto) {
    Page<ProductStockDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findFreezeStockDetailByConditions(page, paginationDto, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 库存明细多条件分页查询(可适用于列表分页查询和公用分页弹框)
   * @param pageable
   * @param paginationDto
   * @return
   */
  public Page<ProductStockDetailVo> findStockOperationDetailByConditions(Pageable pageable, ProductStockDetailPaginationDto paginationDto) {
    Page<ProductStockDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findStockOperationDetailByConditions(page, paginationDto, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 根据订单编码和行编码查询冻结信息
   * @param orderCode
   * @param orderItemCode
   * @return
   */
  public List<ProductStockDetail> findFreezeStockDetailByOrderCodeAndOrderItemCode(String orderCode, String orderItemCode) {
    return this.lambdaQuery()
            .eq(ProductStockDetail::getTenantCode, TenantUtils.getTenantCode())
            .eq(ProductStockDetail::getOrderCode, orderCode)
            .eq(ProductStockDetail::getOrderItemCode, orderItemCode)
            .eq(ProductStockDetail::getProductStockOperation, ProductStockOperation.FROZEN)
            .list();
  }

  /**
   * 根据订单编码和行查询库存信息
   * @param list
   * @return
   */
  public List<DeliveryStockVo> findByOrderCodeAndItemCode(List<String> list) {
    return this.baseMapper.findByOrderCodeAndItemCode(list);
  }

  /**
   * 根据仓库编码、商品编码和操作类型查询库存信息
   * @param storeWarehouseCode
   * @param productCode
   * @param operations
   * @return
   */
  public List<ProductStockDetail> findByWareAndProductAndOperationIn(String storeWarehouseCode, String productCode, List<ProductStockOperation> operations) {
    return this.lambdaQuery()
            .eq(ProductStockDetail::getTenantCode, TenantUtils.getTenantCode())
            .eq(ProductStockDetail::getWarehouseCode, storeWarehouseCode)
            .eq(ProductStockDetail::getProductCode, productCode)
            .in(ProductStockDetail::getProductStockOperation, operations)
            .list();
  }

  public void updateWarehouseMsg(String warehouseCode, String warehouseName) {
    this.lambdaUpdate()
        .eq(ProductStockDetail::getWarehouseCode,warehouseCode)
        .set(ProductStockDetail::getWarehouseName,warehouseName)
        .update();
  }
}
