package com.biz.crm.dms.business.psi.product.local.repository.productstock;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStock;
import com.biz.crm.dms.business.psi.product.local.mapper.productstock.ProductStockMapper;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockPaginationDto;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductStockVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Set;

/**
 * <p>
 * 商品库存表 服务实现类
 * </p>
 *
 * @author Bao Hongbin
 * @since 2022-01-11
 */
@Component
public class ProductStockRepository extends ServiceImpl<ProductStockMapper, ProductStock> {

  /**
   * 根据仓库编码和商品编码查询商品库存数据
   *
   * @param warehouseCode
   * @param productCode
   * @param tenantCode
   * @return
   */
  public ProductStock findByWarehouseCodeAndProductCode(String warehouseCode, String productCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductStock::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductStock::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(ProductStock::getTenantCode, tenantCode)
        .eq(ProductStock::getWarehouseCode, warehouseCode)
        .eq(ProductStock::getProductCode, productCode)
        .one();

  }

  /**
   * 分页查询
   *
   * @param pageable
   * @param paginationDto
   * @return
   */
  public Page<ProductStockVo> findByConditions(Pageable pageable, ProductStockPaginationDto paginationDto) {
    Page<ProductStockVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, paginationDto, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 通过商品编码集合和库存类型查询
   * @param productCodes
   * @return
   */
  public List<ProductStock> findByTypeAndProductCodeIn(List<String> productCodes, String type) {
    return this.lambdaQuery()
            .eq(ProductStock::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(ProductStock::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .eq(ProductStock::getTenantCode, TenantUtils.getTenantCode())
            .eq(ProductStock::getType, type)
            .in(ProductStock::getProductCode, productCodes)
            .list();
  }

  /**
   * 根据仓库编码和商品编码集合查询库存
   * @param warehouseCode
   * @param productCodes
   * @return
   */
  public List<ProductStock> findByWarehouseAndProducts(String warehouseCode, Set<String> productCodes) {
    return this.lambdaQuery()
            .eq(ProductStock::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(ProductStock::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .eq(ProductStock::getTenantCode, TenantUtils.getTenantCode())
            .eq(ProductStock::getWarehouseCode, warehouseCode)
            .in(ProductStock::getProductCode, productCodes)
            .list();
  }

  /**
   * 根据商品编码查询
   * @param productCodes
   * @return
   */
  public List<ProductStockVo> findByProductCodes(List<String> productCodes) {
    return this.baseMapper.findByProductCodes(productCodes, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 根据仓库编码查询
   * @param warehouseCode
   * @return
   */
  public List<ProductStock> findByWarehouseCodes(List<String> warehouseCode) {
    return this.lambdaQuery()
            .eq(ProductStock::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(ProductStock::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .eq(ProductStock::getTenantCode, TenantUtils.getTenantCode())
            .in(ProductStock::getWarehouseCode, warehouseCode)
            .list();
  }

  /**
   * 通过物料编码集合查询
   * @param materialCode
   * @return
   */
  public List<ProductStock> findByMaterialCode(List<String> materialCode) {
    return this.lambdaQuery()
               .eq(ProductStock::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
               .eq(ProductStock::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
               .eq(ProductStock::getTenantCode, TenantUtils.getTenantCode())
               .in(ProductStock::getProductCode, materialCode)
               .list();
  }

  /**
   * 修改 enable_status
   *
   * @param enable
   * @param materialCode
   */
  public void updateEnableStatusByMaterialCode(EnableStatusEnum enable, List<String> materialCode) {
    if (CollectionUtils.isEmpty(materialCode)) {
      return;
    }
    this.lambdaUpdate()
        .set(ProductStock::getEnableStatus, enable.getCode())
        .eq(ProductStock::getTenantCode, TenantUtils.getTenantCode())
        .in(ProductStock::getProductCode, materialCode)
        .update();
  }

  public List<ProductStock> findByWarehouseCode(String warehouseCode) {
    return this.lambdaQuery()
        .eq(ProductStock::getWarehouseCode,warehouseCode)
        .list();
  }

  public void updateWarehouseMsg(String warehouseCode, String warehouseName) {
    this.lambdaUpdate()
        .eq(ProductStock::getWarehouseCode,warehouseCode)
        .set(ProductStock::getWarehouseName,warehouseName)
        .update();
  }
}
