package com.biz.crm.dms.business.psi.product.local.service.delivery.internal;
/**
 * Created by Bao Hongbin on 2022-01-15 15:04.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.psi.product.local.entity.delivery.ProductDeliveryBill;
import com.biz.crm.dms.business.psi.product.local.entity.delivery.ProductDeliveryBillDetail;
import com.biz.crm.dms.business.psi.product.local.repository.delivery.ProductDeliveryBillDetailRepository;
import com.biz.crm.dms.business.psi.product.local.repository.delivery.ProductDeliveryBillRepository;
import com.biz.crm.dms.business.psi.product.sdk.common.constant.ProductDeliveryBillConstant;
import com.biz.crm.dms.business.psi.product.sdk.context.delivery.DeliveryBillProductStockContext;
import com.biz.crm.dms.business.psi.product.sdk.dto.delivery.ProductDeliveryBillCreateDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.delivery.ProductDeliveryBillDetailDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.delivery.ProductDeliveryBillPaginationDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.delivery.ProductDeliveryBillUpdateDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.DeliveryProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.delivery.ProductDeliveryStatus;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockBillType;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperationType;
import com.biz.crm.dms.business.psi.product.sdk.event.ProductDeliveryListener;
import com.biz.crm.dms.business.psi.product.sdk.register.delivery.DeliveryBillProductStockRegister;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductDeliveryBillVoService;
import com.biz.crm.dms.business.psi.product.sdk.vo.delivery.ProductDeliveryBillDetailVo;
import com.biz.crm.dms.business.psi.product.sdk.vo.delivery.ProductDeliveryBillVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @program: crm-dms
 * @description: 商品出库单服务实现
 * @author: Bao Hongbin
 * @create: 2022-01-15 15:04
 **/
@Service
public class ProductDeliveryBillVoServiceImpl implements ProductDeliveryBillVoService {
  @Autowired(required = false)
  private ProductDeliveryBillRepository productDeliveryBillRepository;
  @Autowired(required = false)
  private ProductDeliveryBillDetailRepository productDeliveryBillDetailRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCode;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private DeliveryBillProductStockContext deliveryBillProductStockContext;
  @Autowired(required = false)
  private List<ProductDeliveryListener> productDeliveryListeners;

  @Override
  public Page<ProductDeliveryBillVo> findByConditions(Pageable pageable, ProductDeliveryBillPaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new ProductDeliveryBillPaginationDto());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    return productDeliveryBillRepository.findByConditions(pageable, paginationDto);
  }

  @Override
  public ProductDeliveryBillVo findDetailsById(String id) {
    if (!StringUtils.isNotEmpty(id)) {
      return null;
    }
    //查询主表数据
    ProductDeliveryBill bill = productDeliveryBillRepository.findDetailsById(id, TenantUtils.getTenantCode());
    if (Objects.isNull(bill)) {
      return null;
    }
    ProductDeliveryBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductDeliveryBillVo.class, HashSet.class, ArrayList.class);
    //查询详情
    List<ProductDeliveryBillDetailVo> detailVos =
        (List<ProductDeliveryBillDetailVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                productDeliveryBillDetailRepository.findListByBillId(bill.getId()),
                ProductDeliveryBillDetail.class,
                ProductDeliveryBillDetailVo.class,
                HashSet.class,
                ArrayList.class);
    billVo.setDetailVos(detailVos);
    return billVo;
  }

  @Override
  public ProductDeliveryBillVo findDetailsByCode(String code) {
    if (!StringUtils.isNotEmpty(code)) {
      return null;
    }
    //查询主表数据
    ProductDeliveryBill bill = productDeliveryBillRepository.findDetailsByCode(code, TenantUtils.getTenantCode());
    if (Objects.isNull(bill)) {
      return null;
    }
    ProductDeliveryBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductDeliveryBillVo.class, HashSet.class, ArrayList.class);
    //查询详情
    List<ProductDeliveryBillDetailVo> detailVos =
        (List<ProductDeliveryBillDetailVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                productDeliveryBillDetailRepository.findListByBillId(bill.getId()),
                ProductDeliveryBillDetail.class,
                ProductDeliveryBillDetailVo.class,
                HashSet.class,
                ArrayList.class);
    billVo.setDetailVos(detailVos);
    return billVo;
  }

  @Override
  @Transactional
  public ProductDeliveryBillVo create(ProductDeliveryBillCreateDto createDto) {
    return this.createForm(createDto);
  }

  private ProductDeliveryBillVo createForm(ProductDeliveryBillCreateDto createDto) {
    //验证
    validateCreate(createDto);
    //设置数据
    ProductDeliveryBill bill = this.nebulaToolkitService.copyObjectByWhiteList(
        createDto, ProductDeliveryBill.class, HashSet.class, ArrayList.class);
    bill.setTenantCode(TenantUtils.getTenantCode());
    bill.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    bill.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //TODO 各业务编码前缀以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
    bill.setDeliveryBillCode(generateCode.generateCode(ProductDeliveryBillConstant.CODE, 1).get(0));
    bill.setDeliveryStatus(ProductDeliveryStatus.WAIT_DELIVERY);
    //保存数据
    productDeliveryBillRepository.save(bill);
    ProductDeliveryBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductDeliveryBillVo.class, HashSet.class, ArrayList.class);
    //保存详情
    List<ProductDeliveryBillDetailVo> detailVos = saveDetails(bill, createDto.getDetailDtos());
    billVo.setDetailVos(detailVos);
    //处理库存
    this.handleDeliveryBatch(Lists.newArrayList(bill.getId()));
    //事件通知
    if (CollectionUtils.isNotEmpty(productDeliveryListeners)) {
      productDeliveryListeners.forEach(listener -> {
        listener.onCreate(billVo);
      });
    }
    return billVo;
  }

  private void validateCreate(ProductDeliveryBillCreateDto createDto) {
    Validate.notNull(createDto.getDeliveryTime(), "出库时间不能为空");
    Validate.notBlank(createDto.getProductStockOperationType(), "商品库存操作类型不能为空");
    if (ProductStockOperationType.SALE_DELIVER.equals(createDto.getProductStockOperationType())) {
      Validate.notBlank(createDto.getRelationShipmentOrderCode(), "关联发货单编码不能为空");
      Validate.notBlank(createDto.getRelationShipmentOrderType(), "关联发货单类型不能为空");
    }
    Validate.notBlank(createDto.getDeliveryWarehouseCode(), "出库仓库编码不能为空");
    Validate.notBlank(createDto.getDeliveryWarehouseName(), "出库仓库名称不能为空");
    Validate.notEmpty(createDto.getDetailDtos(), "商品出库单商品行不能为空");
  }

  @Override
  @Transactional
  public ProductDeliveryBillVo update(ProductDeliveryBillUpdateDto updateDto) {
    return this.updateForm(updateDto);
  }

  private ProductDeliveryBillVo updateForm(ProductDeliveryBillUpdateDto updateDto) {
    //验证
    validateUpdate(updateDto);
    //设置数据
    ProductDeliveryBill bill = this.nebulaToolkitService.copyObjectByWhiteList(
        updateDto, ProductDeliveryBill.class, HashSet.class, ArrayList.class);
    bill.setDeliveryStatus(ProductDeliveryStatus.WAIT_DELIVERY);
    //更新数据
    productDeliveryBillRepository.updateById(bill);
    ProductDeliveryBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductDeliveryBillVo.class, HashSet.class, ArrayList.class);
    //删除原明细数据
    productDeliveryBillDetailRepository.deleteByBillId(bill.getId());
    //保存详情
    List<ProductDeliveryBillDetailVo> detailVos = saveDetails(bill, updateDto.getDetailDtos());
    billVo.setDetailVos(detailVos);
    return billVo;
  }

  /**
   * 保存详情
   *
   * @param bill
   * @param detailDtos
   * @return
   */
  private List<ProductDeliveryBillDetailVo> saveDetails(ProductDeliveryBill bill, List<ProductDeliveryBillDetailDto> detailDtos) {
    String productStockOperationType = bill.getProductStockOperationType();
    if (ProductStockOperationType.OTHER_DELIVER.toString().equals(productStockOperationType)
        || ProductStockOperationType.MATERIAL_OTHER_DELIVER.toString().equals(productStockOperationType)) {
      //其他出库
      for (ProductDeliveryBillDetailDto dto : detailDtos) {
        Validate.notNull(dto.getCurrentQuantity(), "本次出库数量不能为空");
        Validate.notNull(dto.getAvailableStock(),"可用库存数量不能为空");
        Validate.isTrue(dto.getCurrentQuantity().compareTo(dto.getAvailableStock()) <= 0, "商品" + dto.getProductCode() + "本次出库数量不能大于可用库存数量");
      }
    }else{
      //订单出库
      for (ProductDeliveryBillDetailDto dto : detailDtos) {
        Validate.notNull(dto.getDeliveryingQuantity(), "待出库数量不能为空");
        Validate.notNull(dto.getCurrentQuantity(), "本次出库数量不能为空");
        Validate.isTrue(dto.getCurrentQuantity().compareTo(dto.getDeliveryingQuantity()) <= 0, "商品" + dto.getProductCode() + "本次出库数量不能大于待出库数量");
      }
    }
    List<ProductDeliveryBillDetail> billDetails =
        (List<ProductDeliveryBillDetail>) nebulaToolkitService.copyCollectionByWhiteList(detailDtos,
            ProductDeliveryBillDetailDto.class,
            ProductDeliveryBillDetail.class,
            HashSet.class,
            ArrayList.class);
    for (ProductDeliveryBillDetail billDetail : billDetails) {
      billDetail.setTenantCode(TenantUtils.getTenantCode());
      billDetail.setDeliveryBillId(bill.getId());
      billDetail.setDeliveryBillCode(bill.getDeliveryBillCode());
      if (Objects.isNull(billDetail.getDeliveryOutQuantity())) {
        billDetail.setDeliveryOutQuantity(BigDecimal.ZERO);
      }
    }
    productDeliveryBillDetailRepository.saveBatch(billDetails);
    return (List<ProductDeliveryBillDetailVo>) nebulaToolkitService.copyCollectionByWhiteList(billDetails,
        ProductDeliveryBillDetail.class,
        ProductDeliveryBillDetailVo.class,
        HashSet.class,
        ArrayList.class);
  }

  private void validateUpdate(ProductDeliveryBillUpdateDto updateDto) {
    Validate.notBlank(updateDto.getId(), "id不能为空");
    Validate.notBlank(updateDto.getDeliveryBillCode(), "出库单编码不能为空");
    Validate.notNull(updateDto.getDeliveryTime(), "出库时间不能为空");
    Validate.notBlank(updateDto.getProductStockOperationType(), "商品库存操作类型不能为空");
    if (ProductStockOperationType.SALE_DELIVER.equals(updateDto.getProductStockOperationType())) {
      Validate.notBlank(updateDto.getRelationShipmentOrderCode(), "关联发货单编码不能为空");
      Validate.notBlank(updateDto.getRelationShipmentOrderType(), "关联发货单类型不能为空");
    }
    Validate.notBlank(updateDto.getDeliveryWarehouseCode(), "出库仓库编码不能为空");
    Validate.notBlank(updateDto.getDeliveryWarehouseName(), "出库仓库名称不能为空");
    Validate.notEmpty(updateDto.getDetailDtos(), "商品出库单商品行不能为空");
    ProductDeliveryBill bill = productDeliveryBillRepository.findDetailsById(updateDto.getId(), TenantUtils.getTenantCode());
    Validate.isTrue(Objects.nonNull(bill), "找不到对应的出库单");
    ProductDeliveryStatus status = bill.getDeliveryStatus();
    Validate.isTrue(ProductDeliveryStatus.WAIT_DELIVERY.equals(status)
        || ProductDeliveryStatus.REJECTED.equals(status), "只有待出库和已驳回的出库单才允许编辑");

  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductDeliveryBill> bills =
        productDeliveryBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    //判断状态
    for (ProductDeliveryBill bill : bills) {
      ProductDeliveryStatus status = bill.getDeliveryStatus();
      Validate.isTrue(ProductDeliveryStatus.WAIT_DELIVERY.equals(status)
              || ProductDeliveryStatus.REJECTED.equals(status) || ProductDeliveryStatus.VOIDED.equals(status),
          "只有待出库、已作废和已驳回的出库单才允许删除");
    }
    //删除
    productDeliveryBillRepository.deleteBatch(ids);
  }

  @Override
  @Transactional
  public void handleDeliveryBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductDeliveryBill> bills =
        productDeliveryBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    for (ProductDeliveryBill bill : bills) {
      //判断状态
      ProductDeliveryStatus status = bill.getDeliveryStatus();
      Validate.isTrue(ProductDeliveryStatus.WAIT_DELIVERY.equals(status), "只有待出库的出库单才允许进行出库操作");
      bill.setDeliveryStatus(ProductDeliveryStatus.DELIVERED);
      //操作库存
      DeliveryBillProductStockRegister register =
          deliveryBillProductStockContext.getRegisterByOperationType(bill.getProductStockOperationType());
      Validate.notNull(register, "无法找到对应类型的出库单操作库存的注册器");
      List<ProductDeliveryBillDetail> details =
          productDeliveryBillDetailRepository.findListByBillId(bill.getId());
      Validate.notEmpty(details, "无法找到出库单详情");
      for (ProductDeliveryBillDetail detail : details) {
        DeliveryProductStockOperationDto operationDto = buildOperationDto(bill, detail);
        //执行操作
        register.execute(operationDto);
      }
    }
    productDeliveryBillRepository.updateBatchById(bills);
  }

  @Override
  @Transactional
  public void handleRejectedBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductDeliveryBill> bills =
        productDeliveryBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    //判断状态
    for (ProductDeliveryBill bill : bills) {
      ProductDeliveryStatus status = bill.getDeliveryStatus();
      Validate.isTrue(ProductDeliveryStatus.WAIT_DELIVERY.equals(status), "只有待出库的出库单才允许驳回");
      bill.setDeliveryStatus(ProductDeliveryStatus.REJECTED);
    }
    //驳回
    productDeliveryBillRepository.updateBatchById(bills);

  }

  @Override
  @Transactional
  public void handleVoidBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductDeliveryBill> bills =
        productDeliveryBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    for (ProductDeliveryBill bill : bills) {
      //判断状态
      ProductDeliveryStatus status = bill.getDeliveryStatus();
      Validate.isTrue(ProductDeliveryStatus.WAIT_DELIVERY.equals(status)
          || ProductDeliveryStatus.DELIVERED.equals(status), "只有待出库和已出库的出库单才允许进行作废操作");
      bill.setDeliveryStatus(ProductDeliveryStatus.VOIDED);
      if (ProductDeliveryStatus.DELIVERED.equals(status)) {
        //如果为已出库时作废需要回退库存数量
        //回退操作库存
        DeliveryBillProductStockRegister register =
            deliveryBillProductStockContext.getRegisterByOperationType(bill.getProductStockOperationType());
        Validate.notNull(register, "无法找到对应类型的出库单操作库存的注册器");
        List<ProductDeliveryBillDetail> details =
            productDeliveryBillDetailRepository.findListByBillId(bill.getId());
        Validate.notEmpty(details, "无法找到出库单详情");
        for (ProductDeliveryBillDetail detail : details) {
          DeliveryProductStockOperationDto operationDto = buildOperationDto(bill, detail);
          //回退操作
          register.rescind(operationDto);
        }
      }
    }
    productDeliveryBillRepository.updateBatchById(bills);
  }

  /**
   * 构建库存操作请求
   *
   * @param bill
   * @param detail
   * @return
   */
  private DeliveryProductStockOperationDto buildOperationDto(ProductDeliveryBill bill, ProductDeliveryBillDetail detail) {
    DeliveryProductStockOperationDto operationDto = nebulaToolkitService.copyObjectByWhiteList(
        detail, DeliveryProductStockOperationDto.class, HashSet.class, ArrayList.class);
    operationDto.setWarehouseCode(bill.getDeliveryWarehouseCode());
    operationDto.setWarehouseName(bill.getDeliveryWarehouseName());
    operationDto.setBillType(ProductStockBillType.DELIVERY_ORDER.name());
    operationDto.setBillCode(bill.getDeliveryBillCode());
    operationDto.setQuantity(detail.getCurrentQuantity());
    return operationDto;
  }
}
