package com.biz.crm.dms.business.psi.product.local.service.productstock.internal;
/**
 * Created by Bao Hongbin on 2022-01-11 14:51.
 */

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStock;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStockDetail;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductStockDetailRepository;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductStockRepository;
import com.biz.crm.dms.business.psi.product.local.service.productstock.ProductStockService;
import com.biz.crm.dms.business.psi.product.sdk.common.constant.ProductStockConstant;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperation;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.StockType;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.warehouse.sdk.service.WarehouseVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * @program: crm-dms
 * @description: 商品库存服务实现
 * @author: Bao Hongbin
 * @create: 2022-01-11 14:51
 **/
@Service
public class ProductStockServiceImpl implements ProductStockService {

  @Autowired(required = false)
  private ProductStockRepository productStockRepository;
  @Autowired(required = false)
  private ProductStockDetailRepository productStockDetailRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCode;
  @Autowired(required = false)
  private ProductVoService productVoService;
  @Autowired(required = false)
  private WarehouseVoService warehouseVoService;
  @Autowired(required = false)
  private MaterialVoService materialVoService;

  /**
   * 初始化库存
   *
   * @param warehouseCode 仓库编码
   * @param productCode   商品编码
   * @return 商品库存
   */
  private ProductStock create(String warehouseCode, String productCode, String type) {
    ProductVo productVo = new ProductVo();
    MaterialVo materialVo = new MaterialVo();
    if(StockType.PRODUCT.getKey().equals(type)) {
      List<ProductVo> productVos =
              productVoService.findMainDetailsByProductCodes(Lists.newArrayList(productCode));
      productVo = productVos.stream().findFirst().orElseThrow(() ->
              new IllegalArgumentException("无法找到对应的商品信息"));
    }
    if(StockType.MATERIAL.getKey().equals(type)) {
      materialVo = materialVoService.findDetailByMaterialCode(productCode);
      Validate.notNull(materialVo, "无法找到对应的物料信息");
    }
    WarehouseVo warehouseVo = warehouseVoService.findDetailsByCode(warehouseCode);
    Validate.notNull(warehouseVo, "无法找到对应的仓库信息");
    ProductStock productStock;
    productStock = new ProductStock();
    productStock.setTenantCode(TenantUtils.getTenantCode());
    productStock.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productStock.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //TODO 各业务编码前缀以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
    productStock.setStockCode(generateCode.generateCode(ProductStockConstant.CODE, 1).get(0));
    if(StockType.PRODUCT.getKey().equals(type)) {
      productStock.setProductCode(productVo.getProductCode());
      productStock.setProductName(productVo.getProductName());
      productStock.setProductLevelCode(productVo.getProductLevelCode());
      productStock.setProductLevelName(productVo.getProductLevelName());
    }
    if(StockType.MATERIAL.getKey().equals(type)) {
      productStock.setProductCode(materialVo.getMaterialCode());
      productStock.setProductName(materialVo.getMaterialName());
      productStock.setProductLevelCode(materialVo.getProductLevelCode());
      productStock.setProductLevelName(materialVo.getProductLevelName());
    }
    productStock.setWarehouseCode(warehouseVo.getWarehouseCode());
    productStock.setWarehouseName(warehouseVo.getWarehouseName());
    productStock.setAvailableStock(BigDecimal.ZERO);
    productStock.setFrozenStock(BigDecimal.ZERO);
    productStock.setTotalStock(BigDecimal.ZERO);
    productStock.setType(type);
    productStockRepository.save(productStock);
    return productStock;
  }

  @Override
  @Transactional
  public ProductStock store(ProductStockOperationDto productStockOperationDto) {
    Validate.notNull(productStockOperationDto.getWarehouseCode(), "变更库存商品数量时仓库编码不能为空");
    Validate.notNull(productStockOperationDto.getProductCode(), "变更库存商品数量时商品编码不能为空");
    Validate.notNull(productStockOperationDto.getQuantity(), "变更库存商品数量不能为空");
    //查询商品库存数据
    ProductStock productStock =
        productStockRepository.findByWarehouseCodeAndProductCode(productStockOperationDto.getWarehouseCode(),
            productStockOperationDto.getProductCode(), TenantUtils.getTenantCode());
    if (Objects.isNull(productStock)) {
      //如果不存在则创建
      productStock = create(productStockOperationDto.getWarehouseCode(), productStockOperationDto.getProductCode(), productStockOperationDto.getType());
    }
    productStock.setTotalStock(productStock.getTotalStock().add(productStockOperationDto.getQuantity().abs()));
    productStock.setAvailableStock(productStock.getAvailableStock().add(productStockOperationDto.getQuantity().abs()));
    productStockRepository.updateById(productStock);
    //记录库存明细
    createProductStockDetail(productStockOperationDto, productStock, ProductStockOperation.STORE,
        productStockOperationDto.getQuantity().abs(), BigDecimal.ZERO);
    return productStock;
  }

  /**
   * 记录库存明细
   *
   * @param productStockOperationDto
   * @param productStock
   * @param productStockOperation
   * @param stockQuantity
   * @param frozenQuantity
   */
  private void createProductStockDetail(ProductStockOperationDto productStockOperationDto, ProductStock productStock,
                                        ProductStockOperation productStockOperation,
                                        BigDecimal stockQuantity, BigDecimal frozenQuantity) {
    ProductStockDetail productStockDetail = new ProductStockDetail();
    productStockDetail.setTenantCode(TenantUtils.getTenantCode());
    productStockDetail.setStockId(productStock.getId());
    productStockDetail.setStockCode(productStock.getStockCode());
    productStockDetail.setWarehouseCode(productStock.getWarehouseCode());
    productStockDetail.setWarehouseName(productStock.getWarehouseName());
    productStockDetail.setProductCode(productStock.getProductCode());
    productStockDetail.setProductName(productStock.getProductName());
    productStockDetail.setProductStockOperation(productStockOperation);
    productStockDetail.setProductStockOperationType(productStockOperationDto.getProductStockOperationType());
    productStockDetail.setCustomerCode(productStockOperationDto.getCustomerCode());
    productStockDetail.setCustomerName(productStockOperationDto.getCustomerName());
    productStockDetail.setOriginalOrderCode(productStockOperationDto.getOriginalOrderCode());
    productStockDetail.setOrderType(productStockOperationDto.getOrderType());
    productStockDetail.setOrderCode(productStockOperationDto.getOrderCode());
    productStockDetail.setOrderItemCode(productStockOperationDto.getOrderItemCode());
    productStockDetail.setBillType(productStockOperationDto.getBillType());
    productStockDetail.setBillCode(productStockOperationDto.getBillCode());
    productStockDetail.setStockQuantity(stockQuantity);
    productStockDetail.setTotalStockQuantity(productStock.getTotalStock());
    productStockDetail.setFrozenQuantity(frozenQuantity);
    productStockDetail.setTotalFrozenQuantity(productStock.getFrozenStock());
    productStockDetailRepository.save(productStockDetail);
  }

  @Override
  @Transactional
  public ProductStock deliver(ProductStockOperationDto productStockOperationDto) {
    Validate.notNull(productStockOperationDto.getWarehouseCode(), "变更库存商品数量时仓库编码不能为空");
    Validate.notNull(productStockOperationDto.getProductCode(), "变更库存商品数量时商品编码不能为空");
    Validate.notNull(productStockOperationDto.getQuantity(), "变更库存商品数量不能为空");
    //查询商品库存数据
    ProductStock productStock =
        productStockRepository.findByWarehouseCodeAndProductCode(productStockOperationDto.getWarehouseCode(),
            productStockOperationDto.getProductCode(), TenantUtils.getTenantCode());
    Validate.notNull(productStock, "无法找到对应的商品库存数据");
    BigDecimal totalStock = productStock.getTotalStock().subtract(productStockOperationDto.getQuantity().abs());
    Validate.isTrue(totalStock.compareTo(BigDecimal.ZERO) >= 0, "库存数量不足");
    productStock.setTotalStock(totalStock);
    BigDecimal availableStock = productStock.getAvailableStock().subtract(productStockOperationDto.getQuantity().abs());
    //出库后商品可用数量不能小于0
    Validate.isTrue(availableStock.compareTo(BigDecimal.ZERO) >= 0, "可用库存数量不足");
    productStock.setAvailableStock(availableStock);
    productStockRepository.updateById(productStock);
    //记录库存明细
    createProductStockDetail(productStockOperationDto, productStock, ProductStockOperation.DELIVER,
        productStockOperationDto.getQuantity().abs().negate(), BigDecimal.ZERO);
    return productStock;
  }

  @Override
  @Transactional
  public ProductStock frozen(ProductStockOperationDto productStockOperationDto) {
    Validate.notNull(productStockOperationDto.getWarehouseCode(), "变更库存商品数量时仓库编码不能为空");
    Validate.notNull(productStockOperationDto.getProductCode(), "变更库存商品数量时商品编码不能为空");
    Validate.notNull(productStockOperationDto.getQuantity(), "变更库存商品数量不能为空");
    //查询商品库存数据
    ProductStock productStock =
        productStockRepository.findByWarehouseCodeAndProductCode(productStockOperationDto.getWarehouseCode(),
            productStockOperationDto.getProductCode(), TenantUtils.getTenantCode());
    Validate.notNull(productStock, "无法找到对应的商品[%s]在仓库[%s]的库存数据",
        productStockOperationDto.getProductCode(), productStockOperationDto.getWarehouseCode());
    BigDecimal availableStock = productStock.getAvailableStock().subtract(productStockOperationDto.getQuantity().abs());
    //2022-4-29前逻辑  冻结库存后，可用商品库存数量不能小于0
    //应产品要求逻辑就是：当不管控订单库存的时候，库存占用那边也不需要管控可用数量大于0。
    productStock.setAvailableStock(availableStock);
    productStock.setFrozenStock(productStock.getFrozenStock().add(productStockOperationDto.getQuantity().abs()));
    productStockRepository.updateById(productStock);
    //记录库存明细
    createProductStockDetail(productStockOperationDto, productStock, ProductStockOperation.FROZEN,
        BigDecimal.ZERO, productStockOperationDto.getQuantity().abs());
    return productStock;
  }


  @Override
  @Transactional
  public ProductStock thaw(ProductStockOperationDto productStockOperationDto) {
    Validate.notNull(productStockOperationDto.getWarehouseCode(), "变更库存商品数量时仓库编码不能为空");
    Validate.notNull(productStockOperationDto.getProductCode(), "变更库存商品数量时商品编码不能为空");
    Validate.notNull(productStockOperationDto.getQuantity(), "变更库存商品数量不能为空");
    //查询商品库存数据
    ProductStock productStock =
        productStockRepository.findByWarehouseCodeAndProductCode(productStockOperationDto.getWarehouseCode(),
            productStockOperationDto.getProductCode(), TenantUtils.getTenantCode());
    Validate.notNull(productStock, "无法找到对应的商品库存数据");
    productStock.setAvailableStock(productStock.getAvailableStock().add(productStockOperationDto.getQuantity().abs()));
    BigDecimal frozenStock = productStock.getFrozenStock().subtract(productStockOperationDto.getQuantity().abs());
    //解冻库存后，冻结库存数量不能小于0
    Validate.isTrue(frozenStock.compareTo(BigDecimal.ZERO) >= 0, "冻结库存数量不能小于0");
    productStock.setFrozenStock(frozenStock);
    productStockRepository.updateById(productStock);
    //记录库存明细
    createProductStockDetail(productStockOperationDto, productStock, ProductStockOperation.THAW,
        BigDecimal.ZERO, productStockOperationDto.getQuantity().abs().negate());
    return productStock;
  }

  /**
   * 通过商品编码集合和库存类型查询
   * @param productCodes
   * @return
   */
  @Override
  public List<ProductStock> findByTypeAndProductCodeIn(List<String> productCodes, String type) {
    if(CollectionUtils.isEmpty(productCodes) || StringUtils.isBlank(type)){
      return Lists.newArrayList();
    }
    return productStockRepository.findByTypeAndProductCodeIn(productCodes, type);
  }

  /**
   * 根据仓库编码和商品编码集合查询库存
   * @param warehouseCode
   * @param productCodes
   * @return
   */
  @Override
  public List<ProductStock> findByWarehouseAndProducts(String warehouseCode, Set<String> productCodes) {
    if(CollectionUtils.isEmpty(productCodes) || StringUtils.isBlank(warehouseCode)){
      return Lists.newArrayList();
    }
    return productStockRepository.findByWarehouseAndProducts(warehouseCode, productCodes);
  }

  @Override
  public List<ProductStock> findByWarehouseCode(String warehouseCode) {
    if (StringUtils.isBlank(warehouseCode)){
      return new ArrayList<>();
    }
    return productStockRepository.findByWarehouseCode(warehouseCode);
  }

  @Override
  public void updateWarehouseMsg(String warehouseCode, String warehouseName) {
   Validate.isTrue(StringUtils.isNoneBlank(warehouseCode,warehouseName),"更新库存信息，缺失仓库编码或者仓库名称");
   productStockRepository.updateWarehouseMsg(warehouseCode,warehouseName);
  }
}
