package com.biz.crm.dms.business.psi.product.local.service.productstock.internal;
/**
 * Created by Bao Hongbin on 2022-01-12 15:58.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStock;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductStockRepository;
import com.biz.crm.dms.business.psi.product.local.service.productstock.ProductStockService;
import com.biz.crm.dms.business.psi.product.sdk.common.constant.ProductStockConstant;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductStockPaginationDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.StockType;
import com.biz.crm.dms.business.psi.product.sdk.service.productstock.ProductStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductStockVo;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.product.level.sdk.dto.RelateProductLevelCodeQueryDto;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @program: crm-dms
 * @description: 商品库存vo服务实现
 * @author: Bao Hongbin
 * @create: 2022-01-12 15:58
 **/
@Service
public class ProductStockVoServiceImpl implements ProductStockVoService {


  @Autowired(required = false)
  private ProductStockService productStockService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private RedissonClient redissonClient;
  @Autowired(required = false)
  private ProductStockRepository productStockRepository;
  @Autowired(required = false)
  private ProductVoService productVoService;
  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;
  @Autowired(required = false)
  private MaterialVoService materialVoService;

  @Override
  public Page<ProductStockVo> findByConditions(Pageable pageable, ProductStockPaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new ProductStockPaginationDto());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    if (StringUtils.isNotBlank(paginationDto.getProductLevelCode())) {
      Set<String> productCodeSet =
          this.findProductCodeSetByProductLevelCode(
              paginationDto.getProductLevelCode(), paginationDto.getType());
      if (CollectionUtils.isEmpty(productCodeSet)) {
        return new Page<>();
      }
      paginationDto.setProductCodeSet(productCodeSet);
      paginationDto.setProductLevelCode(null);
    }
    Page<ProductStockVo> page = productStockRepository.findByConditions(pageable, paginationDto);
    List<ProductStockVo> records = page.getRecords();
    if (CollectionUtils.isNotEmpty(records) && paginationDto.getType().equals(StockType.PRODUCT.getKey())) {
      List<String> productCodes = records.stream().map(ProductStockVo::getProductCode).collect(Collectors.toList());
      Map<String, ProductVo> ProductVoMap = productVoService.findMainDetailsByProductCodes(productCodes)
          .stream().collect(
              Collectors.toMap(
                  ProductVo::getProductCode, Function.identity(), (o1, o2) -> o2));
      for (ProductStockVo record : records) {
        ProductVo productVo = ProductVoMap.get(record.getProductCode());
        Validate.notNull(productVo, "无法获取的商品信息");
        record.setProductLevelCode(productVo.getProductLevelCode());
        record.setProductLevelName(productVo.getProductLevelName());
        record.setProductUnit(productVo.getSaleUnit());
        record.setSpec(productVo.getSpec());
      }
    }
    if (CollectionUtils.isNotEmpty(records) && paginationDto.getType().equals(StockType.MATERIAL.getKey())) {
      Set<String> materialCodes = records.stream().map(ProductStockVo::getProductCode).collect(Collectors.toSet());
      //获取物料信息
      List<MaterialVo> materials = materialVoService.findDetailByMaterialCodes(materialCodes);
      Validate.notEmpty(materials, "未获取到物料信息");
      Map<String, MaterialVo> materialMap = materials
              .stream().collect(
                      Collectors.toMap(
                              MaterialVo::getMaterialCode, Function.identity(), (o1, o2) -> o2));
      for (ProductStockVo record : records) {
        MaterialVo materialVo = materialMap.get(record.getProductCode());
        Validate.notNull(materialVo, "无法获取物料信息");
        record.setCostPrice(materialVo.getCostPrice());
        record.setMaterialType(materialVo.getMaterialType());
        record.setProductUnit(materialVo.getStandardUnit());
        record.setSpec(materialVo.getSpecification());
      }
    }
    return page;
  }

  @Override
  @Transactional
  public ProductStockVo orderFrozen(ProductStockOperationDto productStockOperationDto) {
    Validate.notNull(productStockOperationDto.getOrderType(), "冻结商品库存时，订单类型不能为空");
    Validate.notNull(productStockOperationDto.getOrderCode(), "冻结商品库存时，订单编号不能为空");
    Validate.notNull(productStockOperationDto.getCustomerCode(), "冻结商品库存时，客户编号不能为空");
    Validate.notNull(productStockOperationDto.getCustomerName(), "冻结商品库存时，客户名称不能为空");
    Validate.notNull(productStockOperationDto.getWarehouseCode(), "变更库存商品数量时仓库编码不能为空");
    Validate.notNull(productStockOperationDto.getProductCode(), "变更库存商品数量时商品编码不能为空");
    TransactionStatus transactionStatus = null;
    //添加分布式锁
    RLock rLock = redissonClient.getLock(String.format(
        ProductStockConstant.PRODUCT_STOCK_LOCK,
        productStockOperationDto.getWarehouseCode(),
        productStockOperationDto.getProductCode()
    ));
    rLock.lock(5000, TimeUnit.MILLISECONDS);
    try {
      //冻结库存
      if (StringUtils.isBlank(productStockOperationDto.getProductStockOperationType())) {
        productStockOperationDto.setProductStockOperationType(ProductStockConstant.ORDER_FROZEN_OPERATION_TYPE);
      }
      ProductStock frozen = productStockService.frozen(productStockOperationDto);
      ProductStockVo productStockVo = nebulaToolkitService.copyObjectByWhiteList(
          frozen, ProductStockVo.class, HashSet.class, ArrayList.class);
      return productStockVo;
    } finally {
      //事务提交或回滚后解锁
      if (rLock.isHeldByCurrentThread()) {
        rLock.unlock();
      }
    }
  }

  /**
   * 批量冻结库存
   * @param productStockOperationDtos 库存操作dto
   * @return
   */
  @Override
  public List<ProductStockVo> frozenBatch(List<ProductStockOperationDto> productStockOperationDtos) {
    List<ProductStockVo> list = new ArrayList<>();
    for(ProductStockOperationDto dto : productStockOperationDtos){
      ProductStockVo productStockVo = this.orderFrozen(dto);
      list.add(productStockVo);
    }
    return list;
  }

  /**
   * 解冻库存（批量）
   * @param productStockOperationDtos
   * @return
   */
  @Override
  public List<ProductStockVo> thawBatch(List<ProductStockOperationDto> productStockOperationDtos) {
    List<ProductStockVo> list = new ArrayList<>();
    for(ProductStockOperationDto dto : productStockOperationDtos){
      ProductStockVo productStockVo = this.orderCloseThaw(dto);
      list.add(productStockVo);
    }
    return list;
  }

  @Override
  @Transactional
  public ProductStockVo orderCloseThaw(ProductStockOperationDto productStockOperationDto) {
    Validate.notNull(productStockOperationDto.getOrderType(), "解冻商品库存时，订单类型不能为空");
    Validate.notNull(productStockOperationDto.getOrderCode(), "解冻商品库存时，订单编号不能为空");
    Validate.notNull(productStockOperationDto.getCustomerCode(), "解冻商品库存时，客户编号不能为空");
    Validate.notNull(productStockOperationDto.getCustomerName(), "解冻商品库存时，客户名称不能为空");
    Validate.notNull(productStockOperationDto.getWarehouseCode(), "变更库存商品数量时仓库编码不能为空");
    Validate.notNull(productStockOperationDto.getProductCode(), "变更库存商品数量时商品编码不能为空");
    TransactionStatus transactionStatus = null;
    //添加分布式锁
    RLock rLock = redissonClient.getLock(String.format(
        ProductStockConstant.PRODUCT_STOCK_LOCK,
        productStockOperationDto.getWarehouseCode(),
        productStockOperationDto.getProductCode()
    ));
    rLock.lock(5000, TimeUnit.MILLISECONDS);
    try {
      //解冻库存
      if (StringUtils.isBlank(productStockOperationDto.getProductStockOperationType())) {
        productStockOperationDto.setProductStockOperationType(ProductStockConstant.ORDER_CLOSE_THAW_OPERATION_TYPE);
      }
      ProductStock thaw = productStockService.thaw(productStockOperationDto);
      ProductStockVo productStockVo = nebulaToolkitService.copyObjectByWhiteList(
          thaw, ProductStockVo.class, HashSet.class, ArrayList.class);
      return productStockVo;
    } finally {
      //事务提交或回滚后解锁
      rLock.unlock();
    }
  }

  /**
   * 通过商品编码集合和库存类型查询
   * @param productCodes
   * @return
   */
  @Override
  public List<ProductStockVo> findByTypeAndProductCodeIn(List<String> productCodes, String type) {
    if(CollectionUtils.isEmpty(productCodes) || StringUtils.isBlank(type)){
      return Lists.newArrayList();
    }
    List<ProductStock> productStocks = productStockService.findByTypeAndProductCodeIn(productCodes, type);
    Collection<ProductStockVo> productStockVos = nebulaToolkitService.copyCollectionByWhiteList(productStocks, ProductStock.class, ProductStockVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(productStockVos);
  }

  /**
   * 根据仓库编码和商品编码集合查询库存
   * @param warehouseCode
   * @param productCodes
   * @return
   */
  @Override
  public List<ProductStockVo> findByWarehouseAndProducts(String warehouseCode, Set<String> productCodes) {
    if(CollectionUtils.isEmpty(productCodes) || StringUtils.isBlank(warehouseCode)){
      return Lists.newArrayList();
    }
    List<ProductStock> productStocks = productStockService.findByWarehouseAndProducts(warehouseCode, productCodes);
    Collection<ProductStockVo> productStockVos = nebulaToolkitService.copyCollectionByWhiteList(productStocks, ProductStock.class, ProductStockVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(productStockVos);
  }

  /**
   * 根据商品编码查询
   * @param productCodes
   * @return
   */
  @Override
  public List<ProductStockVo> findByProductCodes(List<String> productCodes) {
    if(CollectionUtils.isEmpty(productCodes)){
      return Lists.newArrayList();
    }
    return productStockRepository.findByProductCodes(productCodes);
  }

  /**
   * 获取物料或商品编码集合（获取产品层级及其下级所有的物料或商品编码集合）
   *
   * @param productLevelCode 产品层级编码
   * @param productLevelCode 仓库类型
   * @return
   */
  private Set<String> findProductCodeSetByProductLevelCode(
      String productLevelCode, String stockType) {
    Set<String> validateSet =
        Sets.newHashSet(StockType.PRODUCT.getKey(), StockType.MATERIAL.getKey());
    if (StringUtils.isAnyBlank(productLevelCode, stockType) || !validateSet.contains(stockType)) {
      return Sets.newHashSet();
    }
    // 获取productLevelCode层级及其下级的未删除的层级编码
    final RelateProductLevelCodeQueryDto levelCodeQueryDto = new RelateProductLevelCodeQueryDto();
    levelCodeQueryDto.setSearchType(-1);
    levelCodeQueryDto.setProductLevelCodeSet(Sets.newHashSet(productLevelCode));
    final Map<String, String> mapLevelCode =
        this.productLevelVoSdkService.findByRelateProductLevelCodeQueryDto(levelCodeQueryDto);
    if (mapLevelCode.isEmpty()) {
      return Sets.newHashSet();
    }
    Set<String> productLevelCodeSet = mapLevelCode.keySet();

    if (stockType.equals(StockType.PRODUCT.getKey())) {
      final List<ProductVo> productList =
          this.productVoService.findByProductLevelCodes(Lists.newArrayList(productLevelCodeSet));
      if (CollectionUtils.isEmpty(productList)) {
        return Sets.newHashSet();
      }
      return productList.stream()
          .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
          .map(ProductVo::getProductCode)
          .collect(Collectors.toSet());
    } else {
      return this.materialVoService.findCodeByProductLevelCodes(productLevelCodeSet);
    }
  }
}
