package com.biz.crm.dms.business.psi.product.local.service.productstock.internal;
/**
 * Created by Bao Hongbin on 2022-01-15 15:39.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductSecurityStock;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductSecurityStockQuantity;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductSecurityStockQuantityWarehouse;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductSecurityStockQuantityRepository;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductSecurityStockQuantityWarehouseRepository;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductSecurityStockRepository;
import com.biz.crm.dms.business.psi.product.local.service.productstock.ProductSecurityStockVoService;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductSecurityStockConditionDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductSecurityStockCreateDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductSecurityStockPaginationDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductSecurityStockQuantityDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductSecurityStockQuantityWarehouseDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.ProductSecurityStockUpdateDto;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductSecurityStockQuantityVo;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductSecurityStockQuantityWarehouseVo;
import com.biz.crm.dms.business.psi.product.sdk.vo.productstock.ProductSecurityStockVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @program: crm-dms
 * @description: 商品安全库存vo服务实现
 * @author: Bao Hongbin
 * @create: 2022-01-15 15:39
 **/
@Service
public class ProductSecurityStockVoServiceImpl implements ProductSecurityStockVoService {
  @Autowired(required = false)
  private ProductSecurityStockRepository productSecurityStockRepository;
  @Autowired(required = false)
  private ProductSecurityStockQuantityRepository productSecurityStockQuantityRepository;
  @Autowired(required = false)
  private ProductSecurityStockQuantityWarehouseRepository productSecurityStockQuantityWarehouseRepository;
  @Autowired(required = false)
  private ProductVoService productVoService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<ProductSecurityStockVo> findByConditions(Pageable pageable, ProductSecurityStockPaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new ProductSecurityStockPaginationDto());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    Page<ProductSecurityStockVo> page = productSecurityStockRepository.findByConditions(pageable, paginationDto);
    List<ProductSecurityStockVo> records = page.getRecords();
    if (CollectionUtils.isNotEmpty(records)) {
      List<String> productCodes = records.stream().map(ProductSecurityStockVo::getProductCode).collect(Collectors.toList());
      Map<String, ProductVo> ProductVoMap = productVoService.findMainDetailsByProductCodes(productCodes)
          .stream().collect(
              Collectors.toMap(
                  ProductVo::getProductCode, Function.identity(), (o1, o2) -> o2));
      for (ProductSecurityStockVo record : records) {
        ProductVo productVo = ProductVoMap.get(record.getProductCode());
        Validate.notNull(productVo, "无法获取的商品信息");
        record.setProductLevelCode(productVo.getProductLevelCode());
        record.setProductLevelName(productVo.getProductLevelName());
        record.setProductSpec(productVo.getSpec());
        record.setProductUnit(productVo.getSaleUnit());
      }
    }
    return page;
  }


  private List<ProductSecurityStockVo> findDetailsByConditions(ProductSecurityStockConditionDto dto) {
    return productSecurityStockRepository.findDetailsByConditions(dto, TenantUtils.getTenantCode());
  }


  @Override
  public ProductSecurityStockVo findDetailsById(String id) {
    if (!StringUtils.isNotEmpty(id)) {
      return null;
    }
    //查询数据
    ProductSecurityStockVo securityStockVo = productSecurityStockRepository.findDetailsById(id, TenantUtils.getTenantCode());
    if (Objects.isNull(securityStockVo)) {
      return null;
    }
    List<ProductVo> productVos =
        productVoService.findMainDetailsByProductCodes(Lists.newArrayList(securityStockVo.getProductCode()));
    ProductVo productVo = productVos.stream().findFirst().orElseThrow(() ->
        new IllegalArgumentException("无法找到对应的商品信息"));
    securityStockVo.setProductSpec(productVo.getSpec());
    securityStockVo.setProductUnit(productVo.getSaleUnit());
    return securityStockVo;
  }

  @Override
  @Transactional
  public ProductSecurityStockVo create(ProductSecurityStockCreateDto createDto) {
    return this.createForm(createDto);
  }

  private ProductSecurityStockVo createForm(ProductSecurityStockCreateDto createDto) {
    //验证
    validateCreate(createDto);
    //设置数据
    ProductSecurityStock securityStock = this.nebulaToolkitService.copyObjectByWhiteList(
        createDto, ProductSecurityStock.class, HashSet.class, ArrayList.class);
    securityStock.setTenantCode(TenantUtils.getTenantCode());
    securityStock.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    securityStock.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //查询主表
    ProductSecurityStockConditionDto dto =new ProductSecurityStockConditionDto();
    dto.setProductCode(securityStock.getProductCode());
    List<ProductSecurityStockVo> detailsByConditions = productSecurityStockRepository.findDetailsByConditions(dto, TenantUtils.getTenantCode());
    if (CollectionUtils.isNotEmpty(detailsByConditions)) {
      Validate.isTrue(detailsByConditions.size()==1,"查询到多条商品数据，可能由于脏数据引起");
      securityStock.setId(detailsByConditions.get(0).getId());
    }
    //保存
    productSecurityStockRepository.saveOrUpdate(securityStock);
    ProductSecurityStockVo securityStockVo = nebulaToolkitService.copyObjectByWhiteList(
        securityStock, ProductSecurityStockVo.class, HashSet.class, ArrayList.class);
    //保存详情
    List<ProductSecurityStockQuantityVo> quantityVos = saveDetails(securityStock, createDto.getQuantityDtos());
    securityStockVo.setQuantityVos(quantityVos);
    List<ProductVo> productVos =
        productVoService.findMainDetailsByProductCodes(Lists.newArrayList(securityStockVo.getProductCode()));
    ProductVo productVo = productVos.stream().findFirst().orElseThrow(() ->
        new IllegalArgumentException("无法找到对应的商品信息"));
    securityStockVo.setProductSpec(productVo.getSpec());
    securityStockVo.setProductUnit(productVo.getSaleUnit());
    return securityStockVo;
  }

  /**
   * 保存安全库存数量
   *
   * @param securityStock
   * @param quantityDtos
   * @return
   */
  private List<ProductSecurityStockQuantityVo> saveDetails(ProductSecurityStock securityStock,
                                                           List<ProductSecurityStockQuantityDto> quantityDtos) {
    return quantityDtos.stream().map(quantityDto -> {
      ProductSecurityStockQuantity securityStockQuantity = nebulaToolkitService.copyObjectByWhiteList(
          quantityDto, ProductSecurityStockQuantity.class, HashSet.class, ArrayList.class);
      securityStockQuantity.setTenantCode(TenantUtils.getTenantCode());
      securityStockQuantity.setSecurityStockId(securityStock.getId());
      //保存数量信息
      productSecurityStockQuantityRepository.save(securityStockQuantity);
      ProductSecurityStockQuantityVo productSecurityStockQuantityVo = nebulaToolkitService.copyObjectByWhiteList(
          securityStockQuantity, ProductSecurityStockQuantityVo.class, HashSet.class, ArrayList.class);
      List<ProductSecurityStockQuantityWarehouseDto> warehouseDtos = quantityDto.getWarehouseDtos();
      if (CollectionUtils.isNotEmpty(warehouseDtos)) {
        List<ProductSecurityStockQuantityWarehouse> productSecurityStockQuantityWarehouses =
            (ArrayList<ProductSecurityStockQuantityWarehouse>) nebulaToolkitService.copyCollectionByWhiteList(warehouseDtos,
                ProductSecurityStockQuantityWarehouseDto.class,
                ProductSecurityStockQuantityWarehouse.class,
                HashSet.class,
                ArrayList.class);
        for (ProductSecurityStockQuantityWarehouse securityStockQuantityWarehouse : productSecurityStockQuantityWarehouses) {
          securityStockQuantityWarehouse.setTenantCode(TenantUtils.getTenantCode());
          securityStockQuantityWarehouse.setSecurityStockQuantityId(securityStockQuantity.getId());
        }
        //保存安全库存数量对应仓库信息
        productSecurityStockQuantityWarehouseRepository.saveBatch(productSecurityStockQuantityWarehouses);
        List<ProductSecurityStockQuantityWarehouseVo> productSecurityStockQuantityWarehouseVos =
            (ArrayList<ProductSecurityStockQuantityWarehouseVo>)
                nebulaToolkitService.copyCollectionByWhiteList(productSecurityStockQuantityWarehouses,
                    ProductSecurityStockQuantityWarehouse.class,
                    ProductSecurityStockQuantityWarehouseVo.class,
                    HashSet.class,
                    ArrayList.class);
        productSecurityStockQuantityVo.setWarehouseVos(productSecurityStockQuantityWarehouseVos);
      }
      return productSecurityStockQuantityVo;
    }).collect(Collectors.toList());
  }

  private void validateCreate(ProductSecurityStockCreateDto createDto) {
    Validate.notNull(createDto.getProductCode(), "安全库存商品编码不能为空");
    Validate.notNull(createDto.getProductName(), "安全库存商品名称不能为空");
    Validate.notEmpty(createDto.getQuantityDtos(), "安全库存数量信息不能为空");
    int count = 0;
    for (ProductSecurityStockQuantityDto quantityDto : createDto.getQuantityDtos()) {
      Validate.notNull(quantityDto.getSecurityStockQuantity(), "安全库存数量不能为空");
      Validate.notNull(quantityDto.getSort(), "安全库存数量排序不能为空");
      Validate.notNull(quantityDto.getApplicableToAll(), "安全库存数量是否使用所有仓库不能为空");
      if (!quantityDto.getApplicableToAll()) {
        Validate.notEmpty(quantityDto.getWarehouseDtos(), "非适用于所有仓库的安全库存数量信息对应仓库数据不能为空");
        for (ProductSecurityStockQuantityWarehouseDto warehouseDto : quantityDto.getWarehouseDtos()) {
          Validate.notNull(warehouseDto.getWarehouseCode(), "安全库存适用仓库编码不能为空");
          Validate.notNull(warehouseDto.getWarehouseName(), "安全库存适用仓库名称不能为空");
        }
      } else {
        Validate.isTrue(CollectionUtils.isEmpty(quantityDto.getWarehouseDtos()), "适用于所有仓库的安全库存数量信息不能有对应仓库数据");
        count++;
      }
    }
    Validate.isTrue(count == 1 || count == 0, "适用于所有仓库唯一");
    /**
     * 校验安全库存的唯一性
     * 1. 拿到商品编码 获取是否适用于全部仓库  校验重复性 
     * 2. 不适用于全部仓库，获取当前商品已有的安全库存仓库， 校验重复性
     */
    String productCode = createDto.getProductCode();
    List<ProductSecurityStockQuantityDto> quantityDtos = createDto.getQuantityDtos();
    for (ProductSecurityStockQuantityDto quantityDto : quantityDtos) {
      if (quantityDto.getApplicableToAll() == true) {
        //如果适用于全部仓库
        ProductSecurityStockConditionDto dto = new ProductSecurityStockConditionDto();
        dto.setProductCode(productCode);
        dto.setApplicableToAll(true);
        List<ProductSecurityStockVo> detailsByConditions = this.findDetailsByConditions(dto);
        Validate.isTrue(CollectionUtils.isEmpty(detailsByConditions), "[商品:" + productCode + "]已存在适用于全部仓库的数据");
      } else {
        //如果不适应于全部仓库
        List<ProductSecurityStockQuantityWarehouseDto> warehouseDtos = quantityDto.getWarehouseDtos();
        for (ProductSecurityStockQuantityWarehouseDto warehouseDto : warehouseDtos) {
          ProductSecurityStockConditionDto dto = new ProductSecurityStockConditionDto();
          dto.setProductCode(productCode);
          dto.setWarehouseCode(warehouseDto.getWarehouseCode());
          List<ProductSecurityStockVo> detailsByConditions = this.findDetailsByConditions(dto);
          Validate.isTrue(CollectionUtils.isEmpty(detailsByConditions), "[商品:" + productCode + "]已存在当前仓库的数据");
        }
      }
    }
  }

  @Override
  @Transactional
  public ProductSecurityStockVo update(ProductSecurityStockUpdateDto updateDto) {
    return this.updateForm(updateDto);
  }

  private ProductSecurityStockVo updateForm(ProductSecurityStockUpdateDto updateDto) {
    //验证
    validateUpdate(updateDto);
    //设置数据
    ProductSecurityStock securityStock = this.nebulaToolkitService.copyObjectByWhiteList(
        updateDto, ProductSecurityStock.class, HashSet.class, ArrayList.class);
    //更新数据
    productSecurityStockRepository.updateById(securityStock);
    ProductSecurityStockVo securityStockVo = nebulaToolkitService.copyObjectByWhiteList(
        securityStock, ProductSecurityStockVo.class, HashSet.class, ArrayList.class);
    //删除旧详情
    List<ProductSecurityStockQuantity> productSecurityStockQuantities =
        productSecurityStockQuantityRepository.findBySecurityStockId(securityStock.getId());
    Validate.notEmpty(productSecurityStockQuantities, "无法找到安全库存的数量信息");
    List<String> quantityIds = productSecurityStockQuantities.stream().map(ProductSecurityStockQuantity::getId).collect(Collectors.toList());
    //删除数量信息
    productSecurityStockQuantityRepository.removeByIds(quantityIds);
    //删除仓库信息
    productSecurityStockQuantityWarehouseRepository.deleteByQuantityIds(quantityIds);

    //保存新详情
    List<ProductSecurityStockQuantityVo> quantityVos = saveDetails(securityStock, updateDto.getQuantityDtos());
    securityStockVo.setQuantityVos(quantityVos);
    List<ProductVo> productVos =
        productVoService.findMainDetailsByProductCodes(Lists.newArrayList(securityStockVo.getProductCode()));
    ProductVo productVo = productVos.stream().findFirst().orElseThrow(() ->
        new IllegalArgumentException("无法找到对应的商品信息"));
    securityStockVo.setProductSpec(productVo.getSpec());
    securityStockVo.setProductUnit(productVo.getSaleUnit());
    return securityStockVo;

  }

  private void validateUpdate(ProductSecurityStockUpdateDto updateDto) {
    Validate.notNull(updateDto.getId(), "变更安全库存时id不能为空");
    Validate.notNull(updateDto.getProductCode(), "安全库存商品编码不能为空");
    Validate.notNull(updateDto.getProductName(), "安全库存商品名称不能为空");
    Validate.notEmpty(updateDto.getQuantityDtos(), "安全库存数量信息不能为空");
    int count = 0;
    for (ProductSecurityStockQuantityDto quantityDto : updateDto.getQuantityDtos()) {
      Validate.notNull(quantityDto.getSecurityStockQuantity(), "安全库存数量不能为空");
      Validate.notNull(quantityDto.getSort(), "安全库存数量排序不能为空");
      Validate.notNull(quantityDto.getApplicableToAll(), "安全库存数量是否使用所有仓库不能为空");
      if (!quantityDto.getApplicableToAll()) {
        Validate.notEmpty(quantityDto.getWarehouseDtos(), "非适用于所有仓库的安全库存数量信息对应仓库数据不能为空");
        for (ProductSecurityStockQuantityWarehouseDto warehouseDto : quantityDto.getWarehouseDtos()) {
          Validate.notNull(warehouseDto.getWarehouseCode(), "安全库存适用仓库编码不能为空");
          Validate.notNull(warehouseDto.getWarehouseName(), "安全库存适用仓库名称不能为空");
        }
      } else {
        Validate.isTrue(CollectionUtils.isEmpty(quantityDto.getWarehouseDtos()), "适用于所有仓库的安全库存数量信息不能有对应仓库数据");
        count++;
      }
    }
    Validate.isTrue(count == 1 || count == 0, "适用于所有仓库唯一");
    /**
     * 因为不熟悉当前业务
     * 以下校验可能与上面的校验存在部分重复
     */
    //对当前报文中的仓库进行重复性校验,比较 set和List的大小  不对等则说明 仓库有重复
    List<ProductSecurityStockQuantityDto> quantityDtos = updateDto.getQuantityDtos();
    HashSet warehouseSet = new HashSet();
    List<String> warehouseList = new ArrayList<>();
    for (ProductSecurityStockQuantityDto quantityDto : quantityDtos) {
      List<ProductSecurityStockQuantityWarehouseDto> warehouseDtos = quantityDto.getWarehouseDtos();
      if (!CollectionUtils.isEmpty(warehouseDtos)) {
        List<String> codes = warehouseDtos.stream().map(ProductSecurityStockQuantityWarehouseDto::getWarehouseCode).collect(Collectors.toList());
        warehouseSet.addAll(codes);
        warehouseList.addAll(codes);
      }
    }
    Validate.isTrue(warehouseSet.size() == warehouseList.size(), "仓库存在重复");
    //对主列表同一商品不同行的情况进行仓库校验
    ProductSecurityStockConditionDto dto = new ProductSecurityStockConditionDto();
    dto.setProductCode(updateDto.getProductCode());
    //查询当前商品所有数据
    List<ProductSecurityStockVo> productSecurityStockVoList = this.findDetailsByConditions(dto);
    //排除当前需要更新的数据
    List<ProductSecurityStockVo> exclude = productSecurityStockVoList.stream().filter(e -> !e.getId().equals(updateDto.getId())).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(exclude)) {
      List<ProductSecurityStockQuantityWarehouseVo> warehouse = new ArrayList<>();
      Boolean isAll = false;
      for (ProductSecurityStockVo vo : exclude) {
        List<ProductSecurityStockQuantityVo> quantityVos = vo.getQuantityVos();
        if (CollectionUtils.isNotEmpty(quantityVos)) {
          for (ProductSecurityStockQuantityVo quantityVo : quantityVos) {
            if (quantityVo.getApplicableToAll()) {
              //如果是默认仓库
              isAll=true;
            } else {
              //统计全部仓库
              warehouse.addAll(quantityVo.getWarehouseVos());
            }
          }
        }
      }
      //如果报文中存在适用于全部仓库的数据
      List<ProductSecurityStockQuantityDto> isTrueList = quantityDtos.stream().filter(e -> e.getApplicableToAll() == true).collect(Collectors.toList());
      if (CollectionUtils.isNotEmpty(isTrueList)) {
        Validate.isTrue(!isAll,"当前商品已存在适用于全部的默认仓库数据");
      }
      //除开适用于全部仓库的其他仓库数据
      if (CollectionUtils.isNotEmpty(warehouse)) {
        //获取数据库中仓库编号
        List<String> wareHouseCodes = warehouse.stream().map(ProductSecurityStockQuantityWarehouseVo::getWarehouseCode).collect(Collectors.toList());
        //与当前报文中的仓库进行比对
        wareHouseCodes.retainAll(warehouseSet);
        Validate.isTrue(CollectionUtils.isEmpty(wareHouseCodes),"当前商品的安全仓库与已有的安全仓库重复,仓库编码["+wareHouseCodes.toString()+"]");
      }
    }

  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    //删除
    productSecurityStockRepository.deleteBatch(ids);

  }

  /**
   * 批量保存
   * 1.重复厂库验证
   * 1.1 过滤适用全部厂库的数据，不需要验证放行
   * 1.2 同一个商品不允许配置重复的厂库
   * <p>
   * 2.保存数据
   * 2.1 组装数据
   * ag：
   * {
   * "productCode": "",
   * "productName": "",
   * "quantityDtos": [
   * {
   * "applicableToAll": true,
   * "securityStockQuantity": 0,
   * "sort": 0,
   * "warehouseDtos": [
   * {
   * "warehouseCode": "",
   * "warehouseName": ""
   * }]
   * }],
   * "remark": ""
   * }
   *
   * @param createDtos
   * @return
   */
  @Override
  @Transactional
  public List<ProductSecurityStockVo> createBatch(List<ProductSecurityStockCreateDto> createDtos) {
    Validate.notEmpty(createDtos, "批量保存时，数据不能为空");
    for (ProductSecurityStockCreateDto createDto : createDtos) {
      validateCreate(createDto);
    }
    //重复厂库验证
    Map<String, List<String>> map = new HashMap<>();
    for (ProductSecurityStockCreateDto dto : createDtos) {
      List<ProductSecurityStockQuantityDto> quantityDtos = dto.getQuantityDtos();
      if (map.containsKey(dto.getProductCode())) {
        if (CollectionUtils.isNotEmpty(quantityDtos)) {
          for (ProductSecurityStockQuantityDto quantityDto : quantityDtos) {
            if (quantityDto.getApplicableToAll()) {
              continue;
            }
            List<ProductSecurityStockQuantityWarehouseDto> warehouseDtos = quantityDto.getWarehouseDtos();
            warehouseDtos.stream().forEach(item -> {
              Validate.isTrue(!(map.get(dto.getProductCode()).contains(item.getWarehouseCode())), "%s商品，%s安全库重复，请检查！", dto.getProductCode(), item.getWarehouseCode());
            });
          }
        }
      } else {
        List<String> warehouse = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(quantityDtos)) {
          for (ProductSecurityStockQuantityDto quantityDto : quantityDtos) {
            if (quantityDto.getApplicableToAll()) {
              continue;
            }
            List<ProductSecurityStockQuantityWarehouseDto> warehouseDtos = quantityDto.getWarehouseDtos();
            warehouseDtos.stream().forEach(item -> warehouse.add(item.getWarehouseCode()));
          }
        }
        map.put(dto.getProductCode(), warehouse);
      }
    }
    //保存
    List<ProductSecurityStockVo> list = new ArrayList<>();
    Map<String, List<ProductSecurityStockCreateDto>> productMap = createDtos.stream().collect(Collectors.groupingBy(ProductSecurityStockCreateDto::getProductCode));

    //组装参数
    for (Map.Entry<String, List<ProductSecurityStockCreateDto>> entry : productMap.entrySet()) {
      ProductSecurityStockCreateDto dto = new ProductSecurityStockCreateDto();
      List<ProductSecurityStockQuantityDto> quantityDtos = new ArrayList<>();
      List<ProductSecurityStockCreateDto> values = entry.getValue();
      dto.setProductCode(values.get(0).getProductCode());
      dto.setProductName(values.get(0).getProductName());
      for (ProductSecurityStockCreateDto value : values) {
        quantityDtos.addAll(value.getQuantityDtos());
      }
      dto.setQuantityDtos(quantityDtos);
      ProductSecurityStockVo productSecurityStockVo = this.create(dto);
      list.add(productSecurityStockVo);
    }
    return list;
  }
}
