package com.biz.crm.dms.business.psi.product.local.service.store.internal;
/**
 * Created by Bao Hongbin on 2022-01-15 15:05.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStock;
import com.biz.crm.dms.business.psi.product.local.entity.productstock.ProductStockDetail;
import com.biz.crm.dms.business.psi.product.local.entity.store.ProductStoredBill;
import com.biz.crm.dms.business.psi.product.local.entity.store.ProductStoredBillDetail;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductStockDetailRepository;
import com.biz.crm.dms.business.psi.product.local.repository.productstock.ProductStockRepository;
import com.biz.crm.dms.business.psi.product.local.repository.store.ProductStoredBillDetailRepository;
import com.biz.crm.dms.business.psi.product.local.repository.store.ProductStoredBillRepository;
import com.biz.crm.dms.business.psi.product.local.service.store.ProductStoredBillVoService;
import com.biz.crm.dms.business.psi.product.sdk.common.constant.ProductStoredBillConstant;
import com.biz.crm.dms.business.psi.product.sdk.context.store.StoredBillProductStockContext;
import com.biz.crm.dms.business.psi.product.sdk.dto.productstock.StoredProductStockOperationDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.store.ProductStoredBillCreateDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.store.ProductStoredBillDetailDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.store.ProductStoredBillPaginationDto;
import com.biz.crm.dms.business.psi.product.sdk.dto.store.ProductStoredBillUpdateDto;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockBillType;
import com.biz.crm.dms.business.psi.product.sdk.enums.productstock.ProductStockOperation;
import com.biz.crm.dms.business.psi.product.sdk.enums.store.ProductStoreStatus;
import com.biz.crm.dms.business.psi.product.sdk.event.ProductStoredListener;
import com.biz.crm.dms.business.psi.product.sdk.register.store.StoredBillProductStockRegister;
import com.biz.crm.dms.business.psi.product.sdk.vo.store.ProductStoredBillDetailVo;
import com.biz.crm.dms.business.psi.product.sdk.vo.store.ProductStoredBillVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @program: crm-dms
 * @description: 商品入库单服务vo实现
 * @author: Bao Hongbin
 * @create: 2022-01-15 15:05
 **/
@Service
public class ProductStoredBillVoServiceImpl implements ProductStoredBillVoService {
  @Autowired(required = false)
  private ProductStoredBillRepository productStoredBillRepository;
  @Autowired(required = false)
  private ProductStoredBillDetailRepository productStoredBillDetailRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCode;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private StoredBillProductStockContext storedBillProductStockContext;
  @Autowired(required = false)
  private ProductStockDetailRepository productStockDetailRepository;
  @Autowired(required = false)
  private ProductStockRepository productStockRepository;
  @Autowired(required = false)
  private List<ProductStoredListener> productStoredListeners;

  @Override
  public Page<ProductStoredBillVo> findByConditions(Pageable pageable, ProductStoredBillPaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new ProductStoredBillPaginationDto());
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    return productStoredBillRepository.findByConditions(pageable, paginationDto);
  }

  @Override
  public ProductStoredBillVo findDetailsById(String id) {
    if (!StringUtils.isNotEmpty(id)) {
      return null;
    }
    //查询主表数据
    ProductStoredBill bill = productStoredBillRepository.findDetailsById(id, TenantUtils.getTenantCode());
    if (Objects.isNull(bill)) {
      return null;
    }
    ProductStoredBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductStoredBillVo.class, HashSet.class, ArrayList.class);
    //查询详情
    List<ProductStoredBillDetailVo> detailVos =
        (List<ProductStoredBillDetailVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                productStoredBillDetailRepository.findListByBillId(bill.getId()),
                ProductStoredBillDetail.class,
                ProductStoredBillDetailVo.class,
                HashSet.class,
                ArrayList.class);
    billVo.setDetailVos(detailVos);
    return billVo;
  }

  @Override
  public ProductStoredBillVo findDetailsByCode(String code) {
    if (!StringUtils.isNotEmpty(code)) {
      return null;
    }
    //查询主表数据
    ProductStoredBill bill = productStoredBillRepository.findDetailsByCode(code, TenantUtils.getTenantCode());
    if (Objects.isNull(bill)) {
      return null;
    }
    ProductStoredBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
            bill, ProductStoredBillVo.class, HashSet.class, ArrayList.class);
    //查询详情
    List<ProductStoredBillDetailVo> detailVos =
            (List<ProductStoredBillDetailVo>)
                    this.nebulaToolkitService.copyCollectionByWhiteList(
                            productStoredBillDetailRepository.findListByBillId(bill.getId()),
                            ProductStoredBillDetail.class,
                            ProductStoredBillDetailVo.class,
                            HashSet.class,
                            ArrayList.class);
    billVo.setDetailVos(detailVos);
    return billVo;
  }

  @Override
  @Transactional
  public ProductStoredBillVo create(ProductStoredBillCreateDto createDto) {
    return this.createForm(createDto);
  }

  private ProductStoredBillVo createForm(ProductStoredBillCreateDto createDto) {
    //验证
    validateCreate(createDto);
    //设置数据
    ProductStoredBill bill = this.nebulaToolkitService.copyObjectByWhiteList(
        createDto, ProductStoredBill.class, HashSet.class, ArrayList.class);
    bill.setTenantCode(TenantUtils.getTenantCode());
    bill.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    bill.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //TODO 各业务编码前缀以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
    bill.setStoredBillCode(generateCode.generateCode(ProductStoredBillConstant.CODE, 1).get(0));
    bill.setStoredStatus(ProductStoreStatus.WAIT_STORE);
    //保存数据
    productStoredBillRepository.save(bill);
    ProductStoredBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductStoredBillVo.class, HashSet.class, ArrayList.class);
    //保存详情
    List<ProductStoredBillDetailVo> detailVos = saveDetails(bill, createDto.getDetailDtos());
    billVo.setDetailVos(detailVos);
    //保存库存
    this.handleStoreBatch(Lists.newArrayList(bill.getId()));
    //事件通知
    if (CollectionUtils.isNotEmpty(productStoredListeners)) {
      productStoredListeners.forEach(listener -> {
        listener.onCreate(billVo);
      });
    }
    return billVo;
  }

  /**
   * 保存详情
   *
   * @param bill
   * @param detailDtos
   * @return
   */
  private List<ProductStoredBillDetailVo> saveDetails(ProductStoredBill bill, List<ProductStoredBillDetailDto> detailDtos) {
    List<ProductStoredBillDetail> billDetails =
        (List<ProductStoredBillDetail>) nebulaToolkitService.copyCollectionByWhiteList(detailDtos,
            ProductStoredBillDetailDto.class,
            ProductStoredBillDetail.class,
            HashSet.class,
            ArrayList.class);
    for (ProductStoredBillDetail billDetail : billDetails) {
      billDetail.setTenantCode(TenantUtils.getTenantCode());
      billDetail.setStoredBillId(bill.getId());
      billDetail.setStoredBillCode(bill.getStoredBillCode());
    }
    productStoredBillDetailRepository.saveBatch(billDetails);
    return (List<ProductStoredBillDetailVo>) nebulaToolkitService.copyCollectionByWhiteList(billDetails,
        ProductStoredBillDetail.class,
        ProductStoredBillDetailVo.class,
        HashSet.class,
        ArrayList.class);
  }

  private void validateCreate(ProductStoredBillCreateDto createDto) {
    Validate.notNull(createDto.getStoredTime(), "入库时间不能为空");
    Validate.notBlank(createDto.getStoreWarehouseCode(), "入库仓库编码不能为空");
    Validate.notBlank(createDto.getStoreWarehouseName(), "入库仓库名称不能为空");
    Validate.notBlank(createDto.getProductStockOperationType(), "商品库存操作类型不能为空");
    Validate.notEmpty(createDto.getDetailDtos(), "商品入库单商品行不能为空");
  }

  @Override
  @Transactional
  public ProductStoredBillVo update(ProductStoredBillUpdateDto updateDto) {
    return this.updateForm(updateDto);
  }

  private ProductStoredBillVo updateForm(ProductStoredBillUpdateDto updateDto) {
    //验证
    validateUpdate(updateDto);
    //设置数据
    ProductStoredBill bill = this.nebulaToolkitService.copyObjectByWhiteList(
        updateDto, ProductStoredBill.class, HashSet.class, ArrayList.class);
    bill.setStoredStatus(ProductStoreStatus.WAIT_STORE);
    //更新数据
    productStoredBillRepository.updateById(bill);
    ProductStoredBillVo billVo = nebulaToolkitService.copyObjectByWhiteList(
        bill, ProductStoredBillVo.class, HashSet.class, ArrayList.class);
    //删除原明细数据
    productStoredBillDetailRepository.deleteByBillId(bill.getId());
    //保存详情
    List<ProductStoredBillDetailVo> detailVos = saveDetails(bill, updateDto.getDetailDtos());
    billVo.setDetailVos(detailVos);
    return billVo;
  }

  private void validateUpdate(ProductStoredBillUpdateDto updateDto) {
    Validate.notBlank(updateDto.getId(), "id不能为空");
    Validate.notBlank(updateDto.getStoredBillCode(), "入库单编码不能为空");
    Validate.notNull(updateDto.getStoredTime(), "入库时间不能为空");
    Validate.notBlank(updateDto.getStoreWarehouseCode(), "入库仓库编码不能为空");
    Validate.notBlank(updateDto.getStoreWarehouseName(), "入库仓库名称不能为空");
    Validate.notBlank(updateDto.getProductStockOperationType(), "商品库存操作类型不能为空");
    Validate.notEmpty(updateDto.getDetailDtos(), "商品入库单商品行不能为空");
    ProductStoredBill bill = productStoredBillRepository.findDetailsById(updateDto.getId(), TenantUtils.getTenantCode());
    Validate.isTrue(Objects.nonNull(bill), "找不到对应的入库单");
    ProductStoreStatus storedStatus = bill.getStoredStatus();
    Validate.isTrue(ProductStoreStatus.WAIT_STORE.equals(storedStatus)
        || ProductStoreStatus.REJECTED.equals(storedStatus), "只有待入库和已驳回的入库单才允许编辑");
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductStoredBill> bills =
        productStoredBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    //判断状态
    for (ProductStoredBill bill : bills) {
      ProductStoreStatus status = bill.getStoredStatus();
      Validate.isTrue(ProductStoreStatus.WAIT_STORE.equals(status)
              || ProductStoreStatus.REJECTED.equals(status) || ProductStoreStatus.VOIDED.equals(status),
          "只有待入库、已作废和已驳回的入库单才允许删除");
    }
    //删除
    productStoredBillRepository.deleteBatch(ids);
  }

  @Override
  @Transactional
  public void handleStoreBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductStoredBill> bills =
        productStoredBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    for (ProductStoredBill bill : bills) {
      //判断状态
      ProductStoreStatus status = bill.getStoredStatus();
      Validate.isTrue(ProductStoreStatus.WAIT_STORE.equals(status), "只有待入库的入库单才允许进行入库操作");
      bill.setStoredStatus(ProductStoreStatus.STORED);
      //操作库存
      StoredBillProductStockRegister register =
          storedBillProductStockContext.getRegisterByOperationType(bill.getProductStockOperationType());
      Validate.notNull(register, "无法找到对应类型的入库单操作库存的注册器");
      List<ProductStoredBillDetail> details =
          productStoredBillDetailRepository.findListByBillId(bill.getId());
      Validate.notEmpty(details, "无法找到入库单详情");
      for (ProductStoredBillDetail detail : details) {
        StoredProductStockOperationDto operationDto = buildOperationDto(bill, detail);
        //执行操作
        register.execute(operationDto);
      }
    }
    productStoredBillRepository.updateBatchById(bills);
  }

  /**
   * 构建库存操作请求
   *
   * @param bill
   * @param detail
   * @return
   */
  private StoredProductStockOperationDto buildOperationDto(ProductStoredBill bill, ProductStoredBillDetail detail) {
    StoredProductStockOperationDto operationDto = nebulaToolkitService.copyObjectByWhiteList(
        detail, StoredProductStockOperationDto.class, HashSet.class, ArrayList.class);
    operationDto.setWarehouseCode(bill.getStoreWarehouseCode());
    operationDto.setWarehouseName(bill.getStoreWarehouseName());
    operationDto.setBillType(ProductStockBillType.STORED_ORDER.name());
    operationDto.setBillCode(bill.getStoredBillCode());
    operationDto.setProductLevelName(detail.getProductLevel());
    operationDto.setQuantity(detail.getStoredQuantity());
    operationDto.setType(bill.getType());
    return operationDto;
  }

  @Override
  public void handleRejectedBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductStoredBill> bills =
        productStoredBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    //判断状态
    for (ProductStoredBill bill : bills) {
      ProductStoreStatus status = bill.getStoredStatus();
      Validate.isTrue(ProductStoreStatus.WAIT_STORE.equals(status), "只有待入库的入库单才允许驳回");
      bill.setStoredStatus(ProductStoreStatus.REJECTED);
    }
    //驳回
    productStoredBillRepository.updateBatchById(bills);
  }

  @Override
  @Transactional
  public void handleVoidBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<ProductStoredBill> bills =
        productStoredBillRepository.findListByIds(ids);
    if (CollectionUtils.isEmpty(bills)) {
      return;
    }
    for (ProductStoredBill bill : bills) {
      //判断状态
      ProductStoreStatus status = bill.getStoredStatus();
      Validate.isTrue(ProductStoreStatus.WAIT_STORE.equals(status)
          || ProductStoreStatus.STORED.equals(status), "只有待入库和已入库的入库单才允许进行作废操作");
      bill.setStoredStatus(ProductStoreStatus.VOIDED);
      if (ProductStoreStatus.STORED.equals(status)) {
        //如果为已入库时作废需要回退库存数量
        //回退操作库存
        StoredBillProductStockRegister register =
            storedBillProductStockContext.getRegisterByOperationType(bill.getProductStockOperationType());
        Validate.notNull(register, "无法找到对应类型的入库单操作库存的注册器");
        List<ProductStoredBillDetail> details =
            productStoredBillDetailRepository.findListByBillId(bill.getId());
        Validate.notEmpty(details, "无法找到入库单详情");
        for(ProductStoredBillDetail detail : details) {
          List<ProductStockOperation> operations = new ArrayList<>();
          operations.add(ProductStockOperation.FROZEN);
          operations.add(ProductStockOperation.DELIVER);
          ProductStock byWarehouseCodeAndProductCode = productStockRepository.findByWarehouseCodeAndProductCode(bill.getStoreWarehouseCode(), detail.getProductCode(), TenantUtils.getTenantCode());
          if (Objects.nonNull(byWarehouseCodeAndProductCode)){
            BigDecimal availableStock = byWarehouseCodeAndProductCode.getAvailableStock();
            BigDecimal storedQuantity = detail.getStoredQuantity();
            Validate.isTrue(availableStock.compareTo(storedQuantity)>=0,"当前可用库存小于作废单入库数量，无法作废");
          }
        }
        for (ProductStoredBillDetail detail : details) {
          StoredProductStockOperationDto operationDto = buildOperationDto(bill, detail);
          //回退操作
          register.rescind(operationDto);
        }
      }
    }
    productStoredBillRepository.updateBatchById(bills);
  }

  @Override
  public  List<ProductStoredBillVo> findDetailsByReceivingCode(String code) {
    List<ProductStoredBill> detailsByReceivingCode = this.productStoredBillRepository.findDetailsByReceivingCode(code);
    if (CollectionUtils.isEmpty(detailsByReceivingCode)){
      return new ArrayList<>(0);
    }
    List<String> ids = detailsByReceivingCode.stream().map(ProductStoredBill::getId).collect(Collectors.toList());
    List<ProductStoredBillDetail> listByBillIds = this.productStoredBillDetailRepository.findListByBillIds(ids);
    Collection<ProductStoredBillDetailVo> productStoredBillDetailVos = this.nebulaToolkitService.copyCollectionByWhiteList(listByBillIds, ProductStoredBillDetail.class, ProductStoredBillDetailVo.class, HashSet.class, ArrayList.class);
    Map<String, List<ProductStoredBillDetailVo>> detailMap = productStoredBillDetailVos.stream().collect(Collectors.groupingBy(ProductStoredBillDetailVo::getStoredBillCode));
    Collection<ProductStoredBillVo> productStoredBillVos = this.nebulaToolkitService.copyCollectionByWhiteList(detailsByReceivingCode, ProductStoredBill.class, ProductStoredBillVo.class, HashSet.class, ArrayList.class);
    for (ProductStoredBillVo vo : productStoredBillVos) {
      List<ProductStoredBillDetailVo> detailVos = detailMap.get(vo.getStoredBillCode());
      if (CollectionUtils.isNotEmpty(detailVos)){
        vo.setDetailVos(detailVos);
      }
    }
    return (List<ProductStoredBillVo>) productStoredBillVos;
  }
}
