package com.biz.crm.dms.business.reconciliation.local.context.reconciliationelement;
/**
 * Created by Bao Hongbin on 2021-12-15 11:11.
 */

import com.biz.crm.dms.business.reconciliation.sdk.register.reconciliationelement.ReconciliationElementRegister;
import com.biz.crm.dms.business.reconciliation.sdk.vo.reconciliationelementinfo.ReconciliationElementInfoVo;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * @program: crm-dms
 * @description: 对账要素上下文
 * @author: Bao Hongbin
 * @create: 2021-12-15 11:11
 **/
@Component
public class ReconciliationElementContext {

  /**
   * 储存对账要素编码和对账要素基本信息的映射关系
   */
  private static final Map<String, ReconciliationElementInfoVo> ELEMENT_INFO_CACHE = new ConcurrentHashMap<>();

  /**
   * 储存对账要素编码和对账要素注册器的映射关系
   */
  private static final Map<String, ReconciliationElementRegister> ELEMENT_REGISTERS_CACHE = new ConcurrentHashMap<>();

  @Autowired(required = false)
  public ReconciliationElementContext(List<ReconciliationElementRegister> elementRegisters) {
    if (CollectionUtils.isNotEmpty(elementRegisters)) {
      for (ReconciliationElementRegister elementRegister : elementRegisters) {
        ReconciliationElementInfoVo reconciliationElementInfoVo = elementRegister.getReconciliationElementInfo();
        Validate.notNull(reconciliationElementInfoVo, "无法获取到对账要素注册器的基本信息");
        ELEMENT_INFO_CACHE.put(reconciliationElementInfoVo.getElementCode(), reconciliationElementInfoVo);
        ELEMENT_REGISTERS_CACHE.put(reconciliationElementInfoVo.getElementCode(), elementRegister);
      }
    }
  }

  /**
   * 获取所有对账要素基本信息
   *
   * @return 对账要素基本信息集合
   */
  public List<ReconciliationElementInfoVo> getAllElementInfo() {
    if (CollectionUtils.isEmpty(ELEMENT_INFO_CACHE.values())) {
      return (List<ReconciliationElementInfoVo>) ELEMENT_INFO_CACHE.values();
    }
    //排序
    List<ReconciliationElementInfoVo> reconciliationElementInfoVos = new ArrayList<>(ELEMENT_INFO_CACHE.values());
    reconciliationElementInfoVos.sort(Comparator.comparing(ReconciliationElementInfoVo::getElementSort));
    return reconciliationElementInfoVos;
  }


  /**
   * 获取所有对账要素注册器
   *
   * @return 对账要素注册器集合
   */
  public Collection<ReconciliationElementRegister> getAllElementRegister() {
    return ELEMENT_REGISTERS_CACHE.values();
  }

  /**
   * 根据对账要素编码获取对账要素基本信息
   *
   * @param elementCode 对账要素编码
   * @return 对账要素基本信息
   */
  public ReconciliationElementInfoVo getElementInfoByCode(String elementCode) {
    if (StringUtils.isBlank(elementCode)) {
      return null;
    }
    if (CollectionUtils.isEmpty(ELEMENT_INFO_CACHE.values())) {
      return null;
    }
    return ELEMENT_INFO_CACHE.get(elementCode);
  }

  /**
   * 根据对账要素编码获取对账要素注册器
   *
   * @param elementCode 对账要素编码
   * @return 对账要素注册器
   */
  public ReconciliationElementRegister getElementRegisterByCode(String elementCode) {
    if (StringUtils.isBlank(elementCode)) {
      return null;
    }
    if (CollectionUtils.isEmpty(ELEMENT_REGISTERS_CACHE.values())) {
      return null;
    }
    return ELEMENT_REGISTERS_CACHE.get(elementCode);
  }
}
