package com.biz.crm.dms.business.reconciliation.local.service.reconciliationtemplate.internal;
/**
 * Created by Bao Hongbin on 2021-12-20 16:58.
 */

import com.biz.crm.dms.business.reconciliation.local.entity.reconciliationtemplate.ReconciliationLetterCalculationTime;
import com.biz.crm.dms.business.reconciliation.local.repository.reconciliationtemplate.ReconciliationLetterCalculationTimeRepository;
import com.biz.crm.dms.business.reconciliation.local.service.reconciliationletter.ReconciliationLetterVoService;
import com.biz.crm.dms.business.reconciliation.local.service.reconciliationtemplate.ReconciliationLetterCalculationTimeVoService;
import com.biz.crm.dms.business.reconciliation.sdk.dto.reconciliationletter.ReconciliationLetterManualCreateDto;
import com.biz.crm.dms.business.reconciliation.sdk.dto.reconciliationtemplate.ReconciliationCalculationTaskCreateDto;
import com.biz.crm.dms.business.reconciliation.sdk.dto.reconciliationtemplate.ReconciliationLetterCalculationTimeCreateDto;
import com.biz.crm.dms.business.reconciliation.sdk.dto.reconciliationtemplate.ReconciliationLetterCalculationTimeUpdateDto;
import com.biz.crm.dms.business.reconciliation.sdk.vo.reconciliationtemplate.ReconciliationLetterCalculationTimeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

/**
 * @program: crm-dms
 * @description:
 * @author: Bao Hongbin
 * @create: 2021-12-20 16:58
 **/
@Service
@Slf4j
public class ReconciliationLetterCalculationTimeVoServiceImpl implements ReconciliationLetterCalculationTimeVoService {

  /**
   * code生成规则
   */
  private final String TASK_CODE_FORMAT = "%s-%s-%s";

  /**
   * 计算定时任务类名称
   */
  private final String TASK_INVOKE_BEAN_NAME = "reconciliationCalculationTaskServiceImpl";

  /**
   * 计算定时任务的方法名
   */
  private final String TASK_METHOD = "handleCalculationTask";

  /**
   * 计算定时任务的描述
   */
  private final String TASK_DESC = "计算并生成指定客户的对账函";

  @Autowired(required = false)
  private ReconciliationLetterCalculationTimeRepository reconciliationLetterCalculationTimeRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;
  @Autowired(required = false)
  private ReconciliationLetterVoService reconciliationLetterVoService;

  @Override
  public List<ReconciliationLetterCalculationTimeVo> findListByTemplateCode(String reconciliationTemplateCode) {
    List<ReconciliationLetterCalculationTime> calculationTimes =
        reconciliationLetterCalculationTimeRepository.findListByTemplateCode(reconciliationTemplateCode, TenantUtils.getTenantCode());
    return (List<ReconciliationLetterCalculationTimeVo>) nebulaToolkitService.copyCollectionByWhiteList(calculationTimes, ReconciliationLetterCalculationTime.class
        , ReconciliationLetterCalculationTimeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public List<ReconciliationLetterCalculationTimeVo> create(String reconciliationTemplateCode, List<ReconciliationLetterCalculationTimeCreateDto> calculationTimes) {
    //验证
    validateCreate(reconciliationTemplateCode, calculationTimes);
    //保存
    List<ReconciliationLetterCalculationTime> reconciliationLetterCalculationTimes =
        (List<ReconciliationLetterCalculationTime>) nebulaToolkitService.copyCollectionByWhiteList(calculationTimes, ReconciliationLetterCalculationTimeCreateDto.class
            , ReconciliationLetterCalculationTime.class, HashSet.class, ArrayList.class);
    save(reconciliationTemplateCode, reconciliationLetterCalculationTimes);
    //新增定时任务
    createTask(reconciliationLetterCalculationTimes);
    return (List<ReconciliationLetterCalculationTimeVo>) nebulaToolkitService.copyCollectionByWhiteList(
        reconciliationLetterCalculationTimes,
        ReconciliationLetterCalculationTime.class,
        ReconciliationLetterCalculationTimeVo.class,
        HashSet.class, ArrayList.class);
  }

  private void validateCreate(String reconciliationTemplateCode, List<ReconciliationLetterCalculationTimeCreateDto> calculationTimes) {
    Validate.notBlank(reconciliationTemplateCode, "对账规则编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(calculationTimes), "新增对账规则计算时间数据不能为空");
    for (ReconciliationLetterCalculationTimeCreateDto calculationTime : calculationTimes) {
      Validate.notNull(calculationTime.getCalculationEndTime(), "对账计算结束时间不能为空");
      Validate.notNull(calculationTime.getCalculationStartTime(), "对账计算开始时间不能为空");
      Validate.notNull(calculationTime.getCalculationTime(), "对账计算时间不能为空");
    }
  }

  @Override
  @Transactional
  public List<ReconciliationLetterCalculationTimeVo> update(String reconciliationTemplateCode,
                                                            List<ReconciliationLetterCalculationTimeUpdateDto> calculationTimes) {
    //验证
    validateUpdate(reconciliationTemplateCode, calculationTimes);
    //删除
    delete(reconciliationTemplateCode);
    //新增数据
    List<ReconciliationLetterCalculationTime> reconciliationLetterCalculationTimes =
        (List<ReconciliationLetterCalculationTime>) nebulaToolkitService.copyCollectionByWhiteList(calculationTimes, ReconciliationLetterCalculationTimeUpdateDto.class
            , ReconciliationLetterCalculationTime.class, HashSet.class, ArrayList.class);
    save(reconciliationTemplateCode, reconciliationLetterCalculationTimes);
    //新增定时任务
    createTask(reconciliationLetterCalculationTimes);
    return (List<ReconciliationLetterCalculationTimeVo>) nebulaToolkitService.copyCollectionByWhiteList(
        reconciliationLetterCalculationTimes,
        ReconciliationLetterCalculationTime.class,
        ReconciliationLetterCalculationTimeVo.class,
        HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void invalidBatchByTemplateCodes(List<String> templateCodes) {
    List<ReconciliationLetterCalculationTime> calculationTimes =
        reconciliationLetterCalculationTimeRepository.findListByTemplateCodes(templateCodes, TenantUtils.getTenantCode());
    Validate.notEmpty(calculationTimes, "无法找到对应的计算时间");
    String[] taskCodes =
        calculationTimes.stream().map(ReconciliationLetterCalculationTime::getTaskCode).toArray(String[]::new);
    dynamicTaskSchedulerVoService.invalid(taskCodes);
  }

  @Override
  @Transactional
  public void effectiveBatchByTemplateCodes(List<String> templateCodes) {
    List<ReconciliationLetterCalculationTime> calculationTimes =
        reconciliationLetterCalculationTimeRepository.findListByTemplateCodes(templateCodes, TenantUtils.getTenantCode());
    Validate.notEmpty(calculationTimes, "无法找到对应的计算时间");
    String[] taskCodes =
        calculationTimes.stream().map(ReconciliationLetterCalculationTime::getTaskCode).toArray(String[]::new);
    dynamicTaskSchedulerVoService.effective(taskCodes);
  }

  @Override
  @Transactional
  public void deleteBatchByTemplateCodes(List<String> templateCodes) {
    List<ReconciliationLetterCalculationTime> calculationTimes =
        reconciliationLetterCalculationTimeRepository.findListByTemplateCodes(templateCodes, TenantUtils.getTenantCode());
    Validate.notEmpty(calculationTimes, "无法找到对应的计算时间");
    String[] taskCodes =
        calculationTimes.stream().map(ReconciliationLetterCalculationTime::getTaskCode).toArray(String[]::new);
    dynamicTaskSchedulerVoService.invalid(taskCodes);
    dynamicTaskSchedulerVoService.deleteByTaskcodes(taskCodes);
  }

  /**
   * 新增定时任务
   *
   * @param reconciliationLetterCalculationTimes
   * @return
   */
  private void createTask(List<ReconciliationLetterCalculationTime> reconciliationLetterCalculationTimes) {
    List<ReconciliationCalculationTaskCreateDto> reconciliationCalculationTaskCreateDtos =
        (List<ReconciliationCalculationTaskCreateDto>) nebulaToolkitService.copyCollectionByWhiteList(
            reconciliationLetterCalculationTimes,
            ReconciliationLetterCalculationTime.class,
            ReconciliationCalculationTaskCreateDto.class,
            HashSet.class, ArrayList.class);
    for (ReconciliationCalculationTaskCreateDto reconciliationCalculationTaskCreateDto : reconciliationCalculationTaskCreateDtos) {
      if(reconciliationCalculationTaskCreateDto.getCalculationTime().getTime() < new Date().getTime() ){
        ReconciliationLetterManualCreateDto reconciliationLetterManualCreateDto = new ReconciliationLetterManualCreateDto();
        reconciliationLetterManualCreateDto.setReconciliationTemplateCode(reconciliationCalculationTaskCreateDto.getReconciliationTemplateCode());
        reconciliationLetterManualCreateDto.setCalculationTimeId(reconciliationCalculationTaskCreateDto.getId());
        //根据选择的对账规则和对应的计算时间手动生成生效范围对应的所有客户的对账函
        reconciliationLetterVoService.handleManualGenerate(reconciliationLetterManualCreateDto);
        continue;
      }
      dynamicTaskSchedulerVoService.createIgnorePrefix(
          reconciliationCalculationTaskCreateDto.getTaskCode(),
          TASK_INVOKE_BEAN_NAME,
          TASK_METHOD,
          1,
          null,
          reconciliationCalculationTaskCreateDto.getCalculationTime(),
          null,
          TASK_DESC,
          reconciliationCalculationTaskCreateDto);
    }
  }

  /**
   * 保存计算时间
   *
   * @param reconciliationTemplateCode
   * @param reconciliationLetterCalculationTimes
   */
  private void save(String reconciliationTemplateCode, List<ReconciliationLetterCalculationTime> reconciliationLetterCalculationTimes) {
    for (ReconciliationLetterCalculationTime reconciliationLetterCalculationTime : reconciliationLetterCalculationTimes) {
      reconciliationLetterCalculationTime.setTenantCode(TenantUtils.getTenantCode());
      reconciliationLetterCalculationTime.setReconciliationTemplateCode(reconciliationTemplateCode);
      String taskCode = String.format(TASK_CODE_FORMAT,
          reconciliationTemplateCode,
          TenantUtils.getTenantCode(),
          DateFormatUtils.format(reconciliationLetterCalculationTime.getCalculationTime(), "yyyyMMddHHmmss"));
      reconciliationLetterCalculationTime.setTaskCode(taskCode);
    }
    reconciliationLetterCalculationTimeRepository.saveBatch(reconciliationLetterCalculationTimes);
  }

  /**
   * 删除计算时间
   *
   * @param reconciliationTemplateCode
   */
  private void delete(String reconciliationTemplateCode) {
    //删除定时任务
    List<ReconciliationLetterCalculationTime> reconciliationLetterCalculationTimes =
        reconciliationLetterCalculationTimeRepository.findListByTemplateCode(
            reconciliationTemplateCode, TenantUtils.getTenantCode());
    String[] taskCodes = reconciliationLetterCalculationTimes
        .stream().map(ReconciliationLetterCalculationTime::getTaskCode).toArray(String[]::new);
    dynamicTaskSchedulerVoService.invalid(taskCodes);
    dynamicTaskSchedulerVoService.deleteByTaskcodes(taskCodes);
    //删除数据
    reconciliationLetterCalculationTimeRepository.deleteByTemplateCode(reconciliationTemplateCode, TenantUtils.getTenantCode());
  }

  private void validateUpdate(String reconciliationTemplateCode, List<ReconciliationLetterCalculationTimeUpdateDto> calculationTimes) {
    Validate.notBlank(reconciliationTemplateCode, "对账规则编码不能为空");
    List<ReconciliationLetterCalculationTime> reconciliationLetterCalculationTimes =
        reconciliationLetterCalculationTimeRepository.findListByTemplateCode(
            reconciliationTemplateCode, TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isNotEmpty(reconciliationLetterCalculationTimes), "无法找到对账规则对应的计算时间数据");
    Validate.isTrue(CollectionUtils.isNotEmpty(calculationTimes), "更新对账规则计算时间数据不能为空");
    for (ReconciliationLetterCalculationTimeUpdateDto calculationTime : calculationTimes) {
      Validate.notNull(calculationTime.getCalculationEndTime(), "对账计算结束时间不能为空");
      Validate.notNull(calculationTime.getCalculationStartTime(), "对账计算开始时间不能为空");
      Validate.notNull(calculationTime.getCalculationTime(), "对账计算时间不能为空");
    }
  }
}
