package com.biz.crm.dms.business.sale.goal.local.service.internal;

import com.biz.crm.dms.business.sale.goal.local.entity.SaleGoalElement;
import com.biz.crm.dms.business.sale.goal.local.repository.SaleGoalElementRepository;
import com.biz.crm.dms.business.sale.goal.local.utils.FieldHandleUtil;
import com.biz.crm.dms.business.sale.goal.enums.GoalType;
import com.biz.crm.dms.business.sale.goal.enums.TaskType;
import com.biz.crm.dms.business.sale.goal.local.entity.SaleGoalElementDetail;
import com.biz.crm.dms.business.sale.goal.local.repository.SaleGoalElementDetailRepository;
import com.biz.crm.dms.business.sale.goal.local.service.SaleGoalSeriesElementService;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSeriesDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSeriesElementDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 系列销量目标要素service
 * @author: rentao
 * @date: 2021/12/1 19:26
 */
@Service
@Slf4j
public class SaleGoalSeriesElementServiceImpl implements SaleGoalSeriesElementService {

  @Autowired(required = false) private SaleGoalElementRepository saleGoalElementRepository;

  @Autowired(required = false) private SaleGoalElementDetailRepository saleGoalElementDetailRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  /**
   * 合同编码查询合同内容
   *
   * @param contractCode
   * @author rentao
   * @date
   */
  @Override
  public SaleGoalSeriesElementDataVo findByContractCode(String contractCode) {
    if (StringUtils.isBlank(contractCode)) {
      return null;
    }
    SaleGoalElement saleGoalElement =
        this.saleGoalElementRepository.findBySaleGoalTypeAndContractCode(
            GoalType.PRODUCT_LEVEL.getKey(), contractCode);
    if (Objects.isNull(saleGoalElement)) {
      return null;
    }
    SaleGoalSeriesElementDataVo saleGoalSeriesElementDataVo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            saleGoalElement, SaleGoalSeriesElementDataVo.class, HashSet.class, ArrayList.class);
    List<SaleGoalElementDetail> saleGoalElementDetails =
        this.saleGoalElementDetailRepository.findBySaleGoalElementId(saleGoalElement.getId());
    if (CollectionUtils.isNotEmpty(saleGoalElementDetails)) {
      List<SaleGoalSeriesDataVo> saleGoalSeriesDataVos =
          (List<SaleGoalSeriesDataVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  saleGoalElementDetails,
                  SaleGoalElementDetail.class,
                  SaleGoalSeriesDataVo.class,
                  HashSet.class,
                  ArrayList.class);
      saleGoalSeriesElementDataVo.setDataVos(saleGoalSeriesDataVos);
    }
    return saleGoalSeriesElementDataVo;
  }


  /**
   * 要素内容新增
   *
   * @param contractCode
   * @param saleGoalSeriesElementDataVo
   * @param indexCode
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public SaleGoalSeriesElementDataVo createSaleGoalSeriesElement(
      String contractCode,
      SaleGoalSeriesElementDataVo saleGoalSeriesElementDataVo,
      Integer indexCode) {
    this.validateSaveOrUpdate(contractCode, saleGoalSeriesElementDataVo);
    SaleGoalElement saleGoalElement =
        this.nebulaToolkitService.copyObjectByWhiteList(
            saleGoalSeriesElementDataVo, SaleGoalElement.class, HashSet.class, ArrayList.class);
    List<SaleGoalSeriesDataVo> saleGoalSeriesDataVos =
        saleGoalSeriesElementDataVo.getDataVos();
    List<SaleGoalElementDetail> saleGoalElementDetails =
        (List<SaleGoalElementDetail>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                saleGoalSeriesDataVos,
                SaleGoalSeriesDataVo.class,
                SaleGoalElementDetail.class,
                HashSet.class,
                ArrayList.class);
    saleGoalElement.setTenantCode(TenantUtils.getTenantCode());
    saleGoalElement.setContractCode(contractCode);
    this.saleGoalElementRepository.save(saleGoalElement);
    saleGoalElementDetails.forEach(
        saleGoalElementDetail -> {
          saleGoalElementDetail.setSaleGoalElementId(saleGoalElement.getId());
        });
    this.saleGoalElementDetailRepository.saveBatch(saleGoalElementDetails);
    return saleGoalSeriesElementDataVo;
  }

  /**
   * 要素内容修改
   *
   * @param contractCode
   * @param saleGoalSeriesElementDataVo
   * @param indexCode
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public SaleGoalSeriesElementDataVo updateSaleGoalSeriesElement(
      String contractCode,
      SaleGoalSeriesElementDataVo saleGoalSeriesElementDataVo,
      Integer indexCode) {
    this.validateSaveOrUpdate(contractCode, saleGoalSeriesElementDataVo);
    SaleGoalElement saleGoalElementOld =
        this.saleGoalElementRepository.findBySaleGoalTypeAndContractCode(
            GoalType.PRODUCT_LEVEL.getKey(), contractCode);
    Validate.notNull(saleGoalElementOld, "合同系列销量目标数据不存在");
    this.saleGoalElementRepository.removeById(saleGoalElementOld.getId());
    this.saleGoalElementDetailRepository.deleteBySaleGoalElementId(saleGoalElementOld.getId());
    SaleGoalElement saleGoalElement =
        this.nebulaToolkitService.copyObjectByWhiteList(
            saleGoalSeriesElementDataVo, SaleGoalElement.class, HashSet.class, ArrayList.class);
    List<SaleGoalSeriesDataVo> saleGoalSeriesDataVos =
        saleGoalSeriesElementDataVo.getDataVos();
    List<SaleGoalElementDetail> saleGoalElementDetails =
        (List<SaleGoalElementDetail>) 
            this.nebulaToolkitService.copyCollectionByWhiteList(
                saleGoalSeriesDataVos,
                SaleGoalSeriesDataVo.class,
                SaleGoalElementDetail.class,
                HashSet.class,
                ArrayList.class);
    saleGoalElement.setTenantCode(TenantUtils.getTenantCode());
    saleGoalElement.setContractCode(contractCode);
    saleGoalElement.setId(null);
    this.saleGoalElementRepository.save(saleGoalElement);
    saleGoalElementDetails.forEach(
        saleGoalElementDetail -> {
          saleGoalElementDetail.setId(null);
          saleGoalElementDetail.setSaleGoalElementId(saleGoalElement.getId());
        });
    this.saleGoalElementDetailRepository.saveBatch(saleGoalElementDetails);
    return saleGoalSeriesElementDataVo;
  }

  /** 合同系列销量目标验证 */
  private void validateSaveOrUpdate(
      String contractCode, SaleGoalSeriesElementDataVo saleGoalSeriesElementDataVo) {
    // 校验入参
    Validate.notBlank(contractCode, "合同编码不能为空");
    Validate.notNull(saleGoalSeriesElementDataVo, "合同系列销量目标为空");
    Validate.notBlank(saleGoalSeriesElementDataVo.getCusCode(), "合同保存系列要素，经销商编码为空");
    Validate.notNull(saleGoalSeriesElementDataVo.getTaskType(), "合同保存系列要素，任务类型为空");
    Validate.notBlank(saleGoalSeriesElementDataVo.getSaleGoalName(), "合同保存系列要素，销量目标名称为空");
    Validate.notNull(saleGoalSeriesElementDataVo.getUnitType(), "合同保存系列要素，销量目标名称为空");
    saleGoalSeriesElementDataVo.setSaleGoalType(GoalType.PRODUCT_LEVEL.getKey());
    TaskType taskType = TaskType.getByKey(saleGoalSeriesElementDataVo.getTaskType());
    Validate.notNull(taskType, "任务类型不存在");
    List<SaleGoalSeriesDataVo> saleGoalSeriesDataVos =
        saleGoalSeriesElementDataVo.getDataVos();
    Validate.isTrue(CollectionUtils.isNotEmpty(saleGoalSeriesDataVos), "合同系列销量目标为空");
    Set<String> onlyKeys = new HashSet<>();
    saleGoalSeriesElementDataVo
        .getDataVos()
        .forEach(
            s -> {
              // 将空置设置为 0
              FieldHandleUtil.initDecimalZero(s);
              Validate.notNull(s.getTargetYear(), "目标年份为空");
              Validate.notNull(s.getProductLevelCode(), "产品系列不能为空");
              Validate.notNull(s.getProductLevelName(), "产品系列不能为空");
              String onlyKey =
                  saleGoalSeriesElementDataVo.getCusCode()
                      + ","
                      + s.getTargetYear()
                      + ","
                      + s.getProductLevelCode();
              s.setOnlyKey(onlyKey);
              s.setContractCode(contractCode);
              onlyKeys.add(onlyKey);
              BigDecimal yearSum = BigDecimal.ZERO;
              switch (taskType) {
                case YEARLY:
                  // 设置年度汇总
                  s.setYearlySum(s.getYearlyNum());
                  break;
                case QUARTER:
                  // 计算年度汇总
                  yearSum =
                      yearSum
                          .add(s.getFirstQuarterNum())
                          .add(s.getSecondQuarterNum())
                          .add(s.getThirdQuarterNum())
                          .add(s.getFourthQuarterNum());
                  s.setYearlySum(yearSum);
                  // 计算季度汇总
                  s.setFirstQuarterSum(s.getFirstQuarterNum());
                  s.setSecondQuarterSum(s.getSecondQuarterNum());
                  s.setThirdQuarterSum(s.getThirdQuarterNum());
                  s.setFourthQuarterSum(s.getFourthQuarterNum());
                  break;
                case MONTHLY:
                  // 第一季度汇总
                  BigDecimal firstQuarterSum =
                      s.getJanTargetNum().add(s.getFebTargetNum()).add(s.getMarTargetNum());
                  s.setFirstQuarterSum(firstQuarterSum);
                  // 第二季度胡总
                  BigDecimal secondQuarterSum =
                      s.getAprTargetNum().add(s.getMayTargetNum()).add(s.getJunTargetNum());
                  s.setSecondQuarterSum(secondQuarterSum);
                  // 第三季度汇总
                  BigDecimal thirdQuarterSum =
                      s.getJulTargetNum().add(s.getAugTargetNum()).add(s.getFebTargetNum());
                  s.setThirdQuarterSum(thirdQuarterSum);
                  // 第四季度汇总
                  BigDecimal fourthQuarterSum =
                      s.getOctTargetNum().add(s.getDecTargetNum()).add(s.getNovTargetNum());
                  s.setFourthQuarterSum(fourthQuarterSum);
                  // 年度汇总
                  yearSum =
                      firstQuarterSum
                          .add(secondQuarterSum)
                          .add(thirdQuarterSum)
                          .add(fourthQuarterSum);
                  s.setYearlySum(yearSum);
                  break;
                default:
                  break;
              }
              Validate.isTrue(s.getYearlySum().compareTo(BigDecimal.ZERO) != 0, "汇总不能空");
            });
    // 客户编码 + 目标年份 确定唯一
    Validate.isTrue(
        saleGoalSeriesElementDataVo.getDataVos().size() == onlyKeys.size(),
        "本次提交,系列销量目标存在重复数据");
    List<SaleGoalElementDetail> saleGoalElementDetails =
        this.saleGoalElementDetailRepository.findByOnlyKeys(onlyKeys);
    if (CollectionUtils.isNotEmpty(saleGoalElementDetails)) {
      List<SaleGoalElementDetail> elements =
          saleGoalElementDetails.stream()
              .filter(saleGoalElement -> !contractCode.equals(saleGoalElement.getContractCode()))
              .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(elements), "经销商合同中已存在该系列目标维度的数据");
    }
  }
}
