package com.biz.crm.dms.business.sale.goal.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.contract.sdk.dto.contract.ContractEventDto;
import com.biz.crm.dms.business.contract.sdk.enums.ContractCancellationState;
import com.biz.crm.dms.business.contract.sdk.event.contract.ContractManagementEventListener;
import com.biz.crm.dms.business.contract.sdk.service.ContractVoService;
import com.biz.crm.dms.business.contract.sdk.vo.contract.ContractVo;
import com.biz.crm.dms.business.sale.goal.local.utils.SaleGoalItemUtil;
import com.biz.crm.dms.business.sale.goal.local.utils.SaleGoalUtil;
import com.biz.crm.dms.business.sale.goal.dto.SaleGoalDetailDto;
import com.biz.crm.dms.business.sale.goal.dto.SaleGoalDto;
import com.biz.crm.dms.business.sale.goal.enums.GoalType;
import com.biz.crm.dms.business.sale.goal.enums.ObjectType;
import com.biz.crm.dms.business.sale.goal.event.SaleGoalEventListener;
import com.biz.crm.dms.business.sale.goal.local.entity.SaleGoal;
import com.biz.crm.dms.business.sale.goal.local.repository.SaleGoalRepository;
import com.biz.crm.dms.business.sale.goal.local.service.SaleGoalDetailService;
import com.biz.crm.dms.business.sale.goal.local.service.SaleGoalService;
import com.biz.crm.dms.business.sale.goal.vo.SaleGoalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 销量目标 service
 * @author: rentao
 * @date: 2021/11/30 11:06
 */
@Service
@Slf4j
public class SaleGoalServiceImpl implements SaleGoalService {

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false) private SaleGoalRepository saleGoalRepository;

  @Autowired(required = false) private SaleGoalDetailService saleGoalDetailService;

  @Autowired(required = false) private ContractVoService contractVoService;

  @Autowired(required = false)
  private List<SaleGoalEventListener> listeners;

  /**
   * 销量目标分页列表
   *
   * @param dto 请求参数dto
   * @param pageable 分页信息
   * @return Page<SaleGoal> 销量目标分页信息
   */
  @Override
  public Page<SaleGoal> findByConditions(Pageable pageable, SaleGoalDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new SaleGoalDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<SaleGoal> entityPage = this.saleGoalRepository.findByConditions(pageable, dto);
    Page<SaleGoal> pageResult =
        new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    List<SaleGoal> entities = entityPage.getRecords();
    if (CollectionUtils.isEmpty(entities)) {
      return pageResult;
    }
    pageResult.setRecords(entities);
    return pageResult;
  }

  /**
   * 创建销量目标
   *
   * @param saleGoalDtos 参数dto
   * @return 创建的销量目标
   */
  @Override
  @Transactional
  public List<SaleGoal> createBatch(List<SaleGoalDto> saleGoalDtos) {
    this.validateForAddBatch(saleGoalDtos);
    List<SaleGoalDetailDto> saleGoalItems = new ArrayList<>();
    List<SaleGoal> entities = new ArrayList<>(saleGoalDtos.size());
    for (SaleGoalDto dto : saleGoalDtos) {
      SaleGoalDetailDto goalItemVo = dto.getSaleGoalDetailDto();
      Validate.notNull(goalItemVo, "目标明细不能为空");
      SaleGoal entity = SaleGoalUtil.packageEntityForAdd(dto);
      entity.setTenantCode(TenantUtils.getTenantCode());
      entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      if(StringUtils.isBlank(entity.getEnableStatus())){
        entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      }
      entity.setDataState(ContractCancellationState.CANCELLATION.getKey());
      this.saleGoalRepository.save(entity);
      goalItemVo.setSaleGoalId(entity.getId());
      goalItemVo.setTaskType(entity.getTaskType());
      entities.add(entity);
      saleGoalItems.add(goalItemVo);
    }
    List<List<SaleGoalDetailDto>> detailList = Lists.partition(saleGoalItems, 500);
    detailList.forEach(
        saleGoalDetailDtos -> {
          this.saleGoalDetailService.createBatch(saleGoalDetailDtos);
        });
    return entities;
  }

  /**
   * 更新销量目标
   *
   * @param dto 参数dto
   * @return 更新的销量目标
   */
  @Override
  @Transactional
  public SaleGoal update(SaleGoalDto dto) {
    Validate.notBlank(dto.getId(), "请指定要编辑的销售目标!");
    SaleGoal entity = this.saleGoalRepository.getById(dto.getId());
    Validate.notNull(entity, "您要编辑的销售目标不存在或者已经被删除!");
    if (StringUtils.isNotBlank(entity.getContractCode())) {
      //没有合同编码则认为不予合同有关联
      ContractVo contractVo = contractVoService.findDetailsByCode(entity.getContractCode());
      Validate.isTrue(Objects.isNull(contractVo), "关联合同的销量目标不能进行编辑");
    }
    SaleGoal newSaleGoal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, SaleGoal.class, HashSet.class, ArrayList.class);
    SaleGoalDetailDto saleGoalDetailDto = dto.getSaleGoalDetailDto();
    saleGoalDetailDto.setTaskType(dto.getTaskType());
    newSaleGoal.setTargetNum(
        SaleGoalItemUtil.sumSingleItem(dto.getSaleGoalDetailDto()).getYearlySum());
    this.saleGoalRepository.updateById(newSaleGoal);
    this.saleGoalDetailService.update(saleGoalDetailDto);
    return entity;
  }

  /**
   * 批量启用
   *
   * @param contractCodes
   */
  @Override
  @Transactional
  public void enableBatchByContractCode(List<String> contractCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(contractCodes), "待修正的数据合同编码不能为空");
    List<SaleGoal> entities = this.saleGoalRepository.findByContractCodeAndDelFlag(contractCodes,DelFlagStatusEnum.NORMAL.getCode());
    if(CollectionUtils.isNotEmpty(entities)){
      List<String> ids = entities.stream().map(SaleGoal::getId).collect(Collectors.toList());
      this.saleGoalRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    }
  }

  /**
   * 批量禁用
   *
   * @param contractCodes
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void disableBatchByContractCode(List<String> contractCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(contractCodes), "待修正的数据合同编码不能为空");
    List<SaleGoal> entities = this.saleGoalRepository.findByContractCodeAndDelFlag(contractCodes,DelFlagStatusEnum.NORMAL.getCode());
    if(CollectionUtils.isNotEmpty(entities)){
      List<String> ids = entities.stream().map(SaleGoal::getId).collect(Collectors.toList());
      this.saleGoalRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    }
  }

  /**
   * 批量删除
   *
   * @param ids
   */
  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<SaleGoal> entities = this.saleGoalRepository.listByIds(ids);
    Validate.isTrue(
        CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    List<String>  contractCodes =
        entities.stream().filter(saleGoal -> StringUtils.isNotBlank(saleGoal.getContractCode())).map(SaleGoal::getContractCode).collect(Collectors.toList());
    List<ContractVo> contractVos = contractVoService.findByContractCodes(contractCodes);
    Validate.isTrue(CollectionUtils.isEmpty(contractVos), "存在关联合同的销量目标不能删除");
    this.saleGoalRepository.updateDelFlagByIds(ids);
    // 删除销量目标事件通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      List<SaleGoalVo> voList =
          (List<SaleGoalVo>)
              nebulaToolkitService.copyCollectionByWhiteList(
                  entities, SaleGoal.class, SaleGoalVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(
          listener -> {
            listener.onDelete(voList);
          });
    }
  }

  @Override
  @Transactional
  public void deleteBatchByContractCodes(List<String> contractCodes) {
    Validate.isTrue(CollectionUtils.isNotEmpty(contractCodes), "缺失合同编码");
    List<SaleGoal> entities = this.saleGoalRepository.findByContractCodeAndDelFlag(contractCodes, DelFlagStatusEnum.NORMAL.getCode());
    if (CollectionUtils.isNotEmpty(entities)) {
      List<String> ids = entities.stream().map(SaleGoal::getId).collect(Collectors.toList());
      this.saleGoalRepository.updateDelFlagByIds(ids);
      // 删除销量目标事件通知
      if (CollectionUtils.isNotEmpty(listeners)) {
        List<SaleGoalVo> voList =
            (List<SaleGoalVo>)
                this.nebulaToolkitService.copyCollectionByWhiteList(
                    entities, SaleGoal.class, SaleGoalVo.class, HashSet.class, ArrayList.class);
        listeners.forEach(
            listener -> {
              listener.onDelete(voList);
            });
      }
    }
  }

  /**
   * AddBatch的验证方法
   *
   * @param saleGoalDtos
   */
  private void validateForAddBatch(List<SaleGoalDto> saleGoalDtos) {
    // 验证参数
    SaleGoalUtil.validateParamForAddBatch(saleGoalDtos);
    // 设置onlyKey值并验证是否数据库存在
    SaleGoalUtil.buildOnlyKey(saleGoalDtos);
    Integer saleGoalType = saleGoalDtos.get(0).getSaleGoalType();
    Integer objectType = saleGoalDtos.get(0).getObjectType();
    List<String> onlyKeys =
        saleGoalDtos.stream().map(SaleGoalDto::getOnlyKey).collect(Collectors.toList());
    // 分批次查询
    List<SaleGoal> saleGoalList = new ArrayList<>();
    List<List<String>> currentOnlyKey = Lists.partition(onlyKeys, 500);
    currentOnlyKey.forEach(
        onlyKeyList -> {
          List<SaleGoal> saleGoals = this.saleGoalRepository.findByOnlyKeysAndDelFlag(onlyKeyList, DelFlagStatusEnum.NORMAL.getCode());
          if (CollectionUtils.isNotEmpty(saleGoals)) {
            saleGoalList.addAll(saleGoals);
          }
        });
    // 异常提示
    StringBuilder msg = new StringBuilder("销量目标已经存在,年份为：");
    StringBuilder errorMsg = new StringBuilder();
    if (CollectionUtils.isNotEmpty(saleGoalList)) {
      for (SaleGoal entity : saleGoalList) {
        if (ObjectType.CUSTOMER.getKey().equals(objectType)) {
          errorMsg.append(entity.getCusName()).append("-");
        } else if (ObjectType.ORGANIZATION.getKey().equals(objectType)) {
          errorMsg.append(entity.getOrgName()).append("-");
        } else if (ObjectType.TERMINAL.getKey().equals(objectType)) {
          errorMsg.append(entity.getTerminalName()).append("-");
        }
        if (saleGoalType.intValue() == GoalType.GOODS.getKey().intValue()) {
          errorMsg
              .append(entity.getProductName())
              .append("-")
              .append(entity.getTargetYear())
              .append(";");
        } else if (saleGoalType.intValue() == GoalType.PRODUCT_LEVEL.getKey().intValue()) {
          errorMsg
              .append(entity.getProductLevelName())
              .append("-")
              .append(entity.getTargetYear())
              .append(";");
        } else {
          errorMsg.append(entity.getTargetYear()).append(";");
        }
      }
      if (saleGoalType.intValue() != GoalType.ROUTINE.getKey().intValue()) {
        msg = new StringBuilder("销量目标已经存在,产品/产品层级和年份为: ");
      }
    }
    Validate.isTrue(CollectionUtils.isEmpty(saleGoalList), msg.append(errorMsg).toString());
  }
}
