package com.biz.crm.dms.business.sale.goal.local.service.internal;

import com.biz.crm.dms.business.sale.goal.local.entity.SaleGoalElement;
import com.biz.crm.dms.business.sale.goal.local.repository.SaleGoalElementRepository;
import com.biz.crm.dms.business.sale.goal.local.utils.FieldHandleUtil;
import com.biz.crm.dms.business.sale.goal.enums.GoalType;
import com.biz.crm.dms.business.sale.goal.enums.TaskType;
import com.biz.crm.dms.business.sale.goal.local.entity.SaleGoalElementDetail;
import com.biz.crm.dms.business.sale.goal.local.repository.SaleGoalElementDetailRepository;
import com.biz.crm.dms.business.sale.goal.local.service.SaleGoalSingleElementService;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSingleDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSingleElementDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 单品销量目标要素实现
 * @author: rentao
 * @date: 2021/12/1 14:46
 */
@Service
@Slf4j
public class SaleGoalSingleElementServiceImpl implements SaleGoalSingleElementService {

  @Autowired(required = false) private SaleGoalElementRepository saleGoalElementRepository;

  @Autowired(required = false) private SaleGoalElementDetailRepository saleGoalElementDetailRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  /**
   * 合同编码查询合同内容
   *
   * @param contractcode
   * @author rentao
   * @date
   */
  @Override
  public SaleGoalSingleElementDataVo findByContractCode(String contractCode) {
    if (StringUtils.isBlank(contractCode)) {
      return null;
    }
    SaleGoalElement saleGoalElement =
        this.saleGoalElementRepository.findBySaleGoalTypeAndContractCode(
            GoalType.GOODS.getKey(), contractCode);
    if (Objects.isNull(saleGoalElement)) {
      return null;
    }
    SaleGoalSingleElementDataVo saleGoalSingleElementDataVo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            saleGoalElement, SaleGoalSingleElementDataVo.class, HashSet.class, ArrayList.class);
    List<SaleGoalElementDetail> saleGoalElementDetails =
        this.saleGoalElementDetailRepository.findBySaleGoalElementId(saleGoalElement.getId());
    if (CollectionUtils.isNotEmpty(saleGoalElementDetails)) {
      List<SaleGoalSingleDataVo> saleGoalSeriesDataVos =
          (List<SaleGoalSingleDataVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  saleGoalElementDetails,
                  SaleGoalElementDetail.class,
                  SaleGoalSingleDataVo.class,
                  HashSet.class,
                  ArrayList.class);
      saleGoalSingleElementDataVo.setDataVos(saleGoalSeriesDataVos);
    }
    return saleGoalSingleElementDataVo;
  }

  /**
   * 要素内容新增
   *
   * @param contractCode
   * @param allowSaleElementDataVo
   * @param indexCode
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public SaleGoalSingleElementDataVo createSaleGoalSingleElement(
      String contractCode,
      SaleGoalSingleElementDataVo saleGoalSingleElementDataVo,
      Integer indexCode) {
    this.validateSaveOrUpdate(contractCode, saleGoalSingleElementDataVo);
    SaleGoalElement saleGoalElement =
        this.nebulaToolkitService.copyObjectByWhiteList(
            saleGoalSingleElementDataVo, SaleGoalElement.class, HashSet.class, ArrayList.class);
    List<SaleGoalSingleDataVo> saleGoalSingleDataVos =
        saleGoalSingleElementDataVo.getDataVos();
    List<SaleGoalElementDetail> saleGoalElementDetails =
        (List<SaleGoalElementDetail>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                saleGoalSingleDataVos,
                SaleGoalSingleDataVo.class,
                SaleGoalElementDetail.class,
                HashSet.class,
                ArrayList.class);
    saleGoalElement.setTenantCode(TenantUtils.getTenantCode());
    saleGoalElement.setContractCode(contractCode);
    this.saleGoalElementRepository.save(saleGoalElement);
    saleGoalElementDetails.forEach(
        saleGoalElementDetail -> {
          saleGoalElementDetail.setSaleGoalElementId(saleGoalElement.getId());
        });
    this.saleGoalElementDetailRepository.saveBatch(saleGoalElementDetails);
    return saleGoalSingleElementDataVo;
  }

  /**
   * 要素内容修改
   *
   * @param contractCode
   * @param saleGoalSingleElementDataVo
   * @param indexCode
   * @author rentao
   * @date
   */
  @Override
  @Transactional
  public SaleGoalSingleElementDataVo updateSaleGoalSingleElement(
      String contractCode,
      SaleGoalSingleElementDataVo saleGoalSingleElementDataVo,
      Integer indexCode) {
    this.validateSaveOrUpdate(contractCode, saleGoalSingleElementDataVo);
    SaleGoalElement saleGoalElementOld =
        this.saleGoalElementRepository.findBySaleGoalTypeAndContractCode(
            GoalType.GOODS.getKey(), contractCode);
    Validate.notNull(saleGoalElementOld, "合同系列销量目标数据不存在");
    this.saleGoalElementRepository.removeById(saleGoalElementOld.getId());
    this.saleGoalElementDetailRepository.deleteBySaleGoalElementId(saleGoalElementOld.getId());
    SaleGoalElement saleGoalElement =
        this.nebulaToolkitService.copyObjectByWhiteList(
            saleGoalSingleElementDataVo, SaleGoalElement.class, HashSet.class, ArrayList.class);
    List<SaleGoalSingleDataVo> saleGoalSingleDataVos =
        saleGoalSingleElementDataVo.getDataVos();
    List<SaleGoalElementDetail> saleGoalElementDetails =
        (List<SaleGoalElementDetail>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                saleGoalSingleDataVos,
                SaleGoalSingleDataVo.class,
                SaleGoalElementDetail.class,
                HashSet.class,
                ArrayList.class);
    saleGoalElement.setTenantCode(TenantUtils.getTenantCode());
    saleGoalElement.setContractCode(contractCode);
    saleGoalElement.setId(null);
    this.saleGoalElementRepository.save(saleGoalElement);
    saleGoalElementDetails.forEach(
        saleGoalElementDetail -> {
          saleGoalElementDetail.setId(null);
          saleGoalElementDetail.setSaleGoalElementId(saleGoalElement.getId());
        });
    this.saleGoalElementDetailRepository.saveBatch(saleGoalElementDetails);
    return saleGoalSingleElementDataVo;
  }

  /** 合同单品销量目标验证 */
  private void validateSaveOrUpdate(
      String contractCode, SaleGoalSingleElementDataVo saleGoalSingleElementDataVo) {
    // 校验入参
    Validate.notBlank(contractCode, "合同编码不能为空");
    Validate.notNull(saleGoalSingleElementDataVo, "合同单品销量目标为空");
    Validate.notBlank(saleGoalSingleElementDataVo.getCusCode(), "合同保存单品要素，经销商编码为空");
    Validate.notNull(saleGoalSingleElementDataVo.getTaskType(), "合同保存单品要素，任务类型为空");
    Validate.notBlank(saleGoalSingleElementDataVo.getSaleGoalName(), "合同保存单品要素，销量目标名称为空");
    Validate.notNull(saleGoalSingleElementDataVo.getUnitType(), "合同保存单品要素，单位类型为空");
    saleGoalSingleElementDataVo.setSaleGoalType(GoalType.GOODS.getKey());
    TaskType taskType = TaskType.getByKey(saleGoalSingleElementDataVo.getTaskType());
    Validate.notNull(taskType, "任务类型不存在");
    List<SaleGoalSingleDataVo> saleGoalSingleDataVos =
        saleGoalSingleElementDataVo.getDataVos();
    Validate.isTrue(CollectionUtils.isNotEmpty(saleGoalSingleDataVos), "合同单品销量目标为空");
    Set<String> onlyKeys = new HashSet<>();
    saleGoalSingleElementDataVo
        .getDataVos()
        .forEach(
            s -> {
              // 将空置设置为 0
              FieldHandleUtil.initDecimalZero(s);
              Validate.notNull(s.getTargetYear(), "目标年份为空");
              Validate.notBlank(s.getProductCode(), "单品产品编码不能为空");
              Validate.notBlank(s.getProductName(), "单品产品名称不能为空");
              String onlyKey =
                  saleGoalSingleElementDataVo.getCusCode()
                      + ","
                      + s.getTargetYear()
                      + ","
                      + s.getProductCode();
              s.setOnlyKey(onlyKey);
              s.setContractCode(contractCode);
              onlyKeys.add(onlyKey);
              BigDecimal yearSum = BigDecimal.ZERO;
              switch (taskType) {
                case YEARLY:
                  s.setYearlySum(s.getYearlyNum());
                  break;
                case QUARTER:
                  yearSum =
                      yearSum
                          .add(s.getFirstQuarterNum())
                          .add(s.getSecondQuarterNum())
                          .add(s.getThirdQuarterNum())
                          .add(s.getFourthQuarterNum());
                  s.setYearlySum(yearSum);
                  // 计算季度汇总
                  s.setFirstQuarterSum(s.getFirstQuarterNum());
                  s.setSecondQuarterSum(s.getSecondQuarterNum());
                  s.setThirdQuarterSum(s.getThirdQuarterNum());
                  s.setFourthQuarterSum(s.getFourthQuarterNum());
                  break;
                case MONTHLY:
                  // 第一季度汇总
                  BigDecimal firstQuarterSum =
                      s.getJanTargetNum().add(s.getFebTargetNum()).add(s.getMarTargetNum());
                  s.setFirstQuarterSum(firstQuarterSum);
                  // 第二季度汇总
                  BigDecimal secondQuarterSum =
                      s.getAprTargetNum().add(s.getMayTargetNum()).add(s.getJunTargetNum());
                  s.setSecondQuarterSum(secondQuarterSum);
                  // 第三季度汇总
                  BigDecimal thirdQuarterSum =
                      s.getJulTargetNum().add(s.getAugTargetNum()).add(s.getFebTargetNum());
                  s.setThirdQuarterSum(thirdQuarterSum);
                  // 第四季度汇总
                  BigDecimal fourthQuarterSum =
                      s.getOctTargetNum().add(s.getDecTargetNum()).add(s.getNovTargetNum());
                  s.setFourthQuarterSum(fourthQuarterSum);
                  // 年度汇总
                  yearSum =
                      firstQuarterSum
                          .add(secondQuarterSum)
                          .add(thirdQuarterSum)
                          .add(fourthQuarterSum);
                  s.setYearlySum(yearSum);
                  break;
                default:
                  break;
              }
              Validate.isTrue(s.getYearlySum().compareTo(BigDecimal.ZERO) != 0, "汇总不能空");
            });
    // 客户编码 + 目标年份 确定唯一
    Validate.isTrue(
        saleGoalSingleElementDataVo.getDataVos().size() == onlyKeys.size(),
        "本次提交,单品销量目标存在重复数据");
    List<SaleGoalElementDetail> saleGoalElementDetails =
        this.saleGoalElementDetailRepository.findByOnlyKeys(onlyKeys);
    if (CollectionUtils.isNotEmpty(saleGoalElementDetails)) {
      List<SaleGoalElementDetail> elements =
          saleGoalElementDetails.stream()
              .filter(saleGoalElement -> !contractCode.equals(saleGoalElement.getContractCode()))
              .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(elements), "经销商合同中已存在该单品目标维度的数据");
    }
  }
}
