package com.biz.crm.dms.business.sale.goal.local.service.notifier;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.contract.sdk.enums.ActApproveStatusExtendEnum;
import com.biz.crm.dms.business.contract.sdk.enums.ContractCancellationState;
import com.biz.crm.dms.business.contract.sdk.enums.EffectiveStatusEnum;
import com.biz.crm.dms.business.contract.sdk.event.contract.ContractEventListener;
import com.biz.crm.dms.business.contract.sdk.event.contractsign.ContractSignEventListener;
import com.biz.crm.dms.business.contract.sdk.vo.contract.ContractVo;
import com.biz.crm.dms.business.sale.goal.dto.SaleGoalDetailDto;
import com.biz.crm.dms.business.sale.goal.dto.SaleGoalDto;
import com.biz.crm.dms.business.sale.goal.enums.GoalType;
import com.biz.crm.dms.business.sale.goal.enums.ObjectType;
import com.biz.crm.dms.business.sale.goal.local.service.SaleGoalService;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalRoutineDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalRoutineElementDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSeriesDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSeriesElementDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSingleDataVo;
import com.biz.crm.dms.business.sale.goal.vo.element.SaleGoalSingleElementDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 合同通知类接口 销量目标实现
 * @author: rentao
 * @date: 2021/12/2 15:21
 */

public class SaleGoalContractEventListenerImpl implements ContractEventListener, ContractSignEventListener {

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private SaleGoalService saleGoalService;

  /**
   * 常规销量目标
   */
  private static final String SALE_GOAL_ROUTINE_ELEMENT = "saleGoalRoutineElementRegister";

  /**
   * 系列销量目标
   */
  private static final String SALE_GOAL_SERIES_ELEMENT = "saleGoalSeriesElementRegister";

  /**
   * 单品销量目标
   */
  private static final String SALE_GOAL_SINGLE_ELEMENT = "saleGoalSingleElementRegister";

  /**
   * 启用时： 合同状态（contractStatus） 已签约 是否作废（cancellationState） 否 delFlag 009 生效状态（effectiveStatus）
   * EFFECTIVE_ACTIVE 生效中 才可以生成销量目标
   *
   * @param list
   * @author rentao
   * @date
   */
  @Override
  @Transactional 
  public void onEnable(List<ContractVo> list) {
    if (CollectionUtils.isNotEmpty(list)) {
      List<String> contractCodes = new ArrayList<>();
      for (ContractVo vo : list) {
        if (EffectiveStatusEnum.EFFECTIVE_ACTIVE.getDictCode().equals(vo.getEffectiveStatus())
            && DelFlagStatusEnum.NORMAL.getCode().equals(vo.getDelFlag())
            && ContractCancellationState.CANCELLATION.getDictCode()
            .equals(vo.getCancellationState())
            && ActApproveStatusExtendEnum.SIGN_CONTRACT.getDictCode()
            .equals(vo.getContractStatus())) {
          contractCodes.add(vo.getContractCode());
        }
      }
      if(CollectionUtils.isNotEmpty(contractCodes)){
        this.saleGoalService.enableBatchByContractCode(contractCodes);
      }
    }
  }

  /**
   * 禁用时：直接禁用
   *
   * @param list
   * @author rentao
   * @date
   */
  @Override
  @Transactional 
  public void onDisable(List<ContractVo> list) {
    Validate.isTrue(CollectionUtils.isNotEmpty(list), "合同数据为空");
    List<String> contractCodes = list.stream().map(ContractVo::getContractCode)
        .collect(Collectors.toList());
    this.saleGoalService.disableBatchByContractCode(contractCodes);
  }


  /**
   * 合同签约事件通知（常规，系列，单品三种 ） 签约后 生效状态的合同 生成销量目标
   *
   * @param vo 合同数据
   */
  @Override
  @Transactional 
  public void onSignContract(ContractVo vo) {
    Validate.notNull(vo, "合同数据为空");
    Validate.notNull(vo.getContractCode(), "合同编码为空");
    Validate.notNull(vo.getElementDataMap(), "合同要素数据为空");
    List<SaleGoalDto> saleGoalDtos = this.transformContractVoToSaleGoalDto(vo);
    if (CollectionUtils.isNotEmpty(saleGoalDtos)) {
      Map<Integer, List<SaleGoalDto>> collect = saleGoalDtos.stream()
          .collect(Collectors.groupingBy(SaleGoalDto::getSaleGoalType));
      collect.forEach((k, v) -> {
        Map<Integer, List<SaleGoalDto>> collect1 = v.stream().collect(Collectors.groupingBy(SaleGoalDto::getObjectType));
        collect1.forEach((key, value) -> {
          this.saleGoalService.createBatch(value);
        });
      });
    }
  }

  /**
   * 合同作废 （删除销量目标）
   *
   * @param vo
   */
  @Override
  @Transactional
  public void onCancellation(ContractVo vo) {
    Validate.notNull(vo, "合同数据为空");
    this.saleGoalService.deleteBatchByContractCodes(Collections.singletonList(vo.getContractCode()));
  }

  /**
   * 合同到期（删除销量目标）
   *
   * @param list
   */
  @Override
  @Transactional
  public void onMaturity(List<ContractVo> list) {
    Validate.isTrue(CollectionUtils.isNotEmpty(list), "合同数据为空");
    List<String> collect = list.stream().map(ContractVo::getContractCode)
        .collect(Collectors.toList());
    this.saleGoalService.deleteBatchByContractCodes(collect);
  }

  /**
   * 合同生效（生成销量目标） 启用销量目标
   *
   * @param list
   */
  @Override
  @Transactional
  public void onEffective(List<ContractVo> list) {
    Validate.isTrue(CollectionUtils.isNotEmpty(list), "合同数据为空");
    List<String> contractCodes = list.stream().map(ContractVo::getContractCode)
        .collect(Collectors.toList());
    this.saleGoalService.enableBatchByContractCode(contractCodes);
  }

  /**
   * 合同转换销量目标要素
   *
   * @param vo
   * @author rentao
   * @date
   */
  public List<SaleGoalDto> transformContractVoToSaleGoalDto(ContractVo vo) {
    Map<String, JSONObject> elementDataMap = vo.getElementDataMap();
    JSONObject objRoutineElement = elementDataMap.get(SALE_GOAL_ROUTINE_ELEMENT);
    JSONObject objSeriesElement = elementDataMap.get(SALE_GOAL_SERIES_ELEMENT);
    JSONObject objSingleElement = elementDataMap.get(SALE_GOAL_SINGLE_ELEMENT);
    List<SaleGoalDto> saleGoalDtos = new ArrayList<>();
    if (Objects.nonNull(objRoutineElement)) {
      SaleGoalRoutineElementDataVo saleGoalRoutineElementDataVo =
          JSON.parseObject(objRoutineElement.toJSONString(), SaleGoalRoutineElementDataVo.class);
      this.transformContractSaleGoalRoutineVoToSaleGoalDto(
          vo, saleGoalRoutineElementDataVo, saleGoalDtos);
    }
    if (Objects.nonNull(objSeriesElement)) {
      SaleGoalSeriesElementDataVo saleGoalSeriesElementDataVo =
          JSON.parseObject(objSeriesElement.toJSONString(), SaleGoalSeriesElementDataVo.class);
      this.transformContractSaleGoalSeriesVoToSaleGoalDto(
          vo, saleGoalSeriesElementDataVo, saleGoalDtos);
    }
    if (Objects.nonNull(objSingleElement)) {
      SaleGoalSingleElementDataVo saleGoalSingleElementDataVo =
          JSON.parseObject(objSingleElement.toJSONString(), SaleGoalSingleElementDataVo.class);
      this.transformContractSaleGoalSingVoToSaleGoalDto(
          vo, saleGoalSingleElementDataVo, saleGoalDtos);
    }
    return saleGoalDtos;
  }

  /**
   * 常规销量目标转换 saleGoalDto
   *
   * @param vo
   * @param saleGoalRoutineElementDataVo
   * @param saleGoalDtos
   * @author rentao
   * @date
   */
  private void transformContractSaleGoalRoutineVoToSaleGoalDto(
      ContractVo vo,
      SaleGoalRoutineElementDataVo saleGoalRoutineElementDataVo,
      List<SaleGoalDto> saleGoalDtos) {
    if (Objects.nonNull(saleGoalRoutineElementDataVo)) {
      List<SaleGoalRoutineDataVo> saleGoalRoutineDataVos =
          saleGoalRoutineElementDataVo.getDataVos();
      if (CollectionUtils.isNotEmpty(saleGoalRoutineDataVos)) {
        saleGoalRoutineDataVos.forEach(
            s -> {
              // 组装常规销量目标 dto
              SaleGoalDto saleGoalDto = this.transformBaseToSaleGoalDto(vo);
              SaleGoalDetailDto saleGoalDetailDto = this.nebulaToolkitService
                  .copyObjectByWhiteList(s, SaleGoalDetailDto.class, HashSet.class,
                      ArrayList.class);
              saleGoalDetailDto.setEnableStatus(saleGoalDto.getEnableStatus());
              saleGoalDto.setSaleGoalType(GoalType.ROUTINE.getKey());
              saleGoalDto.setSaleGoalName(saleGoalRoutineElementDataVo.getSaleGoalName());
              saleGoalDto.setTargetYear(Integer.valueOf(vo.getContractYear()));
              saleGoalDto.setObjectType(ObjectType.CUSTOMER.getKey());
              saleGoalDto.setTaskType(saleGoalRoutineElementDataVo.getTaskType());
              saleGoalDto.setSaleGoalDetailDto(saleGoalDetailDto);
              saleGoalDto.setUnitType(saleGoalRoutineElementDataVo.getUnitType());
              saleGoalDtos.add(saleGoalDto);
            });
      }
    }
  }

  /**
   * 系列目标转换 saleGoalDto
   *
   * @param vo
   * @param saleGoalSeriesElementDataVo
   * @param saleGoalDtos
   * @author rentao
   * @date
   */
  private void transformContractSaleGoalSeriesVoToSaleGoalDto(
      ContractVo vo,
      SaleGoalSeriesElementDataVo saleGoalSeriesElementDataVo,
      List<SaleGoalDto> saleGoalDtos) {
    if (Objects.nonNull(saleGoalSeriesElementDataVo)) {
      List<SaleGoalSeriesDataVo> saleGoalRoutineDataVos =
          saleGoalSeriesElementDataVo.getDataVos();
      if (CollectionUtils.isNotEmpty(saleGoalRoutineDataVos)) {
        saleGoalRoutineDataVos.forEach(
            s -> {
              // 组装常规销量目标 dto
              SaleGoalDto saleGoalDto = this.transformBaseToSaleGoalDto(vo);
              SaleGoalDetailDto saleGoalDetailDto =
                  this.nebulaToolkitService.copyObjectByWhiteList(
                      s, SaleGoalDetailDto.class, HashSet.class, ArrayList.class);
              saleGoalDetailDto.setEnableStatus(saleGoalDto.getEnableStatus());
              saleGoalDto.setProductLevelCode(s.getProductLevelCode());
              saleGoalDto.setProductLevelName(s.getProductLevelName());
              saleGoalDto.setSaleGoalName(saleGoalSeriesElementDataVo.getSaleGoalName());
              saleGoalDto.setTaskType(saleGoalSeriesElementDataVo.getTaskType());
              saleGoalDto.setObjectType(ObjectType.CUSTOMER.getKey());
              saleGoalDto.setSaleGoalType(GoalType.PRODUCT_LEVEL.getKey());
              saleGoalDto.setSaleGoalDetailDto(saleGoalDetailDto);
              saleGoalDto.setUnitType(saleGoalSeriesElementDataVo.getUnitType());
              saleGoalDtos.add(saleGoalDto);
            });
      }
    }
  }

  /**
   * 单品目标转换 saleGoalDto
   *
   * @param vo
   * @param saleGoalSingleElementDataVo
   * @param saleGoalDtos
   * @author rentao
   * @date
   */
  private void transformContractSaleGoalSingVoToSaleGoalDto(
      ContractVo vo,
      SaleGoalSingleElementDataVo saleGoalSingleElementDataVo,
      List<SaleGoalDto> saleGoalDtos) {
    if (Objects.nonNull(saleGoalSingleElementDataVo)) {
      List<SaleGoalSingleDataVo> saleGoalRoutineDataVos =
          saleGoalSingleElementDataVo.getDataVos();
      if (CollectionUtils.isNotEmpty(saleGoalRoutineDataVos)) {
        saleGoalRoutineDataVos.forEach(
            s -> {
              // 组装常规销量目标 dto
              SaleGoalDto saleGoalDto = this.transformBaseToSaleGoalDto(vo);
              SaleGoalDetailDto saleGoalDetailDto =
                  this.nebulaToolkitService.copyObjectByWhiteList(
                      s, SaleGoalDetailDto.class, HashSet.class, ArrayList.class);
              saleGoalDetailDto.setEnableStatus(saleGoalDto.getEnableStatus());
              saleGoalDto.setProductCode(s.getProductCode());
              saleGoalDto.setProductName(s.getProductName());
              saleGoalDto.setProductSpec(s.getProductSpec());
              saleGoalDto.setSaleGoalType(GoalType.GOODS.getKey());
              saleGoalDto.setSaleGoalName(saleGoalSingleElementDataVo.getSaleGoalName());
              saleGoalDto.setTaskType(saleGoalSingleElementDataVo.getTaskType());
              saleGoalDto.setObjectType(ObjectType.CUSTOMER.getKey());
              saleGoalDto.setSaleGoalDetailDto(saleGoalDetailDto);
              saleGoalDto.setUnitType(saleGoalSingleElementDataVo.getUnitType());
              saleGoalDtos.add(saleGoalDto);
            });
      }
    }
  }


  /**
   * 基础数据组装
   *
   * @param vo 合同编码
   * @author rentao
   * @date
   */
  private SaleGoalDto transformBaseToSaleGoalDto(ContractVo vo) {
    SaleGoalDto saleGoalDto = new SaleGoalDto();
    saleGoalDto.setContractCode(vo.getContractCode());
    saleGoalDto.setCusCode(vo.getCusCode());
    saleGoalDto.setCusName(vo.getCusName());
    saleGoalDto.setCusOrgCode(vo.getOrgCode());
    saleGoalDto.setCusOrgName(vo.getOrgName());
    saleGoalDto.setTargetYearType(Integer.valueOf(vo.getYearType()));
    saleGoalDto.setCusChannelCode(vo.getChannelCode());
    saleGoalDto.setCusChannelName(vo.getChannelName());
    saleGoalDto.setTargetYear(Integer.valueOf(vo.getContractYear()));
    if (EffectiveStatusEnum.EFFECTIVE_ACTIVE.getDictCode().equals(vo.getEffectiveStatus())) {
      saleGoalDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    } else {
      saleGoalDto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
    }
    return saleGoalDto;
  }
}
