package com.biz.crm.dms.business.sale.goal.local.utils;

import com.biz.crm.dms.business.sale.goal.dto.SaleGoalDto;
import com.biz.crm.dms.business.sale.goal.enums.GoalType;
import com.biz.crm.dms.business.sale.goal.enums.ObjectType;
import com.biz.crm.dms.business.sale.goal.enums.TaskType;
import com.biz.crm.dms.business.sale.goal.local.entity.SaleGoal;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 销量目标 util
 * @author: rentao
 * @date: 2021/12/1 10:30
 */
public class SaleGoalUtil {

  //每个对象维度对应的唯一约束字段
  private static final Map<Integer, List<String>> OBJECT_TYPE_FILED_MAP = Maps.newHashMap();
  //每个销售类型对应的唯一约束字段
  private static final Map<Integer, List<String>> GOAL_TYPE_FILED_MAP = Maps.newHashMap();

  static {
    //定义一个存放各类型字段的列表，为了日后可拓展，所以这里不用字符串，而用列表
    List<String> fieldList;
    //每个对象维度对应的唯一约束字段
    fieldList = Lists.newArrayList("cusCode");
    OBJECT_TYPE_FILED_MAP.put(ObjectType.CUSTOMER.getKey(), fieldList);
    fieldList = Lists.newArrayList("orgCode");
    OBJECT_TYPE_FILED_MAP.put(ObjectType.ORGANIZATION.getKey(), fieldList);
    fieldList = Lists.newArrayList("terminalCode");
    OBJECT_TYPE_FILED_MAP.put(ObjectType.TERMINAL.getKey(), fieldList);
    //每个销售类型对应的唯一约束字段
    GOAL_TYPE_FILED_MAP.put(GoalType.ROUTINE.getKey(), Lists.newArrayList());
    fieldList = Lists.newArrayList("productCode");
    GOAL_TYPE_FILED_MAP.put(GoalType.GOODS.getKey(), fieldList);
    fieldList = Lists.newArrayList("productLevelCode");
    GOAL_TYPE_FILED_MAP.put(GoalType.PRODUCT_LEVEL.getKey(), fieldList);
  }

  /**
   * 验证添加方法的参数
   * @param saleGoalDto
   */


  /**
   * 验证参数(为addBatch方法)
   * 1、验证目标类型必须是同一个
   * 2、根据不同的对象维度，分别验证经销商、组织编码、终端编码必须是同一个
   * 3、验证详细参数
   * 4、验证参数中产品是否重复
   * @param saleGoalDtos
   */
  public static void validateParamForAddBatch(List<SaleGoalDto> saleGoalDtos) {
    if (CollectionUtils.isEmpty(saleGoalDtos)) {
      return;
    }
    Integer type = saleGoalDtos.get(0).getSaleGoalType();
    Validate.isTrue(CollectionUtils.isEmpty(
        saleGoalDtos.stream()
            .filter(vo -> !Objects.equals(type, vo.getSaleGoalType()))
            .collect(Collectors.toList())
    ),"暂不支持跨销量目标类型批量操作");
    Set<Integer> objectTypes = saleGoalDtos.stream().map(SaleGoalDto::getObjectType).collect(Collectors.toSet());
    Validate.isTrue(objectTypes.size() == 1, "暂时仅支持同一个对象维度批量操作");
    ObjectType objectType = ObjectType.getByKey(objectTypes.iterator().next());
    Validate.notNull(objectType, "对象维度传入有误");
    switch (objectType) {
      case CUSTOMER:
        Set<String> cusCodes = saleGoalDtos.stream()
            .map(SaleGoalDto::getCusCode)
            .collect(Collectors.toSet());
        Validate.isTrue(cusCodes.size() == 1, "暂时仅支持同一客户批量操作");
        Validate.notBlank(cusCodes.iterator().next(), "客户编码不能为空");
        break;
      case ORGANIZATION:
        Set<String> orgCodes = saleGoalDtos.stream()
            .map(SaleGoalDto::getOrgCode)
            .collect(Collectors.toSet());
        Validate.isTrue(orgCodes.size() == 1, "暂时仅支持同一组织批量操作");
        Validate.notBlank(orgCodes.iterator().next(), "组织编码不能为空");
        break;
      case TERMINAL:
        Set<String> terminalCodes = saleGoalDtos.stream()
            .map(SaleGoalDto::getTerminalCode)
            .collect(Collectors.toSet());
        Validate.isTrue(terminalCodes.size() == 1, "暂时仅支持同一组织批量操作");
        Validate.notBlank(terminalCodes.iterator().next(), "终端编码不能为空");
        break;
    }
    saleGoalDtos.forEach(saleGoalDto -> {
      Validate.notNull(saleGoalDto.getObjectType(), "对象维度不能为空");
      Validate.notNull(saleGoalDto.getTargetYearType(), "年度类型不能为空");
      Validate.notNull(saleGoalDto.getSaleGoalType(), "销量目标类型不能为空");
      Validate.notNull(saleGoalDto.getTaskType(), "任务类型不能为空");
      Validate.notNull(saleGoalDto.getUnitType(),"单位类型不能为空");
      Validate.notNull(TaskType.getByKey(saleGoalDto.getTaskType()),"任务类型传入有误");
      Validate.notBlank(saleGoalDto.getSaleGoalName(), "销量目标名称不能为空");
      Validate.notNull(saleGoalDto.getTargetYear(), "目标年份不能为空!");
      if (saleGoalDto.getSaleGoalType().intValue() == GoalType.PRODUCT_LEVEL.getKey()) {
        Validate.notBlank(saleGoalDto.getProductLevelCode(), "产品层级编码不能为空!");
        Validate.notBlank(saleGoalDto.getProductLevelName(), "产品层级名称不能为空!");
      }
      if (saleGoalDto.getSaleGoalType().intValue() == GoalType.GOODS.getKey()) {
        Validate.notBlank(saleGoalDto.getProductCode(), "商品编码不能为空!");
        Validate.notBlank(saleGoalDto.getProductName(), "商品名称不能为空!");
      }
      if(saleGoalDto.getObjectType().intValue() == ObjectType.CUSTOMER.getKey()){
        saleGoalDto.setObjectCode(saleGoalDto.getCusCode());
        saleGoalDto.setObjectName(saleGoalDto.getCusName());
      }
      if(saleGoalDto.getObjectType().intValue() == ObjectType.ORGANIZATION.getKey()){
        saleGoalDto.setObjectCode(saleGoalDto.getOrgCode());
        saleGoalDto.setObjectName(saleGoalDto.getOrgName());
      }
      if(saleGoalDto.getObjectType().intValue() == ObjectType.TERMINAL.getKey()){
        saleGoalDto.setObjectCode(saleGoalDto.getTerminalCode());
        saleGoalDto.setObjectName(saleGoalDto.getTerminalName());
      }
    });
    //验证重复用的map
    Map<String, Object> map = new HashMap<>(saleGoalDtos.size());
    for (SaleGoalDto dto : saleGoalDtos) {
      //验重
      String key = "";
      String repeat = "";
      if (dto.getSaleGoalType().intValue() == GoalType.GOODS.getKey()) {
        key = new StringBuilder(dto.getProductCode()).append(dto.getTargetYear()).toString();
        repeat = dto.getProductName();
      } else if (dto.getSaleGoalType().intValue() == GoalType.PRODUCT_LEVEL.getKey()) {
        key = new StringBuilder(dto.getProductLevelCode()).append(dto.getTargetYear()).toString();
        repeat = dto.getProductLevelName();
      } else {
        key = dto.getTargetYear().toString();
      }
      StringBuilder msg = new StringBuilder("目标重复");
      if (map.get(key) != null) {
        if (StringUtils.isNotBlank(repeat)) {
          msg.append(",产品/产品层级:").append(repeat);
        }
        msg.append(",目标年份:").append(dto.getTargetYear());
      }
      Validate.isTrue(Objects.isNull(map.get(key)),msg.toString());
      map.put(key, key);
    }
  }


  /**
   * 组装实体对象，为add方法
   * @return
   */
  public static SaleGoal packageEntityForAdd(SaleGoalDto dto) {
    //设置唯一约束字段值
    buildOnlyKey(dto);
    SaleGoal entity = new SaleGoal();
    BeanUtils.copyProperties(dto, entity);
    if (dto.getSaleGoalType().intValue() == GoalType.PRODUCT_LEVEL.getKey()) {
      entity.setProductCode(null);
      entity.setProductName(null);
      entity.setProductSpec(null);
      entity.setOnlyKey(new StringBuilder(dto.getObjectCode()).append(",").append(dto.getProductLevelCode()).append(",").append(dto.getTargetYear()).toString());
    }
    if (dto.getSaleGoalType().intValue() == GoalType.ROUTINE.getKey()) {
      entity.setProductCode(null);
      entity.setProductName(null);
      entity.setProductSpec(null);
      entity.setProductLevelCode(null);
      entity.setProductLevelName(null);
      entity.setOnlyKey(new StringBuilder(dto.getObjectCode()).append(",").append(dto.getTargetYear()).toString());
    }
    if (dto.getSaleGoalType().intValue() == GoalType.GOODS.getKey()) {
      entity.setProductLevelCode(null);
      entity.setProductLevelName(null);
      entity.setOnlyKey(new StringBuilder(dto.getObjectCode()).append(",").append(dto.getProductCode()).append(",").append(dto.getTargetYear()).toString());
    }
    //全年总目标
    dto.getSaleGoalDetailDto().setTaskType(dto.getTaskType());
    entity.setTargetNum(SaleGoalItemUtil.sumSingleItem(dto.getSaleGoalDetailDto()).getYearlySum());
    return entity;
  }

  /**
   * 构建销售目标列表每个对象的onlyKey字段值
   * @param saleGoalDtos
   */
  public static void buildOnlyKey(List<SaleGoalDto> saleGoalDtos) {
    if (CollectionUtils.isEmpty(saleGoalDtos)) {
      return;
    }
    saleGoalDtos.forEach(sale -> buildOnlyKey(sale));
  }

  /**
   * 构建单个销售目标对象onlyKey字段值
   * @param sale
   */
  private static String buildOnlyKey(SaleGoalDto sale) {
    if (sale == null) {
      return null;
    }
    List<String> filedNames = Lists.newArrayList();
    filedNames.addAll(OBJECT_TYPE_FILED_MAP.get(sale.getObjectType()));
    filedNames.addAll(GOAL_TYPE_FILED_MAP.get(sale.getSaleGoalType()));
    filedNames.add("targetYear");
    Map<String, Object> valueMap = FieldHandleUtil.fieldsValue(sale, filedNames.toArray(new String[filedNames.size()]));
    List<String> valueList = Lists.newArrayList();
    filedNames.forEach(s -> {
      Validate.notNull(valueMap.get(s), "%s不能为空");
      valueList.add(valueMap.get(s).toString());
    });
    String onlyKey = String.join(",", valueList.toArray(new String[valueList.size()]));
    sale.setOnlyKey(onlyKey);
    return onlyKey;
  }




}
