package com.biz.crm.dms.business.stock.customer.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.dms.business.allow.sale.sdk.list.service.AllowSaleListVoService;
import com.biz.crm.dms.business.stock.customer.local.entity.StockStatistics;
import com.biz.crm.dms.business.stock.customer.local.entity.StockStatisticsDetail;
import com.biz.crm.dms.business.stock.customer.local.repository.StockStatisticsRepository;
import com.biz.crm.dms.business.stock.customer.local.service.StockStatisticsDetailService;
import com.biz.crm.dms.business.stock.customer.local.service.StockStatisticsService;
import com.biz.crm.dms.business.stock.customer.sdk.dto.StockStatisticsPageDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 库存提报主表(StockStatistics)表服务实现类
 *
 * @author HanJiaJun
 * @since 2022-07-26 15:15:59
 */
@Service("stockStatisticsService")
public class StockStatisticsServiceImpl implements StockStatisticsService {


  @Autowired
  private AllowSaleListVoService allowSaleListVoService;
  @Autowired
  private ProductVoService productVoService;
  @Autowired
  private StockStatisticsRepository stockStatisticsRepository;
  @Autowired
  private CustomerVoService customerVoService;
  @Autowired
  private StockStatisticsDetailService stockStatisticsDetailService;

  /**
   * 分页查询数据
   *
   * @param pageable        分页对象
   * @param stockStatistics 实体对象
   * @return
   */
  @Override
  public Page<StockStatistics> findByConditions(Pageable pageable, StockStatisticsPageDto stockStatistics) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(stockStatistics)) {
      stockStatistics = new StockStatisticsPageDto();
    }
    //时间条件
    Date time=null;
    if (!Objects.isNull(stockStatistics.getStockTime())) {
      Date stockTime = stockStatistics.getStockTime();
      Calendar calendar = Calendar.getInstance();
      calendar.setTime(stockTime);
      calendar.set(Calendar.HOUR_OF_DAY, 23);
      calendar.set(Calendar.MINUTE, 59);
      calendar.set(Calendar.SECOND, 59);
      time = calendar.getTime();
      stockStatistics.setStockTime(time);
    }
    stockStatistics.setTenantCode(TenantUtils.getTenantCode());
    Page<StockStatistics> byConditions = this.stockStatisticsRepository.findByConditions(pageable, stockStatistics);
    if (Objects.isNull(stockStatistics.getStockTime())) {
      return byConditions;
    }
    //如果查询条件有时间的查询条件，则查询当前时间之前最新的操作记录作为主表的库存量，库存时间
    List<StockStatistics> records = byConditions.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      return byConditions;
    }
    for (StockStatistics record : records) {
      String cusCode = record.getCusCode();
      String productCode = record.getProductCode();
      List<StockStatisticsDetail> byCusCodeAndProductCodeAndTime = stockStatisticsDetailService.findByCusCodeAndProductCodeAndTime(cusCode, productCode, time);
      if (!CollectionUtils.isEmpty(byCusCodeAndProductCodeAndTime)) {
        //按操作时间倒序，故取第一条为最新
        StockStatisticsDetail stockStatisticsDetail = byCusCodeAndProductCodeAndTime.get(0);
        record.setStockQuantity(stockStatisticsDetail.getQuantity());
        record.setStockTime(stockStatisticsDetail.getOperationTime());
      }else{
        record.setStockTime(null);
        record.setStockQuantity(null);
      }
    }
      return byConditions;
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public StockStatistics findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.stockStatisticsRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param stockStatistics 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public StockStatistics create(StockStatistics stockStatistics) {
    this.createValidate(stockStatistics);
    this.stockStatisticsRepository.saveOrUpdate(stockStatistics);
    return stockStatistics;
  }

  /**
   * 修改新据
   *
   * @param stockStatistics 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public StockStatistics update(StockStatistics stockStatistics) {
    this.updateValidate(stockStatistics);
    this.stockStatisticsRepository.saveOrUpdate(stockStatistics);
    return stockStatistics;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.stockStatisticsRepository.removeByIds(idList);
  }

  /**
   * 查询
   *
   * @param cusCode
   * @param productCode
   * @return
   */
  @Override
  public StockStatistics findByCusCodeAndProductCode(String cusCode, String productCode) {
    if (!StringUtils.isNoneBlank(cusCode, productCode)) {
      return null;
    }
    return this.stockStatisticsRepository.findByCusCodeAndProductCode(cusCode, productCode);
  }

  @Override
  public List<StockStatistics> findByCusCode(String cusCode) {
    if (!StringUtils.isNoneBlank(cusCode)) {
      return null;
    }
    return this.stockStatisticsRepository.findByCusCode(cusCode);
  }

  /**
   * 创建验证
   *
   * @param stockStatistics
   */
  private void createValidate(StockStatistics stockStatistics) {
    Validate.notNull(stockStatistics, "新增时，对象信息不能为空！");
    stockStatistics.setId(null);
    stockStatistics.setTenantCode(TenantUtils.getTenantCode());
    stockStatistics.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    stockStatistics.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    Validate.notNull(stockStatistics.getTenantCode(), "新增数据时，租户编号不能为空！");

  }

  /**
   * 修改验证
   *
   * @param stockStatistics
   */
  private void updateValidate(StockStatistics stockStatistics) {
    Validate.notNull(stockStatistics, "修改时，对象信息不能为空！");
    Validate.notBlank(stockStatistics.getId(), "修改数据时，不能为空！");
    Validate.notBlank(stockStatistics.getTenantCode(), "修改数据时，租户编号不能为空！");

  }
}

