package com.biz.crm.dms.business.stock.customer.local.service.strategy.adjust;

import com.biz.crm.dms.business.stock.customer.local.entity.StockStatistics;
import com.biz.crm.dms.business.stock.customer.local.entity.StockStatisticsDetail;
import com.biz.crm.dms.business.stock.customer.local.service.StockStatisticsDetailService;
import com.biz.crm.dms.business.stock.customer.local.service.StockStatisticsService;
import com.biz.crm.dms.business.stock.customer.sdk.dto.StockStatisticsDto;
import com.biz.crm.dms.business.stock.customer.sdk.strategy.StockStatisticsAdjustTypeStrategy;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe:<p> 库存更新(stock_update)调整类型策略器实现
 * 覆盖主表调整数量。
 * </p>
 * @createTime 2022年07月27日 12:13:00
 */
@Component
public class StockStatisticsAdjustTypeStrategyImpl implements StockStatisticsAdjustTypeStrategy {

  @Autowired
  private StockStatisticsService stockStatisticsService;
  @Autowired
  private StockStatisticsDetailService stockStatisticsDetailService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private CustomerVoService customerVoService;

  @Override
  public String getAdjustTypeCode() {
    return "stock_update";
  }

  @Override
  public String getAdjustTypeName() {
    return "库存更新";
  }

  /**
   * 处理
   *
   * @param statisticsDtoList
   */
  @Override
  public void adjustHandle(List<StockStatisticsDto> statisticsDtoList) {
    /**
     * 1. 通过客户和产品查找主表记录
     * 2. 判断对主表更新还是创建
     * 3. 添加子表记录
     */
    List<String> cusCodes = statisticsDtoList.stream().map(StockStatisticsDto::getCusCode).collect(Collectors.toList());
    List<CustomerVo> byCustomerCodes = this.customerVoService.findByCustomerCodes(cusCodes);
    Validate.notEmpty(byCustomerCodes,"库存提报，未查询到相关客户信息");
    Map<String, CustomerVo> customerVoMap = byCustomerCodes.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, e -> e));
    //确保主表提报时间与子表的操作时间一致，手动创建时间
    Date date = new Date();
    //操作主表
    for (StockStatisticsDto dto : statisticsDtoList) {
      String cusCode = dto.getCusCode();
      String productCode = dto.getProductCode();
      StockStatistics stockStatistics = this.stockStatisticsService.findByCusCodeAndProductCode(cusCode, productCode);
      if (Objects.nonNull(stockStatistics)) {
        //执行更新
        stockStatistics.setStockTime(date);
        stockStatistics.setStockQuantity(dto.getOperationQuantity());
        CustomerVo customerVo = customerVoMap.get(cusCode);
        if (Objects.nonNull(customerVo)) {
          List<CustomerRelateOrgVo> orgList = customerVo.getOrgList();
          if (CollectionUtils.isNotEmpty(orgList)){
            //确保客户关联的企业组织得到更新
            CustomerRelateOrgVo customerRelateOrgVo = orgList.get(0);
            stockStatistics.setCusOrgCode(customerRelateOrgVo.getOrgCode());
            stockStatistics.setCusOrgName(customerRelateOrgVo.getOrgName());
          }
        }
        this.stockStatisticsService.update(stockStatistics);
      } else {
        //执行创建
        StockStatistics entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, StockStatistics.class, HashSet.class, ArrayList.class);
        entity.setStockTime(date);
        entity.setStockQuantity(dto.getOperationQuantity());
        CustomerVo customerVo = customerVoMap.get(cusCode);
        if (Objects.nonNull(customerVo)) {
          List<CustomerRelateOrgVo> orgList = customerVo.getOrgList();
          if (CollectionUtils.isNotEmpty(orgList)){
            //自动带出企业组织
            CustomerRelateOrgVo customerRelateOrgVo = orgList.get(0);
            entity.setCusOrgCode(customerRelateOrgVo.getOrgCode());
            entity.setCusOrgName(customerRelateOrgVo.getOrgName());
          }
        }
        this.stockStatisticsService.create(entity);
      }
    }
    //保存明细操作记录
    Collection<StockStatisticsDetail> stockStatisticsDetails = this.nebulaToolkitService.copyCollectionByWhiteList(statisticsDtoList, StockStatisticsDto.class, StockStatisticsDetail.class, HashSet.class, ArrayList.class);
    for (StockStatisticsDetail stockStatisticsDetail : stockStatisticsDetails) {
      //如果传入对象里存在操作人组织编码则不改变，如果不存在则使用上述客户的企业组织编码（customerVoMap）
      if (StringUtils.isBlank(stockStatisticsDetail.getOperationOrgCode())){
        String cusCode = stockStatisticsDetail.getCusCode();
        CustomerVo customerVo = customerVoMap.get(cusCode);
        if (Objects.nonNull(customerVo)) {
          List<CustomerRelateOrgVo> orgList = customerVo.getOrgList();
          if (CollectionUtils.isNotEmpty(orgList)) {
            //自动带入企业组织编码
            CustomerRelateOrgVo customerRelateOrgVo = orgList.get(0);
            stockStatisticsDetail.setOperationOrgCode(customerRelateOrgVo.getOrgCode());
            stockStatisticsDetail.setOperationOrgName(customerRelateOrgVo.getOrgName());
          }
          stockStatisticsDetail.setOperationTime(date);
          stockStatisticsDetail.setQuantity(stockStatisticsDetail.getOperationQuantity());
        }
      }
    }
    this.stockStatisticsDetailService.createBatch((List<StockStatisticsDetail>) stockStatisticsDetails);
  }
}
