package com.biz.crm.dms.business.stock.customer.local.service.internal;

import com.biz.crm.dms.business.allow.sale.sdk.enums.AllowSaleListTypeEnums;
import com.biz.crm.dms.business.allow.sale.sdk.list.dto.ValidateAllowSaleProductDto;
import com.biz.crm.dms.business.allow.sale.sdk.list.service.AllowSaleListVoService;
import com.biz.crm.dms.business.stock.customer.local.entity.StockStatistics;
import com.biz.crm.dms.business.stock.customer.local.repository.StockStatisticsRepository;
import com.biz.crm.dms.business.stock.customer.local.service.StockStatisticsDetailService;
import com.biz.crm.dms.business.stock.customer.local.service.StockStatisticsService;
import com.biz.crm.dms.business.stock.customer.sdk.dto.StockStatisticsDto;
import com.biz.crm.dms.business.stock.customer.sdk.service.StockStatisticsVoService;
import com.biz.crm.dms.business.stock.customer.sdk.strategy.StockStatisticsOperationTypeStrategy;
import com.biz.crm.dms.business.stock.customer.sdk.vo.StockStatisticsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe:
 * @createTime 2022年07月26日 16:22:00
 */
@Service
public class StockStatisticsVoServiceImpl implements StockStatisticsVoService {

  @Autowired
  private List<StockStatisticsOperationTypeStrategy> stockStatisticsOperationTypeStrategies;
  @Autowired
  private AllowSaleListVoService allowSaleListVoService;
  @Autowired
  private ProductVoService productVoService;
  @Autowired
  private StockStatisticsRepository stockStatisticsRepository;
  @Autowired
  private CustomerVoService customerVoService;
  @Autowired
  private StockStatisticsDetailService stockStatisticsDetailService;
  @Autowired
  private StockStatisticsService stockStatisticsService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 库存提报
   *
   * @param stockStatistics
   * @return
   */
  @Override
  @Transactional
  public void create(List<StockStatisticsDto> stockStatistics) {
    /**
     * 1. 校验
     * 2. 查找操作类型策略器
     */
    Validate.notEmpty(stockStatistics, "库存提报，数据不能为空");
    Validate.notEmpty(stockStatisticsOperationTypeStrategies, "库存提报，未查询到操作类型策略器");
    //校验非空字段
    for (StockStatisticsDto dto : stockStatistics) {
      this.createValidate(dto);
    }
    //校验重复性
    List<String> codes = stockStatistics.stream().map(e -> e.getCusCode().concat(e.getProductCode())).distinct().collect(Collectors.toList());
    Validate.isTrue(codes.size()==stockStatistics.size(),"存在重复数据");
    Map<String, List<StockStatisticsDto>> collect = stockStatistics.stream().collect(Collectors.groupingBy(StockStatisticsDto::getOperationType));
    collect.forEach((k, v) -> {
      for (StockStatisticsOperationTypeStrategy stockStatisticsOperationTypeStrategy : stockStatisticsOperationTypeStrategies) {
        if (stockStatisticsOperationTypeStrategy.getOperationTypeCode().equals(k)) {
          stockStatisticsOperationTypeStrategy.operationHandle(v);
        }
      }
    });
  }

  @Override
  public List<StockStatisticsVo> findDetailByCusCode(StockStatisticsDto statisticsDto) {
    if (Objects.isNull(statisticsDto) || StringUtils.isBlank(statisticsDto.getCusCode())) {
      return null;
    }
    //CustomerCode赋值  是为了不进行模糊查询
    String customerCode = statisticsDto.getCusCode();
    List<StockStatistics> stockStatisticsList = this.stockStatisticsService.findByCusCode(customerCode);
    //如果当前客户存在经销商库存记录，则返回
    List<StockStatisticsVo> byCusCode=new ArrayList<>();
    if (!CollectionUtils.isEmpty(stockStatisticsList)) {
      byCusCode= (List<StockStatisticsVo> )this.nebulaToolkitService.copyCollectionByWhiteList(stockStatisticsList, StockStatistics.class, StockStatisticsVo.class, HashSet.class, ArrayList.class);
      List<String> productCodes = byCusCode.stream().map(StockStatisticsVo::getProductCode).collect(Collectors.toList());
      List<ProductVo> mainDetailsByProductCodes = this.productVoService.findMainDetailsByProductCodes(productCodes);
      if (CollectionUtils.isEmpty(mainDetailsByProductCodes)){
        return byCusCode;
      }
      Map<String, ProductVo> collect = mainDetailsByProductCodes.stream().collect(Collectors.toMap(ProductVo::getProductCode, v -> v));
      for (StockStatisticsVo vo : byCusCode) {
        ProductVo productVo = collect.get(vo.getProductCode());
        if (Objects.nonNull(productVo)){
          vo.setPrimaryPictureUrl(productVo.getPrimaryPictureUrl());
        }
      }
      return byCusCode;
    }
    //如果当前可不存在经销商库存记录则返回允销
    ValidateAllowSaleProductDto dto = new ValidateAllowSaleProductDto();
    dto.setBusinessCode(customerCode);
    dto.setListType(AllowSaleListTypeEnums.CUSTOMER.getCode());
    //查询该客户所有允销商品编码
    List<String> allowSaleProductCodes = this.allowSaleListVoService.findAllowSaleProductCodes(dto);
    if (CollectionUtils.isEmpty(allowSaleProductCodes)){
      return byCusCode;
    }
    List<ProductVo> records = this.productVoService.findMainDetailsByProductCodes(allowSaleProductCodes);
    if (CollectionUtils.isEmpty(records)){
      return byCusCode;
    }
    List<StockStatisticsVo> res = new ArrayList<>();
    CustomerVo customerVo = this.customerVoService.findDetailsByIdOrCode(null, customerCode);
    if (CollectionUtils.isEmpty(records)) {
      return byCusCode;
    }
    for (ProductVo record : records) {
      StockStatisticsVo statistics = new StockStatisticsVo();
      statistics.setCusCode(customerCode);
      statistics.setCusName(customerVo.getCustomerName());
      statistics.setStockQuantity(BigDecimal.ZERO);
      statistics.setProductCode(record.getProductCode());
      statistics.setProductName(record.getProductName());
      statistics.setProductUnit(record.getSaleUnit());
      statistics.setPrimaryPictureUrl(record.getPrimaryPictureUrl());
      statistics.setIsShelf(record.getIsShelf());
      if (Objects.nonNull(customerVo)) {
        statistics.setCusOrgCode(customerVo.getCustomerOrgCode());
        statistics.setCusOrgName(customerVo.getCustomerOrgName());
      }
      res.add(statistics);
    }
    return res;
  }

  /**
   * 非空校验
   * @param dto
   */
  private void createValidate(StockStatisticsDto dto) {
    Validate.notBlank(dto.getOperationSource(),"操作来源不能为空!");
    Validate.notBlank(dto.getOperationType(),"操作类型不能为空!");
    Validate.notBlank(dto.getAdjustType(),"调整类型不能为空!");
    Validate.notBlank(dto.getCusCode(),"客户编码不能为空!");
    Validate.notBlank(dto.getProductCode(),"商品编码不能为空!");
    Validate.notNull(dto.getOperationQuantity(),"操作数量不能为空!");
  }
}
