package com.biz.crm.ems.admin.web.controller;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForEMS;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.ems.admin.web.dto.ResetPasswordDto;
import com.biz.crm.ems.admin.web.service.ResetPasswordService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.dto.MallUserChangeCustomerDto;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaCustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserVo;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.util.ObjectUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm-ems
 * @PackageName com.biz.crm.ems.admin.web.controller
 * @ClassName MallUserLoginController
 * @Author YangWei
 * @Date 2022/3/16 下午1:48
 * @Description 商城登录用户接口
 */
@Slf4j
@RestController
@RequestMapping("/v1/mallUser/mallUser")
@Api(tags = "商城用户: MallUserVo: 商城用户Vo")
public class MallUserVoController {

  @Autowired
  private CustomerVoService customerVoService;

  @Autowired
  private CustomerUserVoService customerUserVoService;

  @Autowired
  private ResetPasswordService resetPasswordService;

  @Autowired
  private LoginUserService loginUserService;

  /**
   * 根据登录账号查询用户信息
   * @param userName
   * @return
   */
  @ApiOperation(value = "根据登录账号查询用户信息")
  @GetMapping(value = {"/findByUserName"})
  public Result<CustomerUserVo> findByUserName(@RequestParam("userName") String userName){
    try {
      CustomerUserVo vo = this.customerUserVoService.findByUserName(userName);
      return Result.ok(vo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 切换客户用户当前客户
   * @param
   * @return
   */
  @ApiOperation(value = "切换客户用户当前客户")
  @PostMapping("/changeUserCurrentCustomer")
  public Result changeUserCurrentCustomer(@RequestBody MallUserChangeCustomerDto dto) {
    try {
      LoginUserDetailsForEMS loginUserDetails = this.loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
      dto.setUserName(loginUserDetails.getAccount());
      Validate.notBlank(dto.getCustomerCode(),"待切换客户编码不能为空");
      List<CustomerVo> customerVos = this.customerVoService.findByCustomerCodes(Collections.singletonList(dto.getCustomerCode()));
      Validate.notEmpty(customerVos, "当前选中客户不存在或已删除");
      Optional<CustomerVo> customerVoEnable = customerVos.stream().filter(c-> EnableStatusEnum.DISABLE.getCode().equals(c.getEnableStatus())).findAny();
      Validate.isTrue(!customerVoEnable.isPresent(), "当前选中客户已经被禁用");
      Optional<CustomerVo> customerVoLock = customerVos.stream().filter(c-> LockStateEnum.LOCK.getCode().equals(c.getLockState())).findAny();
      Validate.isTrue(!customerVoLock.isPresent(), "当前选中客户已经被锁定");
      CustomerUserVo customerUserVo = this.customerUserVoService.findByUserName(dto.getUserName());
      Validate.isTrue((!ObjectUtils.isEmpty(customerUserVo)),"切换失败，未查询到对应用户信息");
      List<CustomerUserRelaCustomerVo> customerInfoList = customerUserVo.getCustomerInfoList();
      Validate.isTrue(!org.springframework.util.CollectionUtils.isEmpty(customerInfoList),"当前用户没有经销商信息");
      Map<String, CustomerUserRelaCustomerVo> customerUserRelaCustomerVoMap = customerInfoList.stream()
          .collect(Collectors.toMap(CustomerUserRelaCustomerVo::getCustomerCode, Function.identity()));
      CustomerUserRelaCustomerVo customerUserRelaCustomerVo = customerUserRelaCustomerVoMap.get(dto.getCustomerCode());
      Validate.isTrue(!ObjectUtils.isEmpty(customerUserRelaCustomerVo),"待切换客户和当前登录用户没有关联关系");
      loginUserDetails.setCustomerCode(customerUserRelaCustomerVo.getCustomerCode());
      loginUserDetails.setCustomerName(customerUserRelaCustomerVo.getCustomerName());
      loginUserDetails.setOrgName(customerUserRelaCustomerVo.getOrgName());
      loginUserDetails.setOrgCode(customerUserRelaCustomerVo.getOrgCode());
      loginUserDetails.setCustomerOrgCode(customerUserRelaCustomerVo.getCustomerOrgCode());
      loginUserDetails.setCustomerOrgName(customerUserRelaCustomerVo.getCustomerOrgName());
      // --
      List<SimpleGrantedAuthority> grantedAuthorities = Lists.newArrayList();
      List<String> roles = loginUserDetails.getRoles();
      Validate.isTrue(!org.springframework.util.CollectionUtils.isEmpty(roles), "当前登录用户没有角色");
      roles.forEach(r-> grantedAuthorities.add(new SimpleGrantedAuthority(r)));
      // --
      this.loginUserService.refreshLogin(loginUserDetails, grantedAuthorities);
      return Result.ok("切换成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 在登录状态中更改密码
   * @param dto
   * @return
   */
  @ApiOperation(value = "在登录状态中更改密码")
  @PostMapping("/updatePasswordInLoginStatus")
  public Result updatePasswordInLoginStatus(@RequestBody ResetPasswordDto dto) {
    try {
      this.resetPasswordService.updatePasswordInLoginStatus(dto);
      return Result.ok("修改成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}