package com.biz.crm.ems.admin.web.strategy.internal;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForEMS;
import com.biz.crm.common.sms.sdk.service.ValiditySmsCodeService;
import com.biz.crm.mdm.business.customer.user.sdk.constant.MallUserConstant;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaCustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaRoleVo;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserVo;
import com.bizunited.nebula.common.util.Aes128Utils;
import com.bizunited.nebula.security.sdk.event.AuthenticationDecisionStrategy;
import com.bizunited.nebula.security.sdk.event.DecisionTypes;
import com.bizunited.nebula.security.sdk.vo.LoginDetails;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 手机号 + 短信验证码方式策略实现
 *
 * @author pengxi
 */
@Slf4j
@Component
public class DefaultPhoneAndSmsCodeStrategy implements AuthenticationDecisionStrategy {

  @Autowired
  private CustomerUserVoService customerUserVoService;

  @Autowired
  private ValiditySmsCodeService validitySmsCodeService;

  @Override
  public DecisionTypes type() {
    return DecisionTypes.PHONEANDSMSCODE;
  }

  @Override
  public boolean onAuthenticate(LoginDetails loginDetails) {
    /*
     * 处理过程如下：
     * 1、首先通过手机号查询当前用户并检查用户有效性
     * 2、检查当前短信验证码的有效性
     * 3、填充security鉴权必要信息：tenantCode,account,password
     * 4、然后完善登录信息中的用户基本信息
     * 5、最后完善登录信息中的岗位组织等信息
     */
    LoginUserDetailsForEMS loginUserDetails = (LoginUserDetailsForEMS) loginDetails;
    log.info("HttpServletRequest phone:{}", loginUserDetails.getPhone());
    log.info("HttpServletRequest smsVerificationCode:{}", loginUserDetails.getSmsVerificationCode());
    Validate.notBlank(loginUserDetails.getPhone(), "登录手机号不能为空！");
    Validate.notBlank(loginUserDetails.getSmsVerificationCode(), "短信验证码不能为空！");
    String phone = Aes128Utils.decrypt(loginUserDetails.getPhone(), MallUserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("decrypt phone:{}", phone);
    String smsVerificationCode = Aes128Utils.decrypt(loginUserDetails.getSmsVerificationCode(), MallUserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    log.error("decrypt smsVerificationCode:{}", smsVerificationCode);
    // 1、
    CustomerUserVo customerUserVo = this.customerUserVoService.findByPhone(phone);
    Validate.notNull(customerUserVo, "用户不存在或已删除！");
    //Validate.notBlank(customerUserVo.getUserType(), "未知用户类型，请联系管理员！");
    //Validate.isTrue(UserTypeEnum.USER.getCode().equals(customerUserVo.getUserType()), "只有企业用户才能登录当前系统！");
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(customerUserVo.getEnableStatus()), "当前用户已禁用！");
    Validate.isTrue(LockStateEnum.UNLOCK.getCode().equals(customerUserVo.getLockState()), "当前用户已锁定！");
    // 2、
    Boolean isAvailable = this.validitySmsCodeService.isAvailableVerificationCode(phone, smsVerificationCode);
    Validate.isTrue(isAvailable, "验证码过期或不正确，请重新获取！");
    // 3、
    loginUserDetails.setAccount(customerUserVo.getUserName());
    loginUserDetails.setPassword(customerUserVo.getUserPassword());
    // 4、（以下属业务侧，按业务需求添加字段信息）
    loginUserDetails.setType(DecisionTypes.PHONEANDSMSCODE.getCode());
    loginUserDetails.setUsertype(customerUserVo.getUserType());
    loginUserDetails.setTenantCode(customerUserVo.getTenantCode());
    loginUserDetails.setUsername(customerUserVo.getFullName());
    loginUserDetails.setPhone(customerUserVo.getUserPhone());
    loginUserDetails.setRealName(customerUserVo.getUserName());
    //经销商
    List<CustomerUserRelaCustomerVo> customerInfoList = customerUserVo.getCustomerInfoList();
    if(!CollectionUtils.isEmpty(customerInfoList)){
      CustomerUserRelaCustomerVo customerUserRelaCustomerVo = customerInfoList.get(0);
      loginUserDetails.setCustomerCode(customerUserRelaCustomerVo.getCustomerCode());
      loginUserDetails.setCustomerName(customerUserRelaCustomerVo.getCustomerName());
      loginUserDetails.setOrgCode(customerUserRelaCustomerVo.getOrgCode());
      loginUserDetails.setOrgName(customerUserRelaCustomerVo.getOrgName());
      loginUserDetails.setCustomerOrgCode(customerUserRelaCustomerVo.getCustomerOrgCode());
      loginUserDetails.setCustomerOrgName(customerUserRelaCustomerVo.getCustomerOrgName());
    }
    //角色
    List<CustomerUserRelaRoleVo> roleInfoList = customerUserVo.getRoleInfoList();
    if(!CollectionUtils.isEmpty(roleInfoList)) {
      loginUserDetails.setRoles(roleInfoList.stream().map(CustomerUserRelaRoleVo::getRoleCode).collect(Collectors.toList()));
    }

    return true;
  }
}
