package com.biz.crm.kms.admin.web.imports.service;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.kms.admin.web.imports.model.EnterpriseOrderImportVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.grab.rule.sdk.enums.GrabType;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGoodsEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGrabEntity;
import com.biz.crm.kms.business.invoice.order.local.model.OrderGrabSyncModel;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGoodsService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGrabService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderService;
import com.biz.crm.kms.business.invoice.order.sdk.constant.OrderConstant;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 企业订货单信息导入
 *
 * @author ning.zhang
 * @date 2022/11/7
 */
@Component
public class EnterpriseOrderImportProcess implements ImportProcess<EnterpriseOrderImportVo> {

  @Autowired
  private OrderGrabService orderGrabService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private OrderService orderService;
  @Autowired
  private OrderGoodsService orderGoodsService;

  @Override
  public Class<EnterpriseOrderImportVo> findCrmExcelVoClass() {
    return EnterpriseOrderImportVo.class;
  }

  @Override
  public String getTemplateCode() {
    return "KMS_ENTERPRISE_ORDER_IMPORT";
  }

  @Override
  public String getTemplateName() {
    return "企业订货单导入";
  }

  @Override
  public Integer getBatchCount() {
    return Integer.MAX_VALUE;
  }

  @Override
  public Map<Integer, String> execute(LinkedHashMap<Integer, EnterpriseOrderImportVo> data
      , TaskGlobalParamsVo paramsVo
      , Map<String, Object> params) {
    List<EnterpriseOrderImportVo> dataList = (List<EnterpriseOrderImportVo>) data.values();
    if (CollectionUtils.isEmpty(dataList)) {
      return Maps.newHashMap();
    }
    Map<Integer, String> errorMap = this.validation(dataList);
    if (!errorMap.isEmpty()) {
      return errorMap;
    }
    List<OrderEntity> orderEntities = this.buildOrderEntity(dataList);
    List<OrderGrabEntity> orderGrabEntities = (List<OrderGrabEntity>) this.nebulaToolkitService.copyCollectionByBlankList(orderEntities, OrderEntity.class
        , OrderGrabEntity.class, HashSet.class, ArrayList.class);
    OrderGrabSyncModel syncModel = this.orderGrabService.buildOrderGrabSyncModel(orderGrabEntities);
    Map<Integer, String> businessErrorMap = this.businessValidation(dataList, syncModel);
    if (!businessErrorMap.isEmpty()) {
      return businessErrorMap;
    }
    this.perfectOrder(orderEntities, syncModel);
    this.orderService.create(orderEntities);
    return Maps.newHashMap();
  }

  /**
   * 导入数据业务校验
   *
   * @param dataList 导入数据
   * @param syncModel 数据解析数据
   * @return 错误信息
   */
  private Map<Integer, String> businessValidation(List<EnterpriseOrderImportVo> dataList, OrderGrabSyncModel syncModel) {
    Map<Integer, String> resultMap = Maps.newHashMap();
    for (EnterpriseOrderImportVo importVo : dataList) {
      try {
        List<DirectStoreVo> directStoreList = syncModel.getDirectStoreMap().get(String.format("%s-%s", importVo.getDirectCode(), importVo.getKaStoreCode()));
        Validate.isTrue(!CollectionUtils.isEmpty(directStoreList), "未查询到与企业门店对应关系");
        Validate.isTrue(directStoreList.size() == 1, "商超门店存在多个对应关系");
        DirectStoreVo directStoreVo = directStoreList.get(0);
        Validate.isTrue(Objects.equals(directStoreVo.getTerminalCode(), importVo.getStoreCode()), "企业门店编号与系统数据不匹配");
        Validate.isTrue(Objects.equals(directStoreVo.getTerminalName(), importVo.getStoreName()), "企业门店名称与系统数据不匹配");
        Validate.isTrue(Objects.equals(directStoreVo.getSupermarketName(), importVo.getKaName()), "系统名称与系统数据不匹配");
        List<TerminalSupplyVo> terminalSupplyList = syncModel.getTerminalSupplyMap().get(importVo.getStoreCode());
        Validate.isTrue(!CollectionUtils.isEmpty(terminalSupplyList), "未匹配售达方信息");
        Validate.isTrue(terminalSupplyList.size() == 1, "企业门店存在多个售达方");
        TerminalSupplyVo terminalSupplyVo = terminalSupplyList.get(0);
        Validate.isTrue(Objects.equals(terminalSupplyVo.getCustomerCode(), importVo.getSoldToPartyCode()), "售达方编码与系统数据不匹配");
        Validate.isTrue(Objects.equals(terminalSupplyVo.getCustomerName(), importVo.getSoldToPartyName()), "售达方名称与系统数据不匹配");
        List<DirectProductVo> productList = syncModel.getDirectProductMap().get(String.format("%s-%s", importVo.getDirectCode(), importVo.getKaGoodsCode()));
        Validate.isTrue(!CollectionUtils.isEmpty(productList), "未匹配到与企业商品对应关系");
        productList = productList.stream().filter(dpv -> dpv.getSellPartyCode().equals(importVo.getSoldToPartyCode())).collect(Collectors.toList());
        Validate.isTrue(!CollectionUtils.isEmpty(productList), String.format("售达方[%s]未查询到与企业商品对应关系", importVo.getSoldToPartyCode()));
        Validate.isTrue(productList.size() == 1, "企业商品存在多个对应关系");
        DirectProductVo directProductVo = productList.get(0);
        Validate.isTrue(Objects.equals(directProductVo.getProductCode(), importVo.getGoodsCode()), "企业产品编号与系统数据不匹配");
        Validate.isTrue(Objects.equals(directProductVo.getProductName(), importVo.getGoodsName()), "企业产品名称与系统数据不匹配");
      } catch (Exception e) {
        resultMap.put(dataList.indexOf(importVo), e.getMessage());
      }
    }
    return resultMap;
  }

  /**
   * 构建订货单抓单数据
   *
   * @param dataList 导入数据
   * @return 订单抓单数据
   */
  private List<OrderEntity> buildOrderEntity(List<EnterpriseOrderImportVo> dataList) {
    Map<String, OrderEntity> resultMap = Maps.newHashMap();
    dataList.forEach(importVo -> {
      OrderEntity orderEntity = resultMap.get(String.format("%s-%s-%s", importVo.getDirectCode()
          , importVo.getKaStoreCode(), importVo.getKaOrderNumber()));
      if (Objects.isNull(orderEntity)) {
        orderEntity = this.nebulaToolkitService.copyObjectByWhiteList(importVo, OrderEntity.class
            , HashSet.class, ArrayList.class, "finalOrderAmountNot", "finalOrderAmount");
        orderEntity.setOrderNumber(this.generateCodeService.generateCode(OrderConstant.ORDER_CODE, 1).get(0));
        orderEntity.setGoodsList(Lists.newArrayList());
        orderEntity.setVersionNumber(1);
        orderEntity.setInvoicesSource(GrabType.IMPORT.getDictCode());
        orderEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        orderEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        orderEntity.setFinalOrderAmountNot(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getFinalOrderAmountNot(), null)));
        orderEntity.setFinalOrderAmount(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getFinalOrderAmount(), null)));
        resultMap.put(importVo.getKaOrderNumber(), orderEntity);
      }
      OrderGoodsEntity orderGoodsEntity = this.nebulaToolkitService.copyObjectByWhiteList(importVo
          , OrderGoodsEntity.class, HashSet.class, ArrayList.class
          , "curUnitOrderQuantity", "unitPrice", "orderAmount", "orderAmountNot", "numerator");
      orderGoodsEntity.setKaName(orderEntity.getKaName());
      orderGoodsEntity.setDirectCode(orderEntity.getDirectCode());
      orderGoodsEntity.setVersionNumber(orderEntity.getVersionNumber());
      orderGoodsEntity.setOrderNumber(orderEntity.getOrderNumber());
      orderGoodsEntity.setKaOrderNumber(orderEntity.getKaOrderNumber());
      orderGoodsEntity.setCurUnitOrderQuantity(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getCurUnitOrderQuantity(), null)));
      orderGoodsEntity.setUnitPrice(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getUnitPrice(), null)));
      orderGoodsEntity.setCurCompanyUnitOrderQuantity(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getCurCompanyUnitOrderQuantity(), null)));
      orderGoodsEntity.setOrderAmount(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getOrderAmount(), null)));
      orderGoodsEntity.setOrderAmountNot(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getOrderAmountNot(), null)));
      orderEntity.getGoodsList().add(orderGoodsEntity);
    });
    return Lists.newArrayList(resultMap.values());
  }

  /**
   * 完善单据信息
   *
   * @param orderEntities 单据数据列表
   */
  private void perfectOrder(List<OrderEntity> orderEntities, OrderGrabSyncModel syncModel) {
    orderEntities.forEach(orderEntity -> {
      this.orderService.buildOrderInfo(orderEntity, syncModel);
      orderEntity.getGoodsList().forEach(orderGoodsEntity -> this.orderGoodsService.buildGoodsInfo(orderGoodsEntity, orderEntity, syncModel));
      this.orderService.buildOrderStatus(orderEntity);
    });
  }

  /**
   * 导入数据校验
   *
   * @param dataList 导入数据
   */
  private Map<Integer, String> validation(List<EnterpriseOrderImportVo> dataList) {
    Map<Integer, String> resultMap = Maps.newHashMap();
    for (EnterpriseOrderImportVo importVo : dataList) {
      try {
        Validate.notBlank(importVo.getDirectCode(), "缺失系统编号");
        Validate.notBlank(importVo.getKaName(), "缺失系统名称");
        Validate.notBlank(importVo.getKaOrderNumber(), "缺失商超订单编号");
        Validate.notBlank(importVo.getStoreCode(), "缺失企业门店编号");
        Validate.notBlank(importVo.getStoreName(), "缺失企业门店名称");
        Validate.notBlank(importVo.getKaStoreCode(), "缺失商超门店编号");
        Validate.notBlank(importVo.getKaStoreName(), "缺失商超门店名称");
        Validate.notBlank(importVo.getOrderDate(), "缺失订单日期");
        Validate.notBlank(importVo.getFinalOrderAmount(), "缺失订货单总金额（含税）");
        Validate.notBlank(importVo.getKaGoodsCode(), "缺失商超产品编号");
        Validate.notBlank(importVo.getKaGoodsName(), "缺失商超产品名称");
        Validate.notBlank(importVo.getGoodsCode(), "缺失企业产品编号");
        Validate.notBlank(importVo.getGoodsName(), "缺失企业产品名称");
        Validate.notBlank(importVo.getCurUnit(), "缺失商超单位");
        Validate.notBlank(importVo.getCurUnitOrderQuantity(), "缺失商超数量");
        Validate.notBlank(importVo.getCurCompanyUnit(), "缺失企业单位");
        Validate.notBlank(importVo.getCurCompanyUnitOrderQuantity(), "缺失企业数量");
        Validate.notBlank(importVo.getUnitPrice(), "缺失商超订货单价");
        Validate.notBlank(importVo.getOrderAmount(), "缺失商超订货金额（含税）");
        Validate.isTrue(NumberUtils.isParsable(importVo.getFinalOrderAmount()), "订货单总金额（含税）格式错误");
        if (StringUtils.isNotBlank(importVo.getFinalOrderAmountNot())) {
          Validate.isTrue(NumberUtils.isParsable(importVo.getFinalOrderAmountNot()), "订货单总金额（不含税）格式错误");
        }
        Validate.isTrue(NumberUtils.isParsable(importVo.getCurUnitOrderQuantity()), "商超数量格式错误");
        Validate.isTrue(NumberUtils.isParsable(importVo.getCurCompanyUnitOrderQuantity()), "企业数量格式错误");
        Validate.isTrue(NumberUtils.isParsable(importVo.getUnitPrice()), "商超订货单价格式错误");
        Validate.isTrue(NumberUtils.isParsable(importVo.getOrderAmount()), "商超订货金额（含税）格式错误");
        if (StringUtils.isNotBlank(importVo.getOrderAmountNot())) {
          Validate.isTrue(NumberUtils.isParsable(importVo.getOrderAmountNot()), "商超订货金额（不含税）格式错误");
        }
      } catch (Exception e) {
        resultMap.put(dataList.indexOf(importVo), e.getMessage());
      }
    }
    return resultMap;
  }
}
