package com.biz.crm.kms.admin.web.imports.service;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.kms.admin.web.imports.model.SupermarketOrderImportVo;
import com.biz.crm.kms.business.grab.rule.sdk.enums.GrabType;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGoodsEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGrabEntity;
import com.biz.crm.kms.business.invoice.order.local.model.OrderGrabSyncModel;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGoodsService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGrabService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderService;
import com.biz.crm.kms.business.invoice.order.sdk.constant.OrderConstant;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 商超订货单信息导入
 *
 * @author ning.zhang
 * @date 2022/11/3
 */
@Component
public class SupermarketOrderImportProcess implements ImportProcess<SupermarketOrderImportVo> {

  @Autowired
  private OrderGrabService orderGrabService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired
  private OrderService orderService;
  @Autowired
  private OrderGoodsService orderGoodsService;

  @Override
  public Class<SupermarketOrderImportVo> findCrmExcelVoClass() {
    return SupermarketOrderImportVo.class;
  }

  @Override
  public String getTemplateCode() {
    return "KMS_SUPERMARKET_ORDER_IMPORT";
  }

  @Override
  public String getTemplateName() {
    return "商超订货单导入";
  }

  @Override
  public Integer getBatchCount() {
    return Integer.MAX_VALUE;
  }

  @Override
  public Map<Integer, String> execute(LinkedHashMap<Integer, SupermarketOrderImportVo> data
      , TaskGlobalParamsVo paramsVo
      , Map<String, Object> params) {
    List<SupermarketOrderImportVo> dataList = (List<SupermarketOrderImportVo>) data.values();
    if (CollectionUtils.isEmpty(dataList)) {
      return Maps.newHashMap();
    }
    Map<Integer, String> errorMap = this.validation(dataList);
    if (!errorMap.isEmpty()) {
      return errorMap;
    }
    List<OrderEntity> orderEntities = this.buildOrderEntity(dataList);
    List<OrderGrabEntity> orderGrabEntities = (List<OrderGrabEntity>) this.nebulaToolkitService.copyCollectionByBlankList(orderEntities, OrderEntity.class
        , OrderGrabEntity.class, HashSet.class, ArrayList.class);
    OrderGrabSyncModel syncModel = this.orderGrabService.buildOrderGrabSyncModel(orderGrabEntities);
    this.perfectOrder(orderEntities, syncModel);
    this.orderService.create(orderEntities);
    return Maps.newHashMap();
  }

  /**
   * 构建订货单抓单数据
   *
   * @param dataList 导入数据
   * @return 订单抓单数据
   */
  private List<OrderEntity> buildOrderEntity(List<SupermarketOrderImportVo> dataList) {
    Map<String, OrderEntity> resultMap = Maps.newHashMap();
    dataList.forEach(importVo -> {
      OrderEntity orderEntity = resultMap.get(String.format("%s-%s-%s", importVo.getDirectCode()
          , importVo.getKaStoreCode(), importVo.getKaOrderNumber()));
      if (Objects.isNull(orderEntity)) {
        orderEntity = this.nebulaToolkitService.copyObjectByWhiteList(importVo, OrderEntity.class
            , HashSet.class, ArrayList.class, "finalOrderAmountNot", "finalOrderAmount");
        orderEntity.setOrderNumber(this.generateCodeService.generateCode(OrderConstant.ORDER_CODE, 1).get(0));
        orderEntity.setGoodsList(Lists.newArrayList());
        orderEntity.setInvoicesSource(GrabType.IMPORT.getDictCode());
        orderEntity.setVersionNumber(1);
        orderEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        orderEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        orderEntity.setFinalOrderAmountNot(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getFinalOrderAmountNot(), null)));
        orderEntity.setFinalOrderAmount(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getFinalOrderAmount(), null)));
        resultMap.put(importVo.getKaOrderNumber(), orderEntity);
      }
      OrderGoodsEntity orderGoodsEntity = this.nebulaToolkitService.copyObjectByWhiteList(importVo
          , OrderGoodsEntity.class, HashSet.class, ArrayList.class
          , "curUnitOrderQuantity", "unitPrice", "orderAmount", "orderAmountNot");
      orderGoodsEntity.setKaName(orderEntity.getKaName());
      orderGoodsEntity.setDirectCode(orderEntity.getDirectCode());
      orderGoodsEntity.setVersionNumber(orderEntity.getVersionNumber());
      orderGoodsEntity.setOrderNumber(orderEntity.getOrderNumber());
      orderGoodsEntity.setKaOrderNumber(orderEntity.getKaOrderNumber());
      orderGoodsEntity.setCurUnitOrderQuantity(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getCurUnitOrderQuantity(), null)));
      orderGoodsEntity.setUnitPrice(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getUnitPrice(), null)));
      orderGoodsEntity.setOrderAmount(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getOrderAmount(), null)));
      orderGoodsEntity.setOrderAmountNot(NumberUtils.createBigDecimal(StringUtils.defaultIfBlank(importVo.getOrderAmountNot(), null)));
      orderEntity.getGoodsList().add(orderGoodsEntity);
    });
    return Lists.newArrayList(resultMap.values());
  }

  /**
   * 完善单据信息
   *
   * @param orderEntities 单据数据列表
   */
  private void perfectOrder(List<OrderEntity> orderEntities, OrderGrabSyncModel syncModel) {
    orderEntities.forEach(orderEntity -> {
      this.orderService.buildOrderInfo(orderEntity, syncModel);
      orderEntity.getGoodsList().forEach(orderGoodsEntity -> this.orderGoodsService.buildGoodsInfo(orderGoodsEntity, orderEntity, syncModel));
      this.orderService.buildOrderStatus(orderEntity);
    });
  }

  /**
   * 导入数据校验
   *
   * @param dataList 导入数据
   */
  private Map<Integer, String> validation(List<SupermarketOrderImportVo> dataList) {
    Map<Integer, String> resultMap = Maps.newHashMap();
    for (SupermarketOrderImportVo importVo : dataList) {
      try {
        Validate.notBlank(importVo.getDirectCode(), "缺失系统编号");
        Validate.notBlank(importVo.getKaName(), "缺失系统名称");
        Validate.notBlank(importVo.getKaOrderNumber(), "缺失商超订单编号");
        Validate.notBlank(importVo.getKaStoreCode(), "缺失商超门店编号");
        Validate.notBlank(importVo.getKaStoreName(), "缺失商超门店名称");
        Validate.notBlank(importVo.getOrderDate(), "缺失订单日期");
        Validate.notBlank(importVo.getFinalOrderAmount(), "缺失订货单总金额（含税）");
        Validate.notBlank(importVo.getKaGoodsCode(), "缺失商超产品编号");
        Validate.notBlank(importVo.getKaGoodsName(), "缺失商超产品名称");
        Validate.notBlank(importVo.getCurUnit(), "缺失商超单位");
        Validate.notBlank(importVo.getCurUnitOrderQuantity(), "缺失商超数量");
        Validate.notBlank(importVo.getUnitPrice(), "缺失商超订货单价");
        Validate.notBlank(importVo.getOrderAmount(), "缺失商超订货金额（含税）");
        Validate.isTrue(NumberUtils.isParsable(importVo.getFinalOrderAmount()), "订货单总金额（含税）格式错误");
        if (StringUtils.isNotBlank(importVo.getFinalOrderAmountNot())) {
          Validate.isTrue(NumberUtils.isParsable(importVo.getFinalOrderAmountNot()), "订货单总金额（不含税）格式错误");
        }
        Validate.isTrue(NumberUtils.isParsable(importVo.getCurUnitOrderQuantity()), "商超数量格式错误");
        Validate.isTrue(NumberUtils.isParsable(importVo.getUnitPrice()), "商超订货单价格式错误");
        Validate.isTrue(NumberUtils.isParsable(importVo.getOrderAmount()), "商超订货金额（含税）格式错误");
        if (StringUtils.isNotBlank(importVo.getOrderAmountNot())) {
          Validate.isTrue(NumberUtils.isParsable(importVo.getOrderAmountNot()), "商超订货金额（不含税）格式错误");
        }
      } catch (Exception e) {
        resultMap.put(dataList.indexOf(importVo), e.getMessage());
      }
    }
    return resultMap;
  }
}
