package com.biz.crm.kms.business.audit.match.local.repository;


import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.kms.business.audit.match.local.entity.AuditMatchInvoice;
import com.biz.crm.kms.business.audit.match.local.mapper.AuditMatchInvoiceMapper;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import com.biz.crm.kms.business.audit.match.sdk.vo.InvoiceReturnOrderVo;
import com.biz.crm.kms.business.audit.match.sdk.vo.MatchInvoiceAcceptance;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 稽查匹配关联单据表(AuditMatchInvoice)表数据库访问层
 *
 * @author songjingen
 * @since 2022-11-10 10:06:40
 */
@Component
public class AuditMatchInvoiceRepository extends ServiceImpl<AuditMatchInvoiceMapper, AuditMatchInvoice> {

  /**
   * 根据模板编码和单据类型集合和单据编码集合查询数据
   *
   * @param templateCode 模板编码
   * @param invoiceTypes 单据类型集合
   * @param relationIds 关联id
   * @return
   */
  public List<AuditMatchInvoice> findByTemplateCodeAndInvoiceTypesAndInvoiceCodes(String templateCode, List<String> invoiceTypes, List<String> relationIds) {
    return this.lambdaQuery().eq(AuditMatchInvoice::getTemplateCode, templateCode)
        .in(AuditMatchInvoice::getInvoiceType, invoiceTypes)
        .in(AuditMatchInvoice::getRelationId, relationIds)
        .list();
  }

  /**
   * 根据模板编码，单据类型，稽核单号查询已匹配的数据
   * @param templateCode
   * @param invoiceTypes
   * @param auditCodes
   * @return
   */
  public List<AuditMatchInvoice> findByTemplateCodeAndInvoiceTypesAndAuditCode(String templateCode, List<String> invoiceTypes, List<String> auditCodes) {
    return this.lambdaQuery().eq(AuditMatchInvoice::getTemplateCode, templateCode)
            .in(AuditMatchInvoice::getInvoiceType, invoiceTypes)
            .in(AuditMatchInvoice::getAuditCode, auditCodes)
            .list();
  }

  /**
   * 根据条件删除数据
   *
   * @param templateCodes 模板编码集合
   * @param invoiceTypes  单据类型集合
   */
  public void deleteByTemplateCodesAndInvoiceTypes(Set<String> templateCodes, Set<String> invoiceTypes) {
    this.lambdaUpdate().in(AuditMatchInvoice::getTemplateCode, templateCodes)
        .in(AuditMatchInvoice::getInvoiceType, invoiceTypes)
        .remove();
  }

  /**
   * 根据稽核编码查询数据
   *
   * @param auditCode 稽核编码
   * @return
   */
  public List<AuditMatchInvoice> findByAuditCode(String auditCode) {
    return this.lambdaQuery().eq(AuditMatchInvoice::getAuditCode, auditCode).list();
  }

  /**
   * 根据主键集合和单据类型查询数据
   *
   * @param ids
   * @return
   */
  public List<AuditMatchInvoice> findByInvoiceByIds(List<String> ids) {
    return this.lambdaQuery().in(AuditMatchInvoice::getId, ids)
        .list();
  }

  /**
   * 根据验收单号查询验收单
   * @param relationIds
   * @return
   */
  public List<MatchInvoiceAcceptance> findAcceptanceByRelationId(List<String> relationIds){
    if (!CollectionUtils.isEmpty(relationIds)){
      return baseMapper.matchInvoice(relationIds);
    }else {
     return CollectionUtil.newArrayList();
    }
  }

  /**
   * 根据退货单号查询退货单
   * @param relationIds
   * @return
   */
  public List<InvoiceReturnOrderVo> findReturnByRelationId(List<String> relationIds){
    if (!CollectionUtils.isEmpty(relationIds)){
      return baseMapper.matchInvoiceReturn(relationIds);
    }else {
      return CollectionUtil.newArrayList();
    }
  }

  /**
   * 根据稽核单号删除数据
   * @param auditCode
   */
  public void deleteByAuditCode(Set<String> auditCode){
    if (!CollectionUtils.isEmpty(auditCode)){
      this.lambdaUpdate().in(AuditMatchInvoice::getAuditCode,auditCode)
              .remove();
    }
  }
}

