package com.biz.crm.kms.business.audit.match.local.repository;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.kms.business.audit.match.local.entity.AuditMatch;
import com.biz.crm.kms.business.audit.match.local.mapper.AuditMatchMapper;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditMatchDto;
import com.biz.crm.kms.business.audit.match.sdk.vo.AuditMatchVo;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditMatchNotReasonDto;
import com.biz.crm.kms.business.audit.match.sdk.vo.AuditSapVo;
import com.biz.crm.kms.business.audit.match.sdk.vo.InvoiceReturnOrderVo;
import com.biz.crm.kms.business.audit.match.sdk.vo.MatchInvoiceAcceptance;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.List;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 稽查匹配表(AuditMatch)表数据库访问层
 *
 * @author songjingen
 * @since 2022-10-26 11:18:44
 */
@Component
public class AuditMatchRepository extends ServiceImpl<AuditMatchMapper, AuditMatch> {

  /**
   * 根据稽核单号查询数据
   *
   * @param auditCodes
   * @return
   */
  public List<AuditMatch> findByAuditCodes(List<String> auditCodes) {
    return this.lambdaQuery().eq(AuditMatch::getTenantCode, TenantUtils.getTenantCode())
        .in(AuditMatch::getAuditCode, auditCodes)
        .list();
  }

  /**
   * 根据系统编码查询数据
   *
   * @param directCodes 系统编码
   * @return 数据集合
   */
  public List<AuditMatch> findByDirectCodes(List<String> directCodes) {
    return this.lambdaQuery().eq(AuditMatch::getTenantCode, TenantUtils.getTenantCode())
        .in(AuditMatch::getDirectCode, directCodes)
        .list();
  }

  /**
   * 根据模板编码查询数据
   *
   * @param templateCode
   * @return
   */
  public List<AuditMatch> findByTemplateCode(String templateCode) {
    return this.lambdaQuery().eq(AuditMatch::getTenantCode, TenantUtils.getTenantCode())
        .eq(AuditMatch::getTemplateCode, templateCode)
        .list();
  }

  /**
   * 根据模板编码时间范围
   *
   * @param templateCode
   * @return
   */
  public List<AuditMatch> findByTemplateCodeAndDate(String templateCode,String beginDate,String endDate) {
    return this.lambdaQuery().eq(AuditMatch::getTenantCode, TenantUtils.getTenantCode())
            .eq(AuditMatch::getTemplateCode, templateCode)
            .between(AuditMatch::getSapPostingDate,beginDate,endDate)
            .list();
  }

  /**
   * 根据模板编码集合查询稽核数据
   * @param templateCodes
   * @param matchStatus
   * @return
   */
  public List<AuditMatch> findByTemplateCodes(List<String> templateCodes,String matchStatus){
    return this.lambdaQuery().eq(AuditMatch::getTenantCode, TenantUtils.getTenantCode())
            .eq(AuditMatch::getMatchStatus,matchStatus)
            .in(AuditMatch::getTemplateCode, templateCodes)
            .list();
  }

  /**
   * 根据模板编码和系统编码查询数据
   *
   * @param templateCode
   * @return
   */
  public List<AuditMatch> findByTemplateCodeAndDirectCode(String templateCode,String directCode) {
    return this.lambdaQuery().eq(AuditMatch::getTenantCode, TenantUtils.getTenantCode())
            .eq(AuditMatch::getTemplateCode, templateCode)
            .eq(AuditMatch::getDirectCode, directCode)
            .list();
  }

  /**
   * 查询未匹配验收单
   * @param pageable
   * @param directCode
   * @param storeCodes
   * @param productCodes
   * @param startTime
   * @param endTime
   * @param orderNumber
   * @param orderCodes
   * @param soldPartyCodes
   * @return
   */
  public List<MatchInvoiceAcceptance> notMatchInvoice(Pageable pageable, String directCode, List<String> storeCodes,
                                                      List<String> productCodes, String startTime,
                                                      String endTime, String orderNumber,
                                                      List<String> orderCodes, List<String> soldPartyCodes) {
    Page<MatchInvoiceAcceptance> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    if (!StringUtils.isNotEmpty(directCode)) {
      return Lists.newArrayList();
    } else {
      return this.baseMapper.notMatchInvoice(page, directCode, storeCodes, productCodes, startTime, endTime, orderNumber,orderCodes,soldPartyCodes);
    }
  }

  /**
   * 查询未匹配退货单
   * @param directCode
   * @param soldToPartyCode
   * @param storeCode
   * @return
   */
  public List<InvoiceReturnOrderVo> notMatchInvoiceReturn(Pageable pageable,String directCode, String soldToPartyCode, String storeCode){
    Page<InvoiceReturnOrderVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.notMatchInvoiceReturn(page,directCode, soldToPartyCode, storeCode);
  }

  /**
   * 更新稽核数据匹配状态
   * @param auditCodes
   * @param matchStatus
   */
  public void updateAuditMatchStatus(List<String> auditCodes,String matchStatus){
    if (CollectionUtils.isEmpty(auditCodes) || StringUtils.isEmpty(matchStatus)){
      return;
    }else {
      this.lambdaUpdate().in(AuditMatch::getAuditCode,auditCodes).eq(AuditMatch::getTenantCode,TenantUtils.getTenantCode()).set(AuditMatch::getMatchStatus,matchStatus).set(AuditMatch::getNotMatchReason,null).update();
    }
  }

  /**
   * 分页查询数据
   * @param pageable
   * @param auditMatchDto
   * @return
   */
  public Page<AuditMatch> findPageByConditions(Pageable pageable, AuditMatchDto auditMatchDto) {
    Page<AuditMatch> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<AuditMatch> pageList = this.baseMapper.findPageByConditions(page, auditMatchDto);
    return pageList;
  }

  public Page<AuditMatchVo> findAuditMatchVoPageByConditions(Pageable pageable, AuditMatchDto auditMatchDto) {
    Page<AuditMatchVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<AuditMatchVo> pageList = this.baseMapper.findAuditMatchVoPageByConditions(page, auditMatchDto);
    return pageList;
  }

  /**
   * 更新未匹配原因
   * @param auditCode
   * @param reason
   */
  public void updateAuditNotReason(String auditCode,String reason) {
    if (StringUtils.isNotEmpty(auditCode) && StringUtils.isNotEmpty(reason)) {
      this.lambdaUpdate().eq(AuditMatch::getAuditCode, auditCode).eq(AuditMatch::getTenantCode,TenantUtils.getTenantCode()).set(AuditMatch::getNotMatchReason, reason).update();
    } else {
      return;
    }
  }

  /**
   * 根据校验字段更新采购单号
   * @param verifyCode
   * @param relationOrderCode
   */
  public void updateAuditRelationOrderCode(String verifyCode,String relationOrderCode){
    if (StringUtils.isNotEmpty(verifyCode) && StringUtils.isNotEmpty(relationOrderCode)){
      this.lambdaUpdate().eq(AuditMatch::getTenantCode,TenantUtils.getTenantCode())
              .eq(AuditMatch::getVerifyCode,verifyCode)
              .set(AuditMatch::getRelationOrderCode,relationOrderCode).update();
    } else {
      return;
    }
  }

  /**
   * 根据id集合查询数据
   * @param ids
   * @return
   */
  public List<AuditMatch> findAuditMatchByIds(List<String> ids){
    if (!CollectionUtils.isEmpty(ids)){
      return this.lambdaQuery().in(AuditMatch::getId,ids).eq(AuditMatch::getTenantCode,TenantUtils.getTenantCode()).list();
    }else {
      return Lists.newArrayList();
    }
  }
}

