package com.biz.crm.kms.business.audit.match.local.repository;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.kms.business.audit.match.local.entity.AuditSapEntity;
import com.biz.crm.kms.business.audit.match.local.mapper.AuditSapMapper;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditSapConditionDto;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditSapDto;
import com.biz.crm.kms.business.audit.match.sdk.dto.SurplusFeePoolBalanceAuditMatchDto;
import com.biz.crm.kms.business.audit.match.sdk.enums.CreateAuditMatchEnum;
import com.biz.crm.kms.business.audit.match.sdk.enums.SapOrderTypeEnum;
import com.biz.crm.kms.business.audit.match.sdk.enums.SapTransStatusEnum;
import com.biz.crm.kms.business.audit.match.sdk.vo.AuditSapVo;
import com.biz.crm.kms.business.audit.match.sdk.vo.SurplusFeePoolBalanceAuditMatchVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

/**
 * @author cyj
 * @version 1.0
 * @date 2022/12/19 17:57
 * @description AuditSapRepository
 */
@Component
public class AuditSapRepository extends ServiceImpl<AuditSapMapper, AuditSapEntity> {

    @Autowired(required = false)
    private AuditSapMapper auditSapMapper;
    /**
     * 根据Sap单号查询sap数据
     * @param sapCode
     * @return
     */
    public List<AuditSapVo> findBySapCode (String sapCode, String materialCode){
        if (StringUtils.isNotEmpty(sapCode) && StringUtils.isNotEmpty(materialCode)){
            return this.baseMapper.findBySalesDeliveryNoAndMaterialCode(sapCode,materialCode);
        }else {
            return CollectionUtil.newArrayList();
        }
    }

    /**
     * 根据Sap校验字段查询sap数据
     * @param verifyCode
     * @return
     */
    public List<AuditSapVo> findByVerifyCode (String verifyCode){
        if (StringUtils.isNotEmpty(verifyCode)){
            return this.baseMapper.findByVerifyCode(verifyCode);
        }else {
            return CollectionUtil.newArrayList();
        }
    }


    /**
     * 条件查询SAP数据
     * @param sapDateStart
     * @param sapDateEnd
     * @param deliveryCodes
     * @return
     */
    public List<AuditSapEntity> findByConditions(String sapDateStart,String sapDateEnd,List<String> deliveryCodes,String saleVouType,String transStatus,String createStatus){
        if (StringUtils.isEmpty(sapDateEnd) && StringUtils.isEmpty(sapDateStart) && CollectionUtil.isEmpty(deliveryCodes)){
            return CollectionUtil.newArrayList();
        }else {
            return this.lambdaQuery().between(AuditSapEntity::getShippingAccountDate,sapDateStart,sapDateEnd)
                    .eq(AuditSapEntity::getSaleVouType,saleVouType)
                    .in(CollectionUtil.isNotEmpty(deliveryCodes),AuditSapEntity::getServiceCode,deliveryCodes)
                    .eq(StringUtils.isNotBlank(transStatus),AuditSapEntity::getTransStatus,transStatus)
                    .eq(AuditSapEntity::getTenantCode, TenantUtils.getTenantCode())
                    .eq(StringUtils.equals(CreateAuditMatchEnum.GOODSMOVESTATUS.getDictCode(),createStatus),AuditSapEntity::getGoodsMoveStatus,"C")
                    .eq(StringUtils.equals(CreateAuditMatchEnum.INVOICESTATUS.getDictCode(),createStatus),AuditSapEntity::getInvoiceStatus,"C")
                    .list();
        }
    }

    /**
     * 根据订单类型和销售单号查询数据
     * @param saleVouType
     * @param salesDeliveryNos
     * @return
     */
    public List<AuditSapEntity> findSapByTypeAndSalesDeliveryNo(String saleVouType,List<String> salesDeliveryNos){
        if (StringUtils.isEmpty(saleVouType) || CollectionUtil.isEmpty(salesDeliveryNos)){
            return Lists.newArrayList();
        }else {
            return this.lambdaQuery()
                    .eq(AuditSapEntity::getTenantCode,TenantUtils.getTenantCode())
                    .eq(AuditSapEntity::getSaleVouType,saleVouType)
                    .in(AuditSapEntity::getSalesDeliveryNo,salesDeliveryNos)
                    .list();
        }
    }

    /**
     * 根据订单类型和采购单号查询数据
     * @param saleVouType
     * @param purchaseNos
     * @return
     */
    public List<AuditSapEntity> findReturnOrderByTypeAndPurchaseNo(String saleVouType,List<String> purchaseNos){
        if (StringUtils.isEmpty(saleVouType) || CollectionUtil.isEmpty(purchaseNos)){
            return Lists.newArrayList();
        }else {
            return this.lambdaQuery()
                    .eq(AuditSapEntity::getTenantCode,TenantUtils.getTenantCode())
                    .eq(AuditSapEntity::getSaleVouType,saleVouType)
                    .in(AuditSapEntity::getPurchaseOrderNo,purchaseNos)
                    .list();
        }
    }

    /**
     * 根据条件分页查询Sap数据
     * @param pageable
     * @param transStatus
     * @return
     */
    public Page<String> findSapPageByTransStatus(Pageable pageable,String transStatus){
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        Page<AuditSapEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findSapPageByTransStatus(page, transStatus,TenantUtils.getTenantCode());
    }

    public void batchSave(Collection<AuditSapEntity> list) {
        if(CollectionUtil.isNotEmpty(list)){
            this.baseMapper.batchSave(list);
        }
    }

    /**
     * 根据sap单号修改状态
     * @param verifyCodes
     * @param status
     */
    public void updateStatus(List<String> verifyCodes, String status) {
        if (CollectionUtil.isNotEmpty(verifyCodes) && StringUtils.isNotEmpty(status)) {
            this.lambdaUpdate()
                    .in(AuditSapEntity::getVerifyCode,verifyCodes)
                    .set(AuditSapEntity::getTransStatus, status).update();
        }
    }

    /**
     * 根据sap单号匹配退货单订单抬头查询退货单
     * @param salesDeliveryCode
     * @return
     */
    public List<AuditSapEntity> findReturnOrderBySalesDeliveryCode(String salesDeliveryCode) {
        if (StringUtils.isNotEmpty(salesDeliveryCode)) {
            return this.lambdaQuery().like(AuditSapEntity::getOrderHeader, salesDeliveryCode)
                    .eq(AuditSapEntity::getTenantCode, TenantUtils.getTenantCode())
                    .eq(AuditSapEntity::getSaleVouType, SapOrderTypeEnum.ZRE2.getDictCode()).list();
        } else {
            return Lists.newArrayList();
        }
    }

    /**
     * 根据条件找退货单
     * @param materialCode
     * @param beginDate
     * @param endDate
     * @return
     */
    public List<AuditSapEntity> findReturnOrderByMaterialCodeAndDate(List<String> materialCode,String beginDate,String endDate) {
        if (CollectionUtils.isNotEmpty(materialCode) && StringUtils.isNotEmpty(beginDate) && StringUtils.isNotEmpty(endDate)) {
            return this.lambdaQuery().in(AuditSapEntity::getMaterialCode,materialCode)
                    .between(AuditSapEntity::getDs,beginDate,endDate)
                    .eq(AuditSapEntity::getTenantCode, TenantUtils.getTenantCode())
                    .eq(AuditSapEntity::getSaleVouType, SapOrderTypeEnum.ZRE2.getDictCode())
                    .list();
        } else {
            return Lists.newArrayList();
        }
    }

    public List<AuditSapEntity> findByVerifyCodes(List<String> verifyCodeList) {
        if (CollectionUtils.isEmpty(verifyCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(AuditSapEntity::getVerifyCode, verifyCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 查询105数据数量
     * @param salesDeliveryNos
     * @return
     */
    public Integer findCountBySalesDeliveryNo(List<String> salesDeliveryNos){
        if (CollectionUtil.isNotEmpty(salesDeliveryNos)){
            return this.lambdaQuery().in(AuditSapEntity::getSalesDeliveryNo,salesDeliveryNos).list().size();
        }else {
            return null;
        }
    }

    /**
     * 条件查询数据
     *
     * @param dto
     * @return
     */
    public List<AuditSapEntity> findByCondition(AuditSapConditionDto dto) {
        if (CollectionUtils.isEmpty(dto.getCodeList()) || StringUtils.isEmpty(dto.getDs())) {
            return new ArrayList<>();
        }
        return auditSapMapper.findByCondition(dto);
    }

    /**
     * 分页查询数据
     * @param pageable 分页对象
     * @param auditSapDto 实体对象
     * @return
     */
    public Page<AuditSapVo> findPageByConditions(Pageable pageable, AuditSapDto auditSapDto) {
        Page<AuditSapVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<AuditSapVo> pageList = this.baseMapper.findPageByConditions(page, auditSapDto);
        return pageList;
    }

    /**
     * 批量保存数据
     *
     * @param saveList
     */
    public void saveBatchData(Collection<AuditSapEntity> saveList) {
        auditSapMapper.saveBatchData(saveList);
    }

    /**
     * 批量更新数据
     * @param updateList
     * @param tenantCode
     */
    public void updateBatchDate(Collection<AuditSapEntity> updateList,String tenantCode){
        if (CollectionUtils.isEmpty(updateList)){
            return;
        }
        if (StringUtils.isEmpty(tenantCode)){
            return;
        }
        updateList.forEach(item ->{
            if (StringUtils.isNotEmpty(item.getId())){
                LambdaUpdateWrapper<AuditSapEntity> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
                lambdaUpdateWrapper.eq(AuditSapEntity::getTenantCode,tenantCode);
                lambdaUpdateWrapper.eq(AuditSapEntity::getId,item.getId());
                this.baseMapper.update(item,lambdaUpdateWrapper);
            }
        });
    }

    /**
     * 折扣率变量剩余费用池余额 统计
     *
     * @param dto 统计参数
     * @return List<SurplusFeePoolBalanceVo>
     */
    public List<SurplusFeePoolBalanceAuditMatchVo> surplusFeePoolBalance(SurplusFeePoolBalanceAuditMatchDto dto) {
        if (Objects.isNull(dto.getYearAndMonth()) || CollectionUtils.isEmpty(dto.getServiceCodeList())) {
            return new ArrayList<>();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        List<SurplusFeePoolBalanceAuditMatchVo> list = this.auditSapMapper.surplusFeePoolBalance(dto);
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        return list;
    }

    /**
     * 根据交货单编码和物料编码查询105数据
     * @param salesDeliveryCodes
     * @param materialCodes
     * @return
     */
    public List<AuditSapEntity> findSapEntityByCodeAndMaterialCode(List<String> salesDeliveryCodes,List<String> materialCodes){
        if (CollectionUtils.isNotEmpty(salesDeliveryCodes) && CollectionUtil.isNotEmpty(materialCodes)){
            return this.lambdaQuery().in(AuditSapEntity::getSalesDeliveryNo,salesDeliveryCodes).in(AuditSapEntity::getMaterialCode,materialCodes).eq(AuditSapEntity::getTenantCode,TenantUtils.getTenantCode()).list();
        }else {
            return Lists.newArrayList();
        }
    }

    /**
     * 根据校验字段更新转换状态
     * @param verifyCodes
     */
    public void updateByVerifyCodes(List<String> verifyCodes,String status) {
        if (CollectionUtil.isNotEmpty(verifyCodes)) {
            this.lambdaUpdate().in(AuditSapEntity::getVerifyCode, verifyCodes).eq(AuditSapEntity::getTenantCode, TenantUtils.getTenantCode()).set(AuditSapEntity::getTransStatus, status).update();
        }else {
            return;
        }
    }

    /**
     * 根据校验字段更新是否匹配
     * @param verifyCodes
     * @param status
     */
    public void updateIsMatchedByVerifyCodes(List<String> verifyCodes,String status){
        if (CollectionUtil.isNotEmpty(verifyCodes)) {
            this.lambdaUpdate().in(AuditSapEntity::getVerifyCode, verifyCodes).eq(AuditSapEntity::getTenantCode, TenantUtils.getTenantCode()).set(AuditSapEntity::getIsMatched, status).update();
        }else {
            return;
        }
    }
}
