package com.biz.crm.kms.business.audit.match.local.service.internal;


import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.kms.business.audit.match.local.entity.*;
import com.biz.crm.kms.business.audit.match.local.model.MatchConsequenceCalculatedAbstract;
import com.biz.crm.kms.business.audit.match.local.model.MatchConsequenceCalculatedModel;
import com.biz.crm.kms.business.audit.match.local.repository.*;
import com.biz.crm.kms.business.audit.match.local.service.AuditMatchInvoiceService;
import com.biz.crm.kms.business.audit.match.local.service.AuditMatchService;
import com.biz.crm.kms.business.audit.match.local.service.AuditMatchThreadService;
import com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant;
import com.biz.crm.kms.business.audit.match.sdk.dto.*;
import com.biz.crm.kms.business.audit.match.sdk.enums.*;
import com.biz.crm.kms.business.audit.match.sdk.listener.AuditMatchLogEventListener;
import com.biz.crm.kms.business.audit.match.sdk.listener.KmsAuditTemplateLogEventListener;
import com.biz.crm.kms.business.audit.match.sdk.register.InvoiceMatchRegister;
import com.biz.crm.kms.business.audit.match.sdk.vo.AuditSapVo;
import com.biz.crm.kms.business.audit.match.sdk.vo.InvoiceVo;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.sdk.enums.ConstantEnums;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant.KMS_AUDIT_SAP_LOOP_MAX;
import static com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant.KMS_AUDIT_SAP_PAGE_SIZE;

/**
 * 稽查匹配表(AuditMatch)表服务实现类
 *
 * @author songjingen
 * @since 2022-10-26 11:18:44
 */
@Slf4j
@Service("auditMatchService")
public class AuditMatchServiceImpl implements AuditMatchService {

    @Autowired
    private AuditMatchRepository auditMatchRepository;

    @Autowired
    private AuditMatchConsequenceRepository auditMatchConsequenceRepository;

    @Autowired(required = false)
    private AuditTemplateSupermarketRepository auditTemplateSupermarketRepository;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private List<InvoiceMatchRegister> invoiceMatchRegisters;

    @Autowired(required = false)
    private AuditTemplateRepository auditTemplateRepository;

    @Autowired(required = false)
    private ApplicationContext applicationContext;

    @Autowired
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired
    private AuditMatchThreadService auditMatchThreadService;

    @Autowired
    private AuditMatchInvoiceService auditMatchInvoiceService;

    @Autowired
    private AuditSapRepository auditSapRepository;

    @Autowired
    private DirectVoService directVoService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private RedisCrmService redisCrmService;

    @Autowired(required = false)
    private DirectStoreVoService directStoreVoService;

    @Autowired(required = false)
    private DirectProductVoService directProductVoService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private MatchInvoiceManuRepository matchInvoiceManuRepository;

    @Autowired(required = false)
    private RedisService redisService;

    /**
     * 订单抬头匹配采购单号常量
     */
    private static String rgx = "^[A-Za-z0-9]+$";

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public AuditMatch findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        AuditMatch auditMatch = this.auditMatchRepository.getById(id);
        if (auditMatch == null) {
            return null;
        }
        List<AuditMatchConsequence> auditMatchConsequences = this.auditMatchConsequenceRepository.findByAuditCodes(Lists.newArrayList(auditMatch.getAuditCode()));
        if (!CollectionUtils.isEmpty(auditMatchConsequences)) {
            auditMatch.setMatchConsequences(auditMatchConsequences.get(0));
        }
        //查询已匹配的单据
        InvoiceVo invoiceByAuditCode = auditMatchInvoiceService.findInvoiceByAuditCode(auditMatch.getAuditCode());
        auditMatch.setInvoiceVo(invoiceByAuditCode);
        return auditMatch;
    }

    @Override
    public void updateAuditStatusByAuditMatchQueryDto(AuditMatchQueryDto dto) {
        Validate.isTrue(!CollectionUtils.isEmpty(this.invoiceMatchRegisters), "触发执行模板匹配任务时，未查询到单据匹配注册器！");
        Validate.notNull(dto, "手动匹配时，查询条件不存在！");
        Validate.notNull(dto.getTemplateCode(), "手动匹配时，模板编码不存在！");
        AuditTemplate auditTemplate = this.auditTemplateRepository.findByTemplateCode(dto.getTemplateCode());
        Validate.notNull(dto.getTemplateCode(), "手动匹配时，未查询到模板！");

        Validate.isTrue(!CollectionUtils.isEmpty(dto.getAuditCodes()), "手动匹配时，稽核编码不存在！");
        List<AuditMatch> auditMatches = this.auditMatchRepository.findByAuditCodes(dto.getAuditCodes());
        Validate.isTrue(!CollectionUtils.isEmpty(auditMatches), "手动匹配时，未查询到匹配数据！");
        List<String> directCodes = auditMatches.stream().map(AuditMatch::getDirectCode).collect(Collectors.toList());
        List<AuditTemplateSupermarket> templateSupermarkets = this.auditTemplateSupermarketRepository.findByTemplateCodeAndDirectCodes(dto.getTemplateCode(), directCodes);
        Map<String, AuditTemplateSupermarket> supermarketMap = new HashMap<>();
        if (CollectionUtils.isEmpty(templateSupermarkets)) {
            supermarketMap = templateSupermarkets.stream().collect(Collectors.toMap(AuditTemplateSupermarket::getDirectCode, o -> o));
        }
        //组装成执行所需要的数据
        List<MatchConsequenceCalculatedModel> calculatedModels = new ArrayList<>();
        List<String> auditCodes = new ArrayList<>();
        for (AuditMatch match : auditMatches) {
            MatchConsequenceCalculatedModel calculatedModel = this.nebulaToolkitService.copyObjectByWhiteList(match, MatchConsequenceCalculatedModel.class, HashSet.class, ArrayList.class);
            AuditTemplateSupermarket supermarket = supermarketMap.get(match.getDirectCode());
            if (supermarket != null) {
                calculatedModel.setPriceValue(supermarket.getPriceValue());
                calculatedModel.setPriceType(supermarket.getPriceType());
            }
            calculatedModels.add(calculatedModel);
            auditCodes.add(match.getAuditCode());
        }
        //查询出已经匹配的数据然后赋值id
        List<AuditMatchConsequence> consequences = this.auditMatchConsequenceRepository.findByAuditCodes(dto.getAuditCodes());
        if (!CollectionUtils.isEmpty(consequences)) {
            Map<String, String> auditMatchConsequenceMap = consequences.stream()
                    .collect(Collectors.toMap(AuditMatchConsequence::getAuditCode, AuditMatchConsequence::getId, (a, b) -> a));
            calculatedModels.stream().forEach(o -> o.setId(auditMatchConsequenceMap.get(o.getAuditCode())));
        }
        //调用匹配策略
        for (InvoiceMatchRegister invoiceMatchRegister : this.invoiceMatchRegisters) {
            if (invoiceMatchRegister.getMatchLogicCode().equals(auditTemplate.getMatchLogicCode())) {
                Class<?> matchLogicCalculatedStrategy = invoiceMatchRegister.getMatchLogicCalculatedAbstract();
                MatchConsequenceCalculatedAbstract bean = (MatchConsequenceCalculatedAbstract) this.applicationContext.getBean(matchLogicCalculatedStrategy);
                bean.execute(dto.getTemplateCode(), calculatedModels);
                break;
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void auditMatch(List<AuditMatchInvoice> auditMatchInvoices,String auditCode) {
        if (CollectionUtils.isEmpty(auditMatchInvoices) && StringUtils.isEmpty(auditCode)) {
            return;
        }
        if (!CollectionUtils.isEmpty(auditMatchInvoices) && StringUtils.isEmpty(auditCode)){
            Validate.isTrue(!CollectionUtils.isEmpty(this.invoiceMatchRegisters), "触发执行模板匹配任务时，未查询到单据匹配注册器！");
            AuditMatchInvoice invoice = auditMatchInvoices.get(0);
            //查询稽核单
            List<String> auditCodes = new ArrayList<>();
            auditCodes.add(invoice.getAuditCode());
            List<AuditMatch> auditMatches = this.auditMatchRepository.findByAuditCodes(auditCodes);
            Validate.isTrue(!CollectionUtils.isEmpty(auditMatches), "手动匹配时，未查询到匹配数据!");
            //查询模板
            AuditTemplate auditTemplate = this.auditTemplateRepository.findByTemplateCode(invoice.getTemplateCode());
            Validate.isTrue(Objects.nonNull(auditTemplate), "手动匹配时，未查询到模板数据！");
            //根据模板查询商超
            List<String> auditTemplateCodeList = new ArrayList<>();
            auditTemplateCodeList.add(auditTemplate.getTemplateCode());
            List<AuditTemplateSupermarket> templateSupermarkets = this.auditTemplateSupermarketRepository.findByTemplateCodes(auditTemplateCodeList);
            Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "未查询到模板编码（任务编码）：【%s】的商超数据！", auditTemplateCodeList.toString());
            Map<String, AuditTemplateSupermarket> supermarketMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(templateSupermarkets)) {
                supermarketMap = templateSupermarkets.stream().collect(Collectors.toMap(AuditTemplateSupermarket::getDirectCode, o -> o));
            }
            //组装成执行所需要的数据
            List<MatchConsequenceCalculatedModel> calculatedModels = new ArrayList<>();
            for (AuditMatch match : auditMatches) {
                MatchConsequenceCalculatedModel calculatedModel = this.nebulaToolkitService.copyObjectByWhiteList(match, MatchConsequenceCalculatedModel.class, HashSet.class, ArrayList.class);
                AuditTemplateSupermarket supermarket = supermarketMap.get(match.getDirectCode());
                if (supermarket != null) {
                    calculatedModel.setPriceValue(supermarket.getPriceValue());
                    calculatedModel.setPriceType(supermarket.getPriceType());
                }
                calculatedModels.add(calculatedModel);
            }
            //调用匹配策略
            for (InvoiceMatchRegister invoiceMatchRegister : this.invoiceMatchRegisters) {
                if (invoiceMatchRegister.getMatchLogicCode().equals(auditTemplate.getMatchLogicCode())) {
                    Class<?> matchLogicCalculatedStrategy = invoiceMatchRegister.getMatchLogicCalculatedAbstract();
                    MatchConsequenceCalculatedAbstract bean = (MatchConsequenceCalculatedAbstract) this.applicationContext.getBean(matchLogicCalculatedStrategy);
                    bean.notAutoExecute(invoice.getTemplateCode(), calculatedModels.get(0), auditMatchInvoices);
                    break;
                }
            }
        }
        if (CollectionUtils.isEmpty(auditMatchInvoices) && StringUtils.isNotEmpty(auditCode)){
            Validate.isTrue(!CollectionUtils.isEmpty(this.invoiceMatchRegisters), "触发执行模板匹配任务时，未查询到单据匹配注册器！");
            //查询稽核单
            List<String> auditCodes = new ArrayList<>();
            auditCodes.add(auditCode);
            List<AuditMatch> auditMatches = this.auditMatchRepository.findByAuditCodes(auditCodes);
            Validate.isTrue(!CollectionUtils.isEmpty(auditMatches), "手动匹配时，未查询到匹配数据!");
            //查询模板
            AuditTemplate auditTemplate = this.auditTemplateRepository.findByTemplateCode(auditMatches.get(0).getTemplateCode());
            Validate.isTrue(Objects.nonNull(auditTemplate), "手动匹配时，未查询到模板数据！");
            //根据模板查询商超
            List<String> auditTemplateCodeList = new ArrayList<>();
            auditTemplateCodeList.add(auditTemplate.getTemplateCode());
            List<AuditTemplateSupermarket> templateSupermarkets = this.auditTemplateSupermarketRepository.findByTemplateCodes(auditTemplateCodeList);
            Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "未查询到模板编码（任务编码）：【%s】的商超数据！", auditTemplateCodeList.toString());
            Map<String, AuditTemplateSupermarket> supermarketMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(templateSupermarkets)) {
                supermarketMap = templateSupermarkets.stream().collect(Collectors.toMap(AuditTemplateSupermarket::getDirectCode, o -> o));
            }
            //组装成执行所需要的数据
            List<MatchConsequenceCalculatedModel> calculatedModels = new ArrayList<>();
            for (AuditMatch match : auditMatches) {
                MatchConsequenceCalculatedModel calculatedModel = this.nebulaToolkitService.copyObjectByWhiteList(match, MatchConsequenceCalculatedModel.class, HashSet.class, ArrayList.class);
                AuditTemplateSupermarket supermarket = supermarketMap.get(match.getDirectCode());
                if (supermarket != null) {
                    calculatedModel.setPriceValue(supermarket.getPriceValue());
                    calculatedModel.setPriceType(supermarket.getPriceType());
                }
                calculatedModels.add(calculatedModel);
            }
            //调用匹配策略
            for (InvoiceMatchRegister invoiceMatchRegister : this.invoiceMatchRegisters) {
                if (invoiceMatchRegister.getMatchLogicCode().equals(auditTemplate.getMatchLogicCode())) {
                    Class<?> matchLogicCalculatedStrategy = invoiceMatchRegister.getMatchLogicCalculatedAbstract();
                    MatchConsequenceCalculatedAbstract bean = (MatchConsequenceCalculatedAbstract) this.applicationContext.getBean(matchLogicCalculatedStrategy);
                    bean.notAutoExecute(auditMatches.get(0).getTemplateCode(), calculatedModels.get(0), null);
                    break;
                }
            }
        }
    }

    /**
     * 自动稽核匹配
     */
    @Override
    public void autoAuditMatch() {
        Validate.isTrue(!CollectionUtils.isEmpty(this.invoiceMatchRegisters), "未查询到单据匹配注册器！");
        //获取全部模板
        List<AuditTemplate> auditTemplateList = this.auditTemplateRepository.listAuditTemplate();
        List<String> auditTemplateCodeList = auditTemplateList.stream().map(AuditTemplate::getTemplateCode).collect(Collectors.toList());
        //获取模板下的商超
        List<AuditTemplateSupermarket> templateSupermarkets = this.auditTemplateSupermarketRepository.findByTemplateCodes(auditTemplateCodeList);
        Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "未查询到模板编码（任务编码）：【%s】的商超数据！", auditTemplateCodeList.toString());
        Map<String, List<AuditTemplateSupermarket>> supermarketMap = templateSupermarkets.stream().collect(Collectors.groupingBy(AuditTemplateSupermarket::getTemplateCode));
        //组装数据
        for (AuditTemplate auditTemplate : auditTemplateList) {
            if (supermarketMap.containsKey(auditTemplate.getTemplateCode())) {
                auditTemplate.setTemplateSupermarkets(supermarketMap.get(auditTemplate.getTemplateCode()));
            }
        }
        //发送MQ消息进行处理
        log.info("=====>    稽核自动匹配 start   <=====");
        boolean lock = redisMutexService.tryLock(AuditMatchConstant.KMS_AUDIT_REDIS_LOCK, TimeUnit.SECONDS, AuditMatchConstant.KMS_AUDIT_REDIS_TIME);
        if (!lock) {
            log.warn("稽核自动匹配中,本次自动匹配被忽略!");
            log.info("=====>    稽核自动匹配 end   <=====");
            return;
        }
        Set<String> templateCodeSet = new HashSet<>(auditTemplateCodeList.size());
        try {
            templateCodeSet.addAll(auditTemplateCodeList);
            this.sendMqMessage(auditTemplateList);
        } catch (Exception e) {
            log.error("", e);
            if (CollectionUtil.isNotEmpty(templateCodeSet)) {
                //异常时,删除当前加锁redis
                redisCrmService.hdel(AuditMatchConstant.KMS_AUDIT_REDIS_LOCK, templateCodeSet.toArray());
            }
        } finally {
            redisMutexService.unlock(AuditMatchConstant.KMS_AUDIT_REDIS_LOCK);
        }
        log.info("=====>    稽核自动匹配 end   <=====");
    }

    @Override
    public void autoAuditMatchTest(List<String> auditCodes) {
        Validate.isTrue(!CollectionUtils.isEmpty(this.invoiceMatchRegisters), "未查询到单据匹配注册器！");
        //获取稽核单
        List<AuditMatch> auditMatchList = auditMatchRepository.findByAuditCodes(auditCodes);
        Validate.isTrue(!CollectionUtils.isEmpty(auditMatchList),"未能根据编码查询到稽核单!");
        //获取模板
        List<String> auditTemplateCodes = auditMatchList.stream().map(AuditMatch::getTemplateCode).distinct().collect(Collectors.toList());
        Validate.isTrue(auditTemplateCodes.size() == 1,"请选择单一模板编码的稽核单!");
        AuditMatch auditMatch = auditMatchList.get(0);
        AuditTemplate auditTemplate = this.auditTemplateRepository.findByTemplateCode(auditMatch.getTemplateCode());
        //获取模板下的商超
        List<AuditTemplateSupermarket> templateSupermarkets = this.auditTemplateSupermarketRepository.findByTemplateCodes(Lists.newArrayList(auditMatch.getTemplateCode()));
        Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "未能根据模板编码查询到商超模板");
        //组装数据
        auditTemplate.setTemplateSupermarkets(templateSupermarkets);
        List<AuditTemplate>  auditTemplateList = new ArrayList<>();
        auditTemplateList.add(auditTemplate);
        this.handleMatchDate(auditTemplateList,auditCodes);
    }

    /**
     * 自动稽核发送 MQ消息
     *
     * @param auditTemplateList 模板集合
     */
    private void sendMqMessage(List<AuditTemplate> auditTemplateList) {
        if (CollectionUtils.isEmpty(auditTemplateList)) {
            return;
        }
        //过滤模板号为空的数据
        auditTemplateList = auditTemplateList.stream()
                .filter(entity -> StringUtils.isNotEmpty(entity.getTemplateCode()))
                .distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(auditTemplateList)) {
            log.error("稽核模板有空模板编号[template_code]存在!");
            return;
        }
        //过滤正在匹配的稽核模板编码
        this.filterSwitchIngOrderNumberList(auditTemplateList);
        if (CollectionUtil.isEmpty(auditTemplateList)) {
            return;
        }

        //发送MQ消息开始匹配稽核
        MqMessageVo message = new MqMessageVo();
        message.setMsgBody(JSON.toJSONString(auditTemplateList));
        message.setTopic(AuditMatchConstant.KMS_AUDIT_TOPIC + RocketMqUtil.mqEnvironment());
        message.setTag(AuditMatchConstant.KMS_AUDIT_MESSAGE_TAG);
        rocketMqProducer.sendMqMsg(message);
        try {
            //单位：毫秒 防止MQ消息发送过于频繁
            Thread.sleep(200);
        } catch (Exception e) {
            log.error("", e);
        }
    }

    /**
     * 过滤正在转换的稽核模板
     *
     * @param auditTemplateList 稽核模板集合
     */
    private void filterSwitchIngOrderNumberList(List<AuditTemplate> auditTemplateList) {
        if (CollectionUtil.isEmpty(auditTemplateList)) {
            return;
        }
        //获取redis已存在key
        List<Object> updateList = redisCrmService.hmget(AuditMatchConstant.KMS_AUDIT_REDIS_LOCK, new HashSet<>(auditTemplateList));
        Map<String, AuditTemplate> auditTemplateMap = auditTemplateList.stream().collect(Collectors.toMap(AuditTemplate::getTemplateCode, Function.identity()));
        if (CollectionUtil.isNotEmpty(updateList)) {
            auditTemplateMap.remove(updateList.stream()
                    .filter(Objects::nonNull)
                    .filter(k -> StringUtils.isNotEmpty(k.toString()))
                    .map(Object::toString)
                    .collect(Collectors.toList()));
        }

        Map<String, String> redisMap = auditTemplateMap.keySet().stream()
                .collect(Collectors.toMap(key -> key, key -> key, (oldValue, newValue) -> newValue));

        //当前执行效率很高,锁定30分钟即可
        redisCrmService.hmset(AuditMatchConstant.KMS_AUDIT_REDIS_LOCK, redisMap, TimeUnit.SECONDS, AuditMatchConstant.KMS_AUDIT_REDIS_TIME);
    }


    /**
     * MQ接收消息，处理转换
     */
    @Override
    public void handleMatchDate(List<AuditTemplate> auditTemplateList,List<String> auditCodes) {
        if (CollectionUtils.isEmpty(auditTemplateList)) {
            log.warn("未接收到稽核模板!");
            return;
        }
        //按模板查询稽核数据
        List<AuditMatch> auditMatches = new ArrayList<>();
        List<String> templateCodes = auditTemplateList.stream().map(AuditTemplate::getTemplateCode).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(auditCodes)){
            auditMatches = this.auditMatchRepository.findByAuditCodes(auditCodes);
        }else {
            auditMatches = this.auditMatchRepository.findByTemplateCodes(templateCodes,MatchStatusEnum.M100.getDictCode());
        }
        Map<String, AuditTemplate> templateMap = auditTemplateList.stream().collect(Collectors.toMap(AuditTemplate::getTemplateCode, Function.identity()));
        //将查询出来的稽核数据按照模板分组，使用线程池，单个线程跑单个模板
        Map<String, List<AuditMatch>> auditMatchMap = auditMatches.stream().collect(Collectors.groupingBy(AuditMatch::getTemplateCode));
        for (List<AuditMatch> auditMatchList : auditMatchMap.values()) {
            this.auditMatchThreadService.threadHandleMatchDate(templateMap.get(auditMatchList.get(0).getTemplateCode()), auditMatchList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String extractSapData(String sapDateStart, String sapDateEnd, String directCode, String deliveryCode) {
        Validate.isTrue(!StringUtils.isEmpty(sapDateStart), "未获取到开始时间！");
        Validate.isTrue(!StringUtils.isEmpty(sapDateEnd), "未获取到结束时间！");
        Validate.isTrue(!StringUtils.isEmpty(directCode),"未获取到系统编码!");
        sapDateStart = sapDateStart.replace("-","");
        sapDateEnd = sapDateEnd.replace("-","");
        String saleVouType = SapOrderTypeEnum.ZOR2.getValue();
        //查询系统
        List<DirectVo> directVoList = directVoService.findByDirectCodes(Lists.newArrayList(directCode));
        Validate.isTrue(!CollectionUtils.isEmpty(directVoList), "未根据系统编码查询到系统!");
        DirectVo directVo = directVoList.get(0);

        //查询模板
        List<AuditTemplateSupermarket> templateSupermarketList = this.auditTemplateSupermarketRepository.findByDirectCode(directCode);
        Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarketList), "未根据商超获取模板信息");
        AuditTemplate template = this.auditTemplateRepository.findByTemplateCode(templateSupermarketList.get(0).getTemplateCode());
        Validate.isTrue(!Objects.isNull(template), "模板为未启用状态!");
        String createAuditStatus = template.getCreateAuditStatus();
        //条件查询sap数据
        List<AuditSapEntity> auditSapEntityList = Lists.newArrayList();
        if (StringUtils.isNotEmpty(deliveryCode)){
            List<AuditSapEntity> deliveryCodeSapList = this.auditSapRepository.findByConditions(sapDateStart, sapDateEnd, Lists.newArrayList(deliveryCode),saleVouType,SapTransStatusEnum.NOT_TRANS.getValue(),createAuditStatus);
            if (CollectionUtils.isEmpty(deliveryCodeSapList)){
                List<AuditSapEntity> deliveryTransList = this.auditSapRepository.findByConditions(sapDateStart, sapDateEnd, Lists.newArrayList(deliveryCode),saleVouType,null,createAuditStatus);
                if (CollectionUtils.isEmpty(deliveryTransList)){
                    String result = String.format("该时间段内该门店没有出库单!");
                    return result;
                }else {
                    Integer count = deliveryTransList.size();
                    String result = String.format("该时间段内该门店共[%s]出库单，已生成条[%s]稽核单,本次生成[%s]",count,count,0);
                    return result;
                }
            }else {
                auditSapEntityList.addAll(deliveryCodeSapList);
            }
        } else {
            DirectStoreConditionDto dto = new DirectStoreConditionDto();
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
            dto.setDirectCode(directCode);
            List<DirectStoreVo> directStoreVoList = directStoreVoService.findByDirectStoreConditionDto(dto);
            Validate.isTrue(!CollectionUtils.isEmpty(directStoreVoList), "未根据系统编码查询到系统!");
            List<String> noDeliveryCodes = directStoreVoList.stream().map(DirectStoreVo::getTerminalCode).distinct().collect(Collectors.toList());
            List<AuditSapEntity> deliveryCodeSapList = this.auditSapRepository.findByConditions(sapDateStart, sapDateEnd, noDeliveryCodes,saleVouType,SapTransStatusEnum.NOT_TRANS.getValue(),createAuditStatus);
            if (CollectionUtils.isEmpty(deliveryCodeSapList)){
                List<AuditSapEntity> deliveryTransList = this.auditSapRepository.findByConditions(sapDateStart, sapDateEnd, noDeliveryCodes,saleVouType,null,createAuditStatus);
                if (CollectionUtils.isEmpty(deliveryTransList)){
                    String result = String.format("该时间段内该门店没有出库单!");
                    return result;
                }else {
                    Integer count = deliveryTransList.size();
                    String result = String.format("该时间段内该门店共[%s]出库单，已生成条[%s]稽核单,本次生成[%s]",count,count,0);
                    return result;
                }
            }else {
                auditSapEntityList.addAll(deliveryCodeSapList);
            }
        }

        if (CollectionUtils.isEmpty(auditSapEntityList)){
            String result = String.format("该时间段内该门店没有出库单!");
            return result;
        }
        //根据系统，送达方，物料编码去找上架产品
        Set<String> directCodes = Sets.newHashSet(directCode);
        Set<String> serviceCodes = auditSapEntityList.stream().map(AuditSapEntity::getServiceCode).collect(Collectors.toSet());
        Set<String> productCodes = auditSapEntityList.stream().map(AuditSapEntity::getMaterialCode).collect(Collectors.toSet());
        Map<String, List<DirectProductVo>> productMaps = this.buildProductInfo(directCodes, serviceCodes.stream().collect(Collectors.toSet()), productCodes);
//        if (MapUtils.isEmpty(productMaps)){
//            log.error("未能根据条件查询到上架产品!");
//        }

        //查询门店信息
        List<String> serviceCodeList = auditSapEntityList.stream().map(AuditSapEntity::getServiceCode).distinct().collect(Collectors.toList());
        List<DirectStoreVo> directStoreVos = directStoreVoService.findByDeliveryCodes(serviceCodeList);
        Validate.isTrue(!CollectionUtil.isEmpty(directStoreVos), "未根据送达方查询到系统和门店信息!");
        Map<String, DirectStoreVo> directStoreVoMap = directStoreVos.stream().filter(vo -> StringUtils.isNotEmpty(vo.getTerminalCode()) && StringUtils.isNotEmpty(vo.getDirectCode())).distinct().collect(Collectors.toMap(entity -> (entity.getDirectCode() + "_" + entity.getTerminalCode()), Function.identity(),(a,b)->a));


        //找退货单
        Map<String, AuditSapEntity> sapReturnMap = this.findByReturnMap(auditSapEntityList);

        //赋值组装数据
        List<AuditMatch> auditMatchList = this.transSapDateToAuditMatch(auditSapEntityList, templateSupermarketList.get(0), sapReturnMap, directVo, productMaps,directStoreVoMap,createAuditStatus);
        if (!CollectionUtils.isEmpty(auditMatchList)) {
            List<String> verifyCodes = auditMatchList.stream().map(AuditMatch::getVerifyCode).distinct().collect(Collectors.toList());
            //日志
            for (AuditMatch auditMatch : auditMatchList) {
                AuditMatchLogEventDto logEventDto = new AuditMatchLogEventDto();
                logEventDto.setOriginal(null);
                AuditMatchDto dto = this.nebulaToolkitService.copyObjectByWhiteList(auditMatch, AuditMatchDto.class, HashSet.class, ArrayList.class);
                logEventDto.setNewest(dto);
                SerializableBiConsumer<AuditMatchLogEventListener, AuditMatchLogEventDto> onCreate =
                        AuditMatchLogEventListener::onCreate;
                this.nebulaNetEventClient.publish(logEventDto, AuditMatchLogEventListener.class, onCreate);
            }
            //更新状态
            this.auditMatchRepository.saveOrUpdateBatch(auditMatchList);
            this.auditSapRepository.updateStatus(verifyCodes, SapTransStatusEnum.SUCCESS.getValue());
            String result = String.format("本次成功生成[%s]张稽核单!",auditMatchList.size());
            return result;
        }else {
            String result = String.format("本次成功生成[%s]张稽核单!",0);
            return result;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateNotMatchReason(List<AuditMatchNotReasonDto> dtoList) {
        for (AuditMatchNotReasonDto dto : dtoList) {
            this.auditMatchRepository.updateAuditNotReason(dto.getAuditCode(),dto.getNotMatchReason());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateVirtualStockOut(List<String> auditCodes) {
        Validate.isTrue(!CollectionUtils.isEmpty(auditCodes),"请选择稽核单!");
        List<AuditMatch> byAuditCodes = this.auditMatchRepository.findByAuditCodes(auditCodes);
        List<String> status = Lists.newArrayList(MatchStatusEnum.M300.getDictCode(), MatchStatusEnum.M200.getDictCode(), MatchStatusEnum.M200.getDictCode()).stream().collect(Collectors.toList());
        List<AuditMatch> auditMatchList = byAuditCodes.stream().filter(vo -> status.contains(status)).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(auditMatchList)){
            Validate.isTrue(!CollectionUtils.isEmpty(auditMatchList),"已匹配和已确认的稽核单不能进行虚出库处理!");
        }
        this.auditMatchRepository.updateAuditMatchStatus(auditCodes,MatchStatusEnum.M400.getDictCode());
    }

    @Override
    public void updateRelationOrderCode(List<AuditMatch> auditMatchs) {
        if (CollectionUtils.isEmpty(auditMatchs)) {
            return;
        }else {
            for (AuditMatch auditMatch : auditMatchs) {
                this.auditMatchRepository.updateAuditRelationOrderCode(auditMatch.getVerifyCode(),auditMatch.getRelationOrderCode());
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteAuditMatchByIds(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids),"未选中数据!");
        List<AuditMatch> auditMatchList = this.auditMatchRepository.findAuditMatchByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(auditMatchList),"未查询到数据!");
        List<String> auditCode = auditMatchList.stream().map(AuditMatch::getAuditCode).distinct().collect(Collectors.toList());
        //已完成稽核单号
        List<String> overAuditCode = auditCode.stream().filter(vo -> MatchStatusEnum.M300.getDictCode().equals(vo)).collect(Collectors.toList());
        Validate.isTrue(CollectionUtils.isEmpty(overAuditCode),"已确认的数据不能进行删除操作!");
        //已匹配稽核单号
        Set<String> matchAuditCode = auditCode.stream().filter(vo -> MatchStatusEnum.M200.getDictCode().equals(vo)).collect(Collectors.toSet());
        List<String> salesDeliveryCodes = auditMatchList.stream().map(AuditMatch::getSapOrderCode).distinct().collect(Collectors.toList());
        List<String> materialCodes = auditMatchList.stream().map(AuditMatch::getSapMaterialCode).distinct().collect(Collectors.toList());
        List<AuditSapEntity> sapEntityList = this.auditSapRepository.findSapEntityByCodeAndMaterialCode(salesDeliveryCodes, materialCodes);
        Map<String, List<AuditSapEntity>> sapMap = sapEntityList.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("-", vo.getSalesDeliveryNo(), vo.getMaterialCode())));
        List<String> verifyCodes = new ArrayList<>();
        for (AuditMatch auditMatch : auditMatchList) {
            List<AuditSapEntity> auditSapEntities = sapMap.get(StringUtils.joinWith("-", auditMatch.getSapOrderCode(), auditMatch.getSapMaterialCode()));
            List<String> collect = auditSapEntities.stream().map(AuditSapEntity::getVerifyCode).distinct().collect(Collectors.toList());
            verifyCodes.addAll(collect);
        }
        if (!CollectionUtils.isEmpty(verifyCodes)){
            this.auditSapRepository.updateByVerifyCodes(verifyCodes,SapTransStatusEnum.NOT_TRANS.getValue());
            this.auditSapRepository.updateIsMatchedByVerifyCodes(verifyCodes,BooleanEnum.FALSE.getCapital());
            this.auditMatchRepository.removeByIds(ids);
            if (!CollectionUtils.isEmpty(matchAuditCode)){
                this.auditMatchConsequenceRepository.deleteByAuditCodes(matchAuditCode.stream().collect(Collectors.toList()));
                this.matchInvoiceManuRepository.deleteByAuditCode(matchAuditCode);
            }
        }
    }

    /**
     * 转换SAP数据为稽核单
     *
     * @param auditSapEntityList
     * @param auditTemplateSupermarket
     * @param sapReturnMap
     * @return
     */
    private List<AuditMatch> transSapDateToAuditMatch(List<AuditSapEntity> auditSapEntityList, AuditTemplateSupermarket auditTemplateSupermarket,
                                                      Map<String, AuditSapEntity> sapReturnMap, DirectVo directVo, Map<String, List<DirectProductVo>> productMaps,
                                                      Map<String, DirectStoreVo> directStoreVoMap,String createAuditStatus) {
        List<AuditMatch> auditMatchList = new ArrayList<>();
        //获取缓存税率
        Object o = redisService.get(AuditMatchConstant.CACHE_PRODUCT_TAX_RATE);
        Map<String,BigDecimal> productTaxMap = (Map<String, BigDecimal>) o;
        //组装数据
        List<AuditSapEntity> auditSapEntities = new ArrayList<>();
        Map<String, List<AuditSapEntity>> materialMap = auditSapEntityList.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("_", vo.getSalesDeliveryNo(), vo.getMaterialCode())));
        for (List<AuditSapEntity> value : materialMap.values()) {
            AuditSapEntity auditSapEntitySummary = value.get(0);
            BigDecimal addNum = BigDecimal.ZERO;
            for (AuditSapEntity auditSapEntity : value) {
                addNum = addNum.add(new BigDecimal(auditSapEntity.getOrderNum1()));
            }
            auditSapEntitySummary.setOrderNum1(String.valueOf(addNum));
            auditSapEntities.add(auditSapEntitySummary);
        }
        for (AuditSapEntity auditSapEntity : auditSapEntities) {
            AuditMatch auditMatch = new AuditMatch();
            if (Objects.nonNull(directVo)) {
                auditMatch.setDirectCode(directVo.getDirectCode());
                auditMatch.setDirectName(directVo.getSupermarketName());
                auditMatch.setBusinessFormatCode(directVo.getBusinessFormatCode());
                auditMatch.setBusinessUnitCode(directVo.getBusinessUnitCode());
            }
            auditMatch.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            auditMatch.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
            auditMatch.setTemplateCode(auditTemplateSupermarket.getTemplateCode());
            auditMatch.setKaName(auditTemplateSupermarket.getSupermarketName());
            auditMatch.setKaCode(auditTemplateSupermarket.getSupermarketCode());
            auditMatch.setSapPostingDate(DateUtil.getDateByFormat(this.transition(auditSapEntity.getShippingAccountDate()), DateUtil.DEFAULT_YEAR_MONTH_DAY));
            auditMatch.setSoldToPartyCode(auditSapEntity.getDealerCode());
            auditMatch.setSoldToPartyName(auditSapEntity.getDealerName());
            //处理订单抬头的采购单号
            if (StringUtils.isNotEmpty(auditSapEntity.getOrderHeader())) {
                List<String> purchaseOrderNo = Arrays.stream(auditSapEntity.getOrderHeader().split("/")).collect(Collectors.toList());
                if (StringUtils.isNotEmpty(purchaseOrderNo.get(0))) {
                    if (Pattern.matches(rgx, purchaseOrderNo.get(0))) {
                        auditMatch.setRelationOrderCode(purchaseOrderNo.get(0));
                    }
                }
            }
            auditMatch.setVerifyCode(auditSapEntity.getVerifyCode());
            auditMatch.setSapPrice(auditSapEntity.getOrderRegularPrice());
            auditMatch.setRelationOrderCode(auditSapEntity.getPurchaseOrderNo());
            auditMatch.setDeliveryCode(auditSapEntity.getServiceCode());
            auditMatch.setDeliveryName(auditSapEntity.getServiceName());
            auditMatch.setSapMaterialCode(auditSapEntity.getMaterialCode());
            auditMatch.setSalesOrderNumber(auditSapEntity.getSalesOrderNo());
            //获取区域
            String areaKey = StringUtils.joinWith("_",auditMatch.getDirectCode(),auditMatch.getDeliveryCode());
            if (MapUtils.isNotEmpty(directStoreVoMap)){
                DirectStoreVo directStoreVo = directStoreVoMap.get(areaKey);
                if (Objects.nonNull(directStoreVo)){
                    auditMatch.setKaStoreCode(directStoreVo.getSupermarketStoreCode());
                    auditMatch.setKaStoreName(directStoreVo.getSupermarketStoreName());
                    auditMatch.setBusinessArea(directStoreVo.getBusinessArea());
                    auditMatch.setProvinceCode(directStoreVo.getProvinceCode());
                    auditMatch.setProvinceName(directStoreVo.getProvinceName());
                }
            }
            //获取物料
            DirectProductVo productVo = this.getGoodsInfo(auditMatch, productMaps);
            if (Objects.nonNull(productVo)) {
                auditMatch.setSapMaterialName(productVo.getProductName());
            } else {
                auditMatch.setSapMaterialName("未根据条件查询到上架产品!");
            }
            //获取税率
            BigDecimal taxRate = BigDecimal.ZERO;
            if (MapUtils.isNotEmpty(productTaxMap)){
                BigDecimal productTax = productTaxMap.get(auditMatch.getSapMaterialCode());
                if (productTax != null && productTax.compareTo(BigDecimal.ZERO) != 0){
                    taxRate = taxRate.add(productTax);
                }
            }
            //退货单,减数量算金额
            BigDecimal orderNum = new BigDecimal(auditSapEntity.getOrderNum1() == null ? "0":auditSapEntity.getOrderNum1());
            String returnKey = StringUtils.joinWith("_", auditSapEntity.getSalesDeliveryNo(),auditSapEntity.getMaterialCode());
            AuditSapEntity returnOrder = sapReturnMap.get(returnKey);
            BigDecimal sapTotalAmount = BigDecimal.ZERO;
            if (CreateAuditMatchEnum.INVOICESTATUS.equals(createAuditStatus)){
                sapTotalAmount = auditSapEntity.getDeliveryAfterDiscountAmt().subtract(auditSapEntity.getInvoiceDiscountAmt() == null ? BigDecimal.ZERO:auditSapEntity.getInvoiceDiscountAmt());
            }else {
                sapTotalAmount = auditSapEntity.getDeliveryDeforeDiscountAmt() == null ? BigDecimal.ZERO:auditSapEntity.getDeliveryDeforeDiscountAmt();
            }
            if (!Objects.isNull(returnOrder)) {
                BigDecimal returnNum = new BigDecimal(returnOrder.getOrderNum1()).add(orderNum);
                if (BigDecimal.ZERO.compareTo(returnNum) == 0){
                    continue;
                }
                BigDecimal sapTotalAmountTaxExclusive = BigDecimal.ZERO;
                BigDecimal returnMoney = BigDecimal.ZERO;
                if (CreateAuditMatchEnum.INVOICESTATUS.equals(createAuditStatus)){
                    returnMoney = returnOrder.getDeliveryAfterDiscountAmt().subtract(returnOrder.getInvoiceDiscountAmt() == null ? BigDecimal.ZERO:returnOrder.getInvoiceDiscountAmt());
                }else {
                    returnMoney = returnOrder.getDeliveryDeforeDiscountAmt() == null ? BigDecimal.ZERO:returnOrder.getDeliveryDeforeDiscountAmt();
                }
                sapTotalAmount = sapTotalAmount.add(returnMoney);
                BigDecimal noTax = sapTotalAmount.divide(BigDecimal.ONE.add(taxRate),4,BigDecimal.ROUND_HALF_UP);
                sapTotalAmountTaxExclusive = sapTotalAmountTaxExclusive.add(noTax);
                auditMatch.setSapEaCount(returnNum);
                auditMatch.setSapTotalAmount(sapTotalAmount);
                auditMatch.setSapTotalAmountTaxExclusive(sapTotalAmountTaxExclusive);
            } else {
                auditMatch.setSapEaCount(orderNum);
                auditMatch.setSapTotalAmount(sapTotalAmount);
                auditMatch.setSapTotalAmountTaxExclusive(sapTotalAmount.divide(BigDecimal.ONE.add(taxRate),4,BigDecimal.ROUND_HALF_UP));
            }
            auditMatch.setMatchStatus(MatchStatusEnum.M100.getDictCode());
            auditMatch.setWaitDays(0);
            auditMatch.setSapOrderCode(auditSapEntity.getSalesDeliveryNo());
            auditMatch.setTenantCode(TenantUtils.getTenantCode());
            auditMatchList.add(auditMatch);
        }
        if (!CollectionUtils.isEmpty(auditMatchList)) {
            List<String> auditCodes = this.generateCodeService.generateCode(AuditMatchConstant.AUDIT_CODE, auditMatchList.size());
            for (int i = 0; i < auditMatchList.size(); i++) {
                auditMatchList.get(i).setAuditCode(auditCodes.get(i));
            }
        }
        return auditMatchList;
    }

    private Map<String, List<DirectProductVo>> buildProductInfo(Set<String> directCodeSet,
                                                                Set<String> deliveryPartyCodeSet,
                                                                Set<String> productCodes) {
        Map<String, List<DirectProductVo>> directProductMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(directCodeSet)
                || CollectionUtil.isEmpty(deliveryPartyCodeSet)
                || CollectionUtil.isEmpty(productCodes)) {
            return directProductMap;
        }
        List<List<String>> productCodeGroupList = Lists.partition(Lists.newArrayList(productCodes), ConstantEnums.PRODUCT_SIZE.getValue());
        DirectProductDto directProductDto = new DirectProductDto();
        directProductDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        directProductDto.setTenantCode(TenantUtils.getTenantCode());
        directProductDto.setOnShelfStatus(BooleanEnum.TRUE.getCapital());
        directProductDto.setDirectCodes(new ArrayList<>(directCodeSet));
        directProductDto.setDeliveryPartyCodes(new ArrayList<>(deliveryPartyCodeSet));
        directProductDto.setOrderType("acceptance");
        productCodeGroupList.forEach(productCodeList -> {
            directProductDto.setProductCodes(productCodeList);
            List<DirectProductVo> productList = this.directProductVoService.findByDirectProductDto(directProductDto);
            if (CollectionUtil.isNotEmpty(productList)) {
                directProductMap.putAll(productList.stream()
                        .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getOnShelfStatus()))
                        .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getDeliveryPartyCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getProductCode()))
                        .collect(Collectors.groupingBy(k -> k.getDirectCode()
                                + "_" + k.getDeliveryPartyCode()
                                + "_" + k.getProductCode())));
            }
        });
        return directProductMap;
    }

    /**
     * 构建商品转换明细信息
     *
     * @param auditMatch
     * @param directProductMap
     * @return
     */
    private DirectProductVo getGoodsInfo(AuditMatch auditMatch,
                                         Map<String, List<DirectProductVo>> directProductMap) {
        DirectProductVo productVo = null;
        if (StringUtils.isBlank(auditMatch.getSapMaterialCode())) {
            return null;
        }
        if (MapUtils.isEmpty(directProductMap)) {
            return null;
        }
        List<DirectProductVo> productVoList = directProductMap.get(auditMatch.getDirectCode() + "_" +
                auditMatch.getDeliveryCode() + "_" + auditMatch.getSapMaterialCode());
        if (CollectionUtil.isEmpty(productVoList)) {
            return null;
        } else {
            productVo = productVoList.get(0);
        }
        return productVo;
    }

    /**
     * @param prodDate
     * @return
     */
    private String transition(String prodDate) {
        StringBuffer str = new StringBuffer(prodDate);
        str.insert(6, "-");
        str.insert(4, "-");
        return String.valueOf(str);
    }

    /**
     * 根据条件找退货单退货单
     * @param auditSapEntityList
     * @return
     */
    private Map<String, AuditSapEntity> findByReturnMap(List<AuditSapEntity> auditSapEntityList) {
        Map<String, AuditSapEntity> returnMap = new HashMap<>();
        if (CollectionUtils.isEmpty(auditSapEntityList)) {
            return returnMap;
        }
        //物料编码
        List<String> materialCode = auditSapEntityList.stream().map(AuditSapEntity::getMaterialCode).distinct().collect(Collectors.toList());
        //往前1个月的时间分区到现在(当数据量大的时候缩小范围)
        String beginDate = DateUtil.format(DateUtil.getFirstDayOfLastMonth(),DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH);
        String endDate = DateUtil.format(new Date(),DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH);
        //获取退货单
        List<AuditSapEntity> returnOrder = this.auditSapRepository.findReturnOrderByMaterialCodeAndDate(materialCode, beginDate, endDate);
        if (CollectionUtils.isEmpty(returnOrder)){
            return returnMap;
        }
        Map<String, List<AuditSapEntity>> collect = returnOrder.stream().filter(vo -> StringUtils.isNotEmpty(vo.getOrderHeader())).collect(Collectors.groupingBy(vo -> StringUtils.joinWith("_", this.getOrderHeaderFirst(vo.getOrderHeader()), vo.getMaterialCode())));
        for (List<AuditSapEntity> value : collect.values()) {
            BigDecimal num = BigDecimal.ZERO;
            for (AuditSapEntity auditSapEntity : value) {
                num = num.add(new BigDecimal(auditSapEntity.getOrderNum1()));
            }
            AuditSapEntity auditSapEntity = value.get(0);
            String key = StringUtils.joinWith("_", this.getOrderHeaderFirst(auditSapEntity.getOrderHeader()), auditSapEntity.getMaterialCode());
            auditSapEntity.setOrderNum1(String.valueOf(num));
            returnMap.put(key, auditSapEntity);
        }
        return returnMap;
    }

    private String getOrderHeaderFirst(String param){
        String[] split = StringUtils.split(param,"/");
        return split[0];
    }
}