package com.biz.crm.kms.business.audit.match.local.service.internal;


import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.kms.business.audit.match.local.entity.*;
import com.biz.crm.kms.business.audit.match.local.model.AuditMatchInvoiceModel;
import com.biz.crm.kms.business.audit.match.local.model.MatchConsequenceCalculatedAbstract;
import com.biz.crm.kms.business.audit.match.local.model.MatchConsequenceCalculatedModel;
import com.biz.crm.kms.business.audit.match.local.repository.AuditMatchConsequenceRepository;
import com.biz.crm.kms.business.audit.match.local.repository.AuditMatchRepository;
import com.biz.crm.kms.business.audit.match.local.repository.AuditTemplateRepository;
import com.biz.crm.kms.business.audit.match.local.repository.AuditTemplateSupermarketRepository;
import com.biz.crm.kms.business.audit.match.local.service.AuditTemplateService;
import com.biz.crm.kms.business.audit.match.local.service.AuditTemplateSupermarketService;
import com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditTemplateDto;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditTemplateSupermarketDto;
import com.biz.crm.kms.business.audit.match.sdk.dto.KmsAuditTemplateLogEventDto;
import com.biz.crm.kms.business.audit.match.sdk.listener.KmsAuditTemplateLogEventListener;
import com.biz.crm.kms.business.audit.match.sdk.register.InvoiceMatchRegister;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.invoice.sdk.enums.ConstantEnums;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.task.service.DynamicTaskSchedulerVoService;
import com.google.common.collect.Lists;

import java.util.*;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 稽查模板表(AuditTemplate)表服务实现类
 *
 * @author songjingen
 * @since 2022-10-13 14:40:28
 */
@Service("auditTemplateService")
public class AuditTemplateServiceImpl implements AuditTemplateService {

  @Autowired
  private AuditTemplateRepository auditTemplateRepository;

  @Autowired
  private AuditTemplateSupermarketService auditTemplateSupermarketService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private List<InvoiceMatchRegister> invoiceMatchRegisters;

  @Autowired(required = false)
  private DynamicTaskSchedulerVoService dynamicTaskSchedulerVoService;

  @Autowired(required = false)
  private ApplicationContext applicationContext;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private AuditMatchRepository auditMatchRepository;

  @Autowired(required = false)
  private AuditMatchConsequenceRepository auditMatchConsequenceRepository;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private AuditTemplateSupermarketRepository auditTemplateSupermarketRepository;

  @Autowired(required = false)
  private DirectVoService directVoService;

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public AuditTemplate findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    AuditTemplate auditTemplate = this.auditTemplateRepository.getById(id);
    if (auditTemplate == null) {
      return null;
    }
    //查询子表模板商超数据信息
    List<AuditTemplateSupermarket> auditTemplateSupermarkets = this.auditTemplateSupermarketService.findByTemplateCodes(Lists.newArrayList(auditTemplate.getTemplateCode()));
    auditTemplate.setTemplateSupermarkets(auditTemplateSupermarkets);
    return auditTemplate;
  }

  /**
   * 新增数据
   *
   * @param auditTemplate 实体对象
   * @return 新增结果
   */
  @Transactional(rollbackFor = Exception.class)
  @Override
  public AuditTemplate create(AuditTemplate auditTemplate) {
    this.createValidate(auditTemplate);

    //保存子表商超数据
    List<AuditTemplateSupermarket> templateSupermarkets = auditTemplate.getTemplateSupermarkets();
    Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "保存数据时，未获取到商超数据信息！");
    List<String> directCodes = templateSupermarkets.stream().map(AuditTemplateSupermarket::getDirectCode).collect(Collectors.toList());
    //校验重复
    List<AuditTemplateSupermarket> auditTemplateSupermarketList = this.auditTemplateSupermarketRepository.findByDirectCodes(directCodes);
    Validate.isTrue(CollectionUtils.isEmpty(auditTemplateSupermarketList),"选中的系统已经关联其他模板!");
    //模板上添加业态、业务单元、销售组织
    List<DirectVo> directVos = this.directVoService.findByDirectCodes(directCodes);
    Validate.isTrue(!CollectionUtils.isEmpty(directVos),"未查询到对应的系统");
    Set<String> businessUnitCodes = directVos.stream().map(DirectVo::getBusinessUnitCode).collect(Collectors.toSet());
    Validate.isTrue(!CollectionUtils.isEmpty(businessUnitCodes),"所选系统中的业态为空");
    Set<String> businessFormatCodes = directVos.stream().map(DirectVo::getBusinessFormatCode).collect(Collectors.toSet());
    Validate.isTrue(!CollectionUtils.isEmpty(businessFormatCodes),"所选系统中的业务单元为空");
    Set<String> salesOrgCodes = directVos.stream().map(DirectVo::getSalesOrgCode).collect(Collectors.toSet());
    Validate.isTrue(!CollectionUtils.isEmpty(salesOrgCodes),"所选系统中的销售组织为空");
    Validate.isTrue(businessUnitCodes.size()== 1 && businessFormatCodes.size() == 1 && salesOrgCodes.size() == 1,"所选商超中的业态、业务单元、销售组织不唯一");
    DirectVo directVo = directVos.get(0);
    auditTemplate.setBusinessUnitCode(directVo.getBusinessUnitCode());
    auditTemplate.setBusinessFormatCode(directVo.getBusinessFormatCode());
    auditTemplate.setSalesOrgCode(directVo.getSalesOrgCode());
    auditTemplate.setSalesOrgName(directVo.getSalesOrgName());

    templateSupermarkets.stream().forEach(auditTemplateSupermarket -> auditTemplateSupermarket.setTemplateCode(auditTemplate.getTemplateCode()));
    AuditTemplateDto dto = this.nebulaToolkitService.copyObjectByWhiteList(auditTemplate,AuditTemplateDto.class,HashSet.class, ArrayList.class);
    KmsAuditTemplateLogEventDto logEventDto = new KmsAuditTemplateLogEventDto();
    logEventDto.setOriginal(null);
    logEventDto.setNewest(dto);
    SerializableBiConsumer<KmsAuditTemplateLogEventListener, KmsAuditTemplateLogEventDto> onCreate =
            KmsAuditTemplateLogEventListener::onCreate;
    this.nebulaNetEventClient.publish(logEventDto, KmsAuditTemplateLogEventListener.class, onCreate);
    this.auditTemplateRepository.saveOrUpdate(auditTemplate);
    this.auditTemplateSupermarketService.createBatch(templateSupermarkets);
    //创建定时任务
//    String matchTime = auditTemplate.getMatchTime();
//    String[] matchTimeSplit = matchTime.split(":");
//    String matchTimeCron = StringUtils.joinWith(" ", "0", matchTimeSplit[1], matchTimeSplit[0], "*", "*", "?");
//    this.dynamicTaskSchedulerVoService.createIgnorePrefix(auditTemplate.getTemplateCode(), "auditTemplateService", "handleDynamicTaskScheduler", matchTimeCron, null, auditTemplate.getTemplateName(),
//        auditTemplate.getTemplateCode());
//    this.dynamicTaskSchedulerVoService.start(new String[]{auditTemplate.getTemplateCode()});
    return auditTemplate;
  }

  /**
   * 修改新据
   *
   * @param auditTemplate 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public AuditTemplate update(AuditTemplate auditTemplate) {
    Validate.notBlank(auditTemplate.getId(), "修改模板时，id不能为空！");
    AuditTemplate current = this.auditTemplateRepository.getById(auditTemplate.getId());
    AuditTemplateDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(current, AuditTemplateDto.class, HashSet.class, ArrayList.class);
    Validate.notNull(current, "修改模板时，未查询到数据！");
    BeanUtils.copyProperties(auditTemplate, current, "id", "templateCode", "enableStatus", "delFlag", "createTime", "createName", "createAccount");
    this.updateValidate(current);
    //保存子表商超数据
    List<AuditTemplateSupermarket> templateSupermarkets = auditTemplate.getTemplateSupermarkets();
    Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "保存数据时，未获取到商超数据信息！");
    List<AuditTemplateSupermarket> oldTemplateSupermarket = this.auditTemplateSupermarketService.findByTemplateCodes(Lists.newArrayList(current.getTemplateCode()));
    List<AuditTemplateSupermarketDto> oldDtoList = (List<AuditTemplateSupermarketDto>) this.nebulaToolkitService.copyCollectionByBlankList(oldTemplateSupermarket, AuditTemplateSupermarket.class,AuditTemplateSupermarketDto.class, HashSet.class, ArrayList.class);
    //日志
    AuditTemplateDto dto = this.nebulaToolkitService.copyObjectByWhiteList(current, AuditTemplateDto.class, HashSet.class, ArrayList.class);
    List<AuditTemplateSupermarketDto> supermarketDtoList = (List<AuditTemplateSupermarketDto>) this.nebulaToolkitService.copyCollectionByBlankList(templateSupermarkets, AuditTemplateSupermarket.class,AuditTemplateSupermarketDto.class, HashSet.class, ArrayList.class);
    dto.setTemplateSupermarkets(supermarketDtoList);
    KmsAuditTemplateLogEventDto logEventDto = new KmsAuditTemplateLogEventDto();
    oldDto.setTemplateSupermarkets(oldDtoList);
    logEventDto.setOriginal(oldDto);
    logEventDto.setNewest(dto);
    SerializableBiConsumer<KmsAuditTemplateLogEventListener, KmsAuditTemplateLogEventDto> onUpdate =
            KmsAuditTemplateLogEventListener::onUpdate;
    this.nebulaNetEventClient.publish(logEventDto, KmsAuditTemplateLogEventListener.class, onUpdate);
    this.auditTemplateRepository.saveOrUpdate(current);
    this.auditTemplateSupermarketService.deleteByTemplateCodes(Lists.newArrayList(current.getTemplateCode()));
    List<String> directCodes = templateSupermarkets.stream().map(AuditTemplateSupermarket::getDirectCode).collect(Collectors.toList());
    List<AuditTemplateSupermarket> auditTemplateSupermarketList = this.auditTemplateSupermarketRepository.findByDirectCodes(directCodes);
    Validate.isTrue(CollectionUtils.isEmpty(auditTemplateSupermarketList),"选中的系统已经关联其他模板!");
    templateSupermarkets.stream().forEach(auditTemplateSupermarket -> auditTemplateSupermarket.setTemplateCode(current.getTemplateCode()));
    this.auditTemplateSupermarketService.createBatch(templateSupermarkets);
    //修改定时任务
//    String matchTime = auditTemplate.getMatchTime();
//    String[] matchTimeSplit = matchTime.split(":");
//    String matchTimeCron = StringUtils.joinWith(" ", "0", matchTimeSplit[1], matchTimeSplit[0], "*", "*", "?");
//    this.dynamicTaskSchedulerVoService.updateIgnorePrefix(auditTemplate.getTemplateCode(), matchTimeCron, null, auditTemplate.getTemplateName(), auditTemplate.getTemplateCode());

    return current;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional(rollbackFor = Exception.class)
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<AuditTemplate> auditTemplates = this.auditTemplateRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(auditTemplates), "删除数据时，未查询到需要删除的数据！");
    auditTemplates.stream().forEach(auditTemplate -> auditTemplate.setDelFlag(DelFlagStatusEnum.DELETE.getCode()));
    this.auditTemplateRepository.saveOrUpdateBatch(auditTemplates);
    List<String> templateCodes = auditTemplates.stream().map(AuditTemplate::getTemplateCode).collect(Collectors.toList());
    //删除关联表信息
    this.auditTemplateSupermarketService.deleteByTemplateCodes(templateCodes);
    //删除定时任务
//    this.dynamicTaskSchedulerVoService.deleteByTaskcodes(templateCodes.toArray(new String[0]));
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional(rollbackFor = Exception.class)
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    List<AuditTemplate> auditTemplates = this.auditTemplateRepository.findByEnableStatusAndIds(EnableStatusEnum.DISABLE, idList);
    Validate.isTrue(!CollectionUtils.isEmpty(auditTemplates), "启用数据时，未查询到需要启用的数据！");
    auditTemplates.stream().forEach(auditTemplate -> auditTemplate.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    //日志
    List<AuditTemplateDto> dtoList = (List<AuditTemplateDto>) this.nebulaToolkitService.copyCollectionByWhiteList(auditTemplates, AuditTemplate.class,AuditTemplateDto.class, HashSet.class, ArrayList.class);
    dtoList.forEach(dto ->{
      KmsAuditTemplateLogEventDto logEventDto = new KmsAuditTemplateLogEventDto();
      dto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
      logEventDto.setOriginal(dto);
      dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      logEventDto.setNewest(dto);
      SerializableBiConsumer<KmsAuditTemplateLogEventListener, KmsAuditTemplateLogEventDto> onEnable =
              KmsAuditTemplateLogEventListener::onEnable;
      this.nebulaNetEventClient.publish(logEventDto, KmsAuditTemplateLogEventListener.class, onEnable);
    });
    this.auditTemplateRepository.saveOrUpdateBatch(auditTemplates);
//    //启动定时任务
//    List<String> templateCodes = auditTemplates.stream().map(AuditTemplate::getTemplateCode).collect(Collectors.toList());
//    this.dynamicTaskSchedulerVoService.start(templateCodes.toArray(new String[0]));
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional(rollbackFor = Exception.class)
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    List<AuditTemplate> auditTemplates = this.auditTemplateRepository.findByEnableStatusAndIds(EnableStatusEnum.ENABLE, idList);
    Validate.isTrue(!CollectionUtils.isEmpty(auditTemplates), "禁用数据时，未查询到需要禁用的数据！");
    auditTemplates.stream().forEach(auditTemplate -> auditTemplate.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
    //日志
    List<AuditTemplateDto> dtoList = (List<AuditTemplateDto>) this.nebulaToolkitService.copyCollectionByWhiteList(auditTemplates, AuditTemplate.class,AuditTemplateDto.class, HashSet.class, ArrayList.class);
    dtoList.forEach(dto ->{
      KmsAuditTemplateLogEventDto logEventDto = new KmsAuditTemplateLogEventDto();
      dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      logEventDto.setOriginal(dto);
      dto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
      logEventDto.setNewest(dto);
      SerializableBiConsumer<KmsAuditTemplateLogEventListener, KmsAuditTemplateLogEventDto> onDisable =
              KmsAuditTemplateLogEventListener::onDisable;
      this.nebulaNetEventClient.publish(logEventDto, KmsAuditTemplateLogEventListener.class, onDisable);
    });
    this.auditTemplateRepository.saveOrUpdateBatch(auditTemplates);
//    //暂停定时任务
//    List<String> templateCodes = auditTemplates.stream().map(AuditTemplate::getTemplateCode).collect(Collectors.toList());
//    this.dynamicTaskSchedulerVoService.stop(templateCodes.toArray(new String[0]));
  }

  /**
   * 触发执行定时任务的方法
   *
   * @param templateCode 模板编码（任务编码）
   */
  @Transactional(rollbackFor = Exception.class)
  public void handleDynamicTaskScheduler(String templateCode) {
    Validate.isTrue(!CollectionUtils.isEmpty(this.invoiceMatchRegisters), "触发执行模板匹配任务时，未查询到单据匹配注册器！");
    Validate.notBlank(templateCode, "触发执行模板匹配任务时，模板编码（任务编码）未获取到！");
    AuditTemplate auditTemplate = this.auditTemplateRepository.findByTemplateCode(templateCode);
    Validate.notNull(auditTemplate, "触发执行模板匹配任务时，未查询到模板编码（任务编码）：【%s】的数据！", templateCode);
    List<AuditTemplateSupermarket> templateSupermarkets = this.auditTemplateSupermarketService.findByTemplateCodes(Lists.newArrayList(templateCode));
    Validate.isTrue(!CollectionUtils.isEmpty(templateSupermarkets), "触发执行模板匹配任务时，未查询到模板编码（任务编码）：【%s】的数据！", templateCode);
    List<MatchConsequenceCalculatedModel> matchConsequenceCalculatedModels = this.covertMatchLogicCalculatedModels(templateSupermarkets, templateCode);
    for (InvoiceMatchRegister invoiceMatchRegister : this.invoiceMatchRegisters) {
      if (invoiceMatchRegister.getMatchLogicCode().equals(auditTemplate.getMatchLogicCode())) {
        Class<?> matchLogicCalculatedStrategy = invoiceMatchRegister.getMatchLogicCalculatedAbstract();
        MatchConsequenceCalculatedAbstract bean = (MatchConsequenceCalculatedAbstract) this.applicationContext.getBean(matchLogicCalculatedStrategy);
        bean.execute(templateCode, matchConsequenceCalculatedModels);
        break;
      }
    }
  }

  /**
   * 创建验证
   *
   * @param auditTemplate
   */
  private void createValidate(AuditTemplate auditTemplate) {
    Validate.notNull(auditTemplate, "新增时，对象信息不能为空！");
    auditTemplate.setId(null);
    auditTemplate.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    auditTemplate.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    auditTemplate.setTenantCode(TenantUtils.getTenantCode());
    auditTemplate.setTemplateCode(this.generateCodeService.generateCode(AuditMatchConstant.TEMPLATE_CODE, 1).get(0));
    Validate.notBlank(auditTemplate.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(auditTemplate.getMatchLogicCode(), "新增数据时，匹配逻辑编码不能为空！");
    Validate.notBlank(auditTemplate.getCreateAuditStatus(),"新增数据时，稽核单生成条件不能为空!");
//    Validate.notNull(auditTemplate.getMatchTime(), "新增数据时，稽核匹配时间（eg：每日2:30 抽取发货数据并执行自动匹配）不能为空！");
    Validate.notBlank(auditTemplate.getTemplateCode(), "新增数据时，模板编码不能为空！");
    Validate.notBlank(auditTemplate.getTemplateName(), "新增数据时，模板名称不能为空！");
    //验证匹配时间格式
//    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s", auditTemplate.getMatchTime(), "00")), "非法的结束时间格式,可用的格式:[HH:mm]");
    //验证名称不能重复
    List<AuditTemplate> auditTemplates = this.auditTemplateRepository.findByTemplateName(auditTemplate.getTemplateName());
    Validate.isTrue(CollectionUtils.isEmpty(auditTemplates), "新增数据时，模板名称已存在！");
  }

  /**
   * 修改验证
   *
   * @param auditTemplate
   */
  private void updateValidate(AuditTemplate auditTemplate) {
    auditTemplate.setTenantCode(TenantUtils.getTenantCode());
    Validate.notNull(auditTemplate, "修改时，对象信息不能为空！");
    Validate.notBlank(auditTemplate.getId(), "修改数据时，不能为空！");
    Validate.notBlank(auditTemplate.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(auditTemplate.getMatchLogicCode(), "修改数据时，匹配逻辑编码不能为空！");
    Validate.notBlank(auditTemplate.getCreateAuditStatus(),"新增数据时，稽核单生成条件不能为空!");
//    Validate.notNull(auditTemplate.getMatchTime(), "修改数据时，稽核匹配时间（eg：每日2:30 抽取发货数据并执行自动匹配）不能为空！");
    Validate.notBlank(auditTemplate.getTemplateCode(), "修改数据时，模板编码不能为空！");
    Validate.notBlank(auditTemplate.getTemplateName(), "修改数据时，模板名称不能为空！");
    //验证匹配时间格式
//    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s", auditTemplate.getMatchTime(), "00")), "非法的结束时间格式,可用的格式:[HH:mm]");
    //验证名称不能重复
    List<AuditTemplate> auditTemplates = this.auditTemplateRepository.findByTemplateName(auditTemplate.getTemplateName());
    if (!CollectionUtils.isEmpty(auditTemplates)) {
      List<String> ids = auditTemplates.stream().map(AuditTemplate::getId).collect(Collectors.toList());
      Validate.isTrue(ids.contains(auditTemplate.getId()), "新增数据时，模板名称已存在！");
    }


  }

  /**
   * 根据需要的商超模板数据，转换成需要匹配的数据格式
   *
   * @param supermarkets 商超模板数据
   * @param templateCode 模板编码
   * @return 匹配的数据
   */
  private List<MatchConsequenceCalculatedModel> covertMatchLogicCalculatedModels(List<AuditTemplateSupermarket> supermarkets, String templateCode) {
    if (CollectionUtils.isEmpty(supermarkets)) {
      return Lists.newArrayList();
    }
    Map<String, AuditTemplateSupermarket> supermarketMap = supermarkets.stream().collect(Collectors.toMap(AuditTemplateSupermarket::getDirectCode, a -> a));
    //查询绑定模板的稽核数据
    List<AuditMatch> auditMatches = this.auditMatchRepository.findByTemplateCode(templateCode);
    if (CollectionUtils.isEmpty(auditMatches)) {
      return Lists.newArrayList();
    }
    //组装成可执行的数据
    List<MatchConsequenceCalculatedModel> calculatedModels = new ArrayList<>();
    List<String> auditCodes = new ArrayList<>();
    for (AuditMatch match : auditMatches) {
      MatchConsequenceCalculatedModel calculatedModel = this.nebulaToolkitService.copyObjectByWhiteList(match, MatchConsequenceCalculatedModel.class, HashSet.class, ArrayList.class);
      AuditTemplateSupermarket templateSupermarket = supermarketMap.get(match.getDirectCode());
      if (templateSupermarket != null) {
        calculatedModel.setPriceValue(templateSupermarket.getPriceValue());
        calculatedModel.setPriceType(templateSupermarket.getPriceType());
      }
      calculatedModels.add(calculatedModel);
      auditCodes.add(match.getAuditCode());
    }
    //查询出已经匹配的数据然后赋值id
    List<AuditMatchConsequence> auditMatchConsequences = this.auditMatchConsequenceRepository.findByAuditCodes(auditCodes);
    if (!CollectionUtils.isEmpty(auditMatchConsequences)) {
      Map<String, String> auditMatchConsequenceMap = auditMatchConsequences.stream()
          .collect(Collectors.toMap(AuditMatchConsequence::getAuditCode, AuditMatchConsequence::getId, (a, b) -> a));
      calculatedModels.stream().forEach(o -> o.setId(auditMatchConsequenceMap.get(o.getAuditCode())));
    }
    return calculatedModels;
  }
}

