package com.biz.crm.kms.business.audit.match.local.service.internal;


import cn.hutool.core.date.DateUtil;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.kms.business.audit.match.local.entity.*;
import com.biz.crm.kms.business.audit.match.local.repository.AuditMatchRepository;
import com.biz.crm.kms.business.audit.match.local.repository.AuditSapRepository;
import com.biz.crm.kms.business.audit.match.local.repository.AuditTemplateRepository;
import com.biz.crm.kms.business.audit.match.local.repository.KmsAuditSummaryRepository;
import com.biz.crm.kms.business.audit.match.local.service.AuditTemplateSupermarketService;
import com.biz.crm.kms.business.audit.match.local.service.KmsAuditSummaryService;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditSapDto;
import com.biz.crm.kms.business.audit.match.sdk.enums.MatchParamsEnum;
import com.biz.crm.kms.business.audit.match.sdk.enums.MatchStatusEnum;
import com.biz.crm.kms.business.audit.match.sdk.vo.AuditSapVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.acceptance.sdk.enums.AcceptanceStatus;
import com.biz.crm.kms.business.invoice.acceptance.sdk.service.InvoiceAcceptanceVoService;
import com.biz.crm.kms.business.invoice.acceptance.sdk.vo.InvoiceAcceptanceGoodsVo;
import com.biz.crm.kms.business.invoice.acceptance.sdk.vo.InvoiceAcceptanceVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant.KMS_AUDIT_SAP_LOOP_MAX;
import static com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant.KMS_AUDIT_SAP_PAGE_SIZE;

/**
 * 稽核汇总表(KmsAuditSummary)表服务实现类
 *
 * @author cyj
 * @since 2023-01-04 14:44:49
 */
@Service("kmsAuditSummaryService")
public class KmsAuditSummaryServiceImpl implements KmsAuditSummaryService {

    @Autowired(required = false)
    private KmsAuditSummaryRepository kmsAuditSummaryRepository;

    @Autowired(required = false)
    private AuditTemplateSupermarketService auditTemplateSupermarketService;

    @Autowired(required = false)
    private DirectVoService directVoService;

    @Autowired(required = false)
    private DirectStoreVoService directStoreVoService;

    @Autowired(required = false)
    private AuditMatchRepository auditMatchRepository;

    @Autowired(required = false)
    private AuditTemplateRepository auditTemplateRepository;

    @Autowired(required = false)
    private InvoiceAcceptanceVoService invoiceAcceptanceVoService;

    @Autowired(required = false)
    private AuditSapRepository auditSapRepository;

    /**
     * 分页查询数据
     *
     * @param pageable        分页对象
     * @param kmsAuditSummary 实体对象
     * @return
     */
    @Override
    public Page<KmsAuditSummary> findByConditions(Pageable pageable, KmsAuditSummary kmsAuditSummary) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(kmsAuditSummary)) {
            kmsAuditSummary = new KmsAuditSummary();
        }
        return this.kmsAuditSummaryRepository.findByConditions(pageable, kmsAuditSummary);
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public KmsAuditSummary findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        return this.kmsAuditSummaryRepository.getById(id);
    }

    /**
     * 新增数据
     *
     * @param kmsAuditSummary 实体对象
     * @return 新增结果
     */
    @Transactional
    @Override
    public KmsAuditSummary create(KmsAuditSummary kmsAuditSummary) {
        this.createValidate(kmsAuditSummary);
        this.kmsAuditSummaryRepository.saveOrUpdate(kmsAuditSummary);
        return kmsAuditSummary;
    }

    /**
     * 修改新据
     *
     * @param kmsAuditSummary 实体对象
     * @return 修改结果
     */
    @Transactional
    @Override
    public KmsAuditSummary update(KmsAuditSummary kmsAuditSummary) {
        this.updateValidate(kmsAuditSummary);
        this.kmsAuditSummaryRepository.saveOrUpdate(kmsAuditSummary);
        return kmsAuditSummary;
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        this.kmsAuditSummaryRepository.removeByIds(idList);
    }

    @Override
    public void summaryAudit(String auditTemplateCode, String beginDate, String endDate) {
        Validate.isTrue(StringUtils.isNotEmpty(auditTemplateCode),"请输入模板!");
        Validate.isTrue(StringUtils.isNotEmpty(beginDate),"请输入开始时间!");
        Validate.isTrue(StringUtils.isNotEmpty(endDate),"请输入结束时间!");
        //1、根据模板获取系统编码
        AuditTemplate auditTemplate = this.auditTemplateRepository.findByTemplateCode(auditTemplateCode);
        Validate.isTrue(Objects.nonNull(auditTemplate),"未能根据模板编码获取模板!");
        List<AuditTemplateSupermarket> auditTemplateSupermarketList = this.auditTemplateSupermarketService.findByTemplateCodes(Lists.newArrayList(auditTemplateCode));
        Validate.isTrue(!CollectionUtils.isEmpty(auditTemplateSupermarketList),"未能根据模板编码获取模板系统!");
        List<String> directCodes = auditTemplateSupermarketList.stream().map(AuditTemplateSupermarket::getDirectCode).distinct().collect(Collectors.toList());
        //2、查询系统获取组织
        List<DirectVo> directVoList = this.directVoService.findByDirectCodes(directCodes);
        Validate.isTrue(!CollectionUtils.isEmpty(directVoList),"未能根据系统编码获取系统!");
        //3、根据系统获取送达方
        DirectStoreConditionDto dto = new DirectStoreConditionDto();
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
        dto.setDirectCodes(directCodes.stream().collect(Collectors.toSet()));
        List<DirectStoreVo> directStoreVoList = directStoreVoService.findByDirectStoreConditionDto(dto);
        List<String> serviceCodes = directStoreVoList.stream().map(DirectStoreVo::getTerminalCode).distinct().collect(Collectors.toList());
        Validate.isTrue(!CollectionUtils.isEmpty(directStoreVoList), "未根据系统编码查询到门店!");
        Map<String, String> storeMap = directStoreVoList.stream().collect(Collectors.toMap(DirectStoreVo::getTerminalCode, DirectStoreVo::getDirectCode, (a, b) -> a));
        //4、根据模板查询稽核数据
        List<AuditMatch> auditMatchList = this.auditMatchRepository.findByTemplateCodeAndDate(auditTemplateCode,beginDate,endDate);
        Validate.isTrue(!CollectionUtils.isEmpty(auditMatchList),"未根据模板查询到稽核数据!");
        //5、查验收单数据
//        List<InvoiceAcceptanceGoodsVo> acceptanceOrder = this.findAcceptanceByTemplate(auditTemplate, auditMatchList, auditTemplateSupermarketList, beginDate, endDate);
        List<InvoiceAcceptanceGoodsVo> acceptanceOrder = this.invoiceAcceptanceVoService.findAllByConditions(null, null, null, beginDate,endDate, null, directCodes);
        Validate.isTrue(!CollectionUtils.isEmpty(acceptanceOrder),"未根据条件查询到验收单!");
        //6、验收单头转换成功数量
        List<InvoiceAcceptanceVo> acceptanceSuccessOrderVo = this.invoiceAcceptanceVoService.findAcceptanceOrderVo(beginDate, endDate, directCodes);
        //7、查稽核单数据
        Pageable auditSapVoPageable = PageRequest.of(1,KMS_AUDIT_SAP_PAGE_SIZE);
        List<AuditSapVo> auditSapVoList = new ArrayList<>();
        AuditSapDto sapDto = new AuditSapDto();
        sapDto.setTenantCode(TenantUtils.getTenantCode());
        sapDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        sapDto.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
        sapDto.setBeginDate(beginDate.replace("-",""));
        sapDto.setEndDate(endDate.replace("-",""));
        sapDto.setServiceCodes(serviceCodes);
        Page<AuditSapVo> pageByConditions = this.auditSapRepository.findPageByConditions(auditSapVoPageable, sapDto);
        auditSapVoList.addAll(pageByConditions.getRecords());
        while (pageByConditions.hasNext()
                && KMS_AUDIT_SAP_LOOP_MAX >= auditSapVoPageable.getPageNumber()) {
            auditSapVoPageable = auditSapVoPageable.next();
            pageByConditions = this.auditSapRepository.findPageByConditions(auditSapVoPageable, sapDto);
            auditSapVoList.addAll(pageByConditions.getRecords());
        }
        Map<String, List<AuditSapVo>> directSap = this.buildDirectSap(auditSapVoList, storeMap);
        //8、保存
        List<KmsAuditSummary> kmsAuditSummaryList = this.buildSummary(auditTemplate,directVoList, auditMatchList, acceptanceOrder,acceptanceSuccessOrderVo,directSap);
        //9.设置汇总时间
        kmsAuditSummaryList = kmsAuditSummaryList.stream().map(a ->{
            a.setTimeRangeStart(beginDate);
            a.setTimeRangeEnd(endDate);
            return a;
        }).collect(Collectors.toList());
        this.kmsAuditSummaryRepository.saveBatch(kmsAuditSummaryList);
    }

    public Map<String,List<AuditSapVo>> buildDirectSap(List<AuditSapVo> auditSapVoList,Map<String, String> storeMap){
        Map<String,List<AuditSapVo>> map = new HashMap<>();
        if (!CollectionUtils.isEmpty(auditSapVoList)){
            Map<String, List<AuditSapVo>> sapMap = auditSapVoList.stream().collect(Collectors.groupingBy(AuditSapVo::getServiceCode));
            for (Map.Entry<String, List<AuditSapVo>> stringListEntry : sapMap.entrySet()) {
                String directCode = storeMap.get(stringListEntry.getKey());
                if (StringUtils.isNotEmpty(directCode)){
                    List<AuditSapVo> auditSapVos = map.get(directCode) == null ? new ArrayList<>() : map.get(directCode);
                    if (CollectionUtils.isEmpty(auditSapVos)){
                        map.put(directCode,stringListEntry.getValue());
                    }else {
                        map.remove(directCode);
                    }
                    auditSapVos.addAll(stringListEntry.getValue());
                    map.put(directCode,auditSapVos);
                }
            }
            return map;
        }else {
            return map;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteAuditSummary(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids),"未选中数据!");
        this.kmsAuditSummaryRepository.deleteByIds(ids);
    }

    /**
     * 查询验收单
     * @param auditTemplate
     * @param auditMatchList
     * @param auditTemplateSupermarkets
     * @param beginDate
     * @param endDate
     * @return
     */
    private List<InvoiceAcceptanceGoodsVo> findAcceptanceByTemplate(AuditTemplate auditTemplate,List<AuditMatch> auditMatchList,List<AuditTemplateSupermarket> auditTemplateSupermarkets,String beginDate,String endDate){
        List<String> productCodes = Lists.newArrayList();
        List<String> storeCodes = Lists.newArrayList();
        List<String> orderCodes = Lists.newArrayList();
        List<String> soldToPartyCodes = Lists.newArrayList();
        List<String> params = Arrays.stream(auditTemplate.getMatchParams().split(",")).collect(Collectors.toList());
        String maxAcceptanceDate = null;
        String minAcceptanceDate = null;
        if (params.contains(MatchParamsEnum.PRODUCT.getDictCode())) {
            List<String> goodsCodes = auditMatchList.stream().map(AuditMatch::getSapMaterialCode).distinct().collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(goodsCodes)){
                productCodes.addAll(goodsCodes);
            }
        }
        if (params.contains(MatchParamsEnum.STORE.getDictCode())) {
            List<String> deliveryCodes = auditMatchList.stream().map(AuditMatch::getDeliveryCode).distinct().collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(deliveryCodes)){
                storeCodes.addAll(deliveryCodes);
            }
        }
        if (params.contains(MatchParamsEnum.ORDER.getDictCode())) {
            List<String> relationOrderCodes = auditMatchList.stream().map(AuditMatch::getRelationOrderCode).distinct().collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(relationOrderCodes)) {
                orderCodes.addAll(relationOrderCodes);
            }
        }
        //2、1、1 处理容差时间(选容差时间最大的值，一次性吧数据查出来)
        if (params.contains(MatchParamsEnum.DATE.getDictCode())) {
            //获取最大时间和最小时间
            Date maxDate = DateUtil.parseDate(endDate);
            Date minDate = DateUtil.parseDate(beginDate);
            //获取最大时间容差
            Integer delayDay = auditTemplateSupermarkets.stream().filter(entity -> entity.getDelayDays() != null).max(Comparator.comparing(AuditTemplateSupermarket::getDelayDays)).get().getDelayDays();
            //按上下浮动来处理
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(maxDate);
            calendar.add(Calendar.DATE,delayDay);
            maxAcceptanceDate = DateUtil.format(calendar.getTime(),"yyyy-MM-dd");
            calendar.setTime(minDate);
            calendar.add(Calendar.DATE,-1 * delayDay);
            minAcceptanceDate = DateUtil.format(calendar.getTime(),"yyyy-MM-dd");
        }
        if (params.contains(MatchParamsEnum.SOLDTOPARTY.getDictCode())) {
            List<String> soldCodes = auditMatchList.stream().map(AuditMatch::getSoldToPartyCode).distinct().collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(soldCodes)){
                soldToPartyCodes.addAll(soldCodes);
            }
        }
        List<String> directCodes = auditTemplateSupermarkets.stream().map(AuditTemplateSupermarket::getDirectCode).collect(Collectors.toList());
        List<InvoiceAcceptanceGoodsVo> acceptanceOrder = this.invoiceAcceptanceVoService.findAllByConditions(productCodes, storeCodes, orderCodes, minAcceptanceDate,maxAcceptanceDate, soldToPartyCodes, directCodes);
        return acceptanceOrder;
    }

    private List<KmsAuditSummary> buildSummary(AuditTemplate auditTemplate,List<DirectVo> directVoList,List<AuditMatch> auditMatchList,List<InvoiceAcceptanceGoodsVo> acceptanceOrder,List<InvoiceAcceptanceVo> acceptanceSuccessOrderVo,Map<String, List<AuditSapVo>> directSap){
        Map<String, List<AuditMatch>> directAuditMap = auditMatchList.stream().collect(Collectors.groupingBy(AuditMatch::getDirectCode));
        Map<String, List<InvoiceAcceptanceGoodsVo>> acceptanceDirectMap = acceptanceOrder.stream().collect(Collectors.groupingBy(InvoiceAcceptanceGoodsVo::getDirectCode));
        Map<String, List<InvoiceAcceptanceVo>> acceptanceMap = acceptanceSuccessOrderVo.stream().collect(Collectors.groupingBy(InvoiceAcceptanceVo::getDirectCode));
        List<KmsAuditSummary> kmsAuditSummaryList = new ArrayList<>();
        for (DirectVo directVo : directVoList) {
            StringBuffer buffer = new StringBuffer();
            KmsAuditSummary kmsAuditSummary = new KmsAuditSummary();
            //设置销售组织和系统
            kmsAuditSummary.setSalesOrgCode(directVo.getSalesOrgCode());
            kmsAuditSummary.setSalesOrgName(directVo.getSalesOrgName());
            kmsAuditSummary.setDirectCode(directVo.getDirectCode());
            kmsAuditSummary.setDirectName(directVo.getSupermarketName());
            //设置稽核单单据量
            List<AuditMatch> auditMatchDirectList = directAuditMap.get(kmsAuditSummary.getDirectCode());
            if (CollectionUtils.isEmpty(auditMatchDirectList)){
                kmsAuditSummary.setSapTotal(BigDecimal.ZERO);
                kmsAuditSummary.setSapTotalMatched(BigDecimal.ZERO);
            }else {
                kmsAuditSummary.setSapTotal(new BigDecimal(auditMatchDirectList.size()));
                //设置已匹配105单据量
                List<AuditMatch> auditMatchMatchedList = auditMatchDirectList.stream().filter(vo -> Lists.newArrayList(MatchStatusEnum.M200.getDictCode(),MatchStatusEnum.M300.getDictCode()).contains(vo.getMatchStatus())).collect(Collectors.toList());
                kmsAuditSummary.setSapTotalMatched(new BigDecimal(auditMatchMatchedList.size()));
            }
            //设置105原始数据量
            List<AuditSapVo> auditSapVos = directSap.get(kmsAuditSummary.getDirectCode());
            if (CollectionUtils.isEmpty(auditSapVos)){
                kmsAuditSummary.setGrabSapTotal(BigDecimal.ZERO);
            }else {
                kmsAuditSummary.setGrabSapTotal(new BigDecimal(auditSapVos.size()));
            }
            //设置验收单头信息
            List<InvoiceAcceptanceVo> acceptanceVoList = acceptanceMap.get(kmsAuditSummary.getDirectCode());
            if (CollectionUtils.isEmpty(acceptanceVoList)){
                kmsAuditSummary.setAcceptanceTotal(BigDecimal.ZERO);
            }else {
                kmsAuditSummary.setAcceptanceTotal(new BigDecimal(acceptanceVoList.size()));
            }
            if (!CollectionUtils.isEmpty(acceptanceVoList)){
                List<InvoiceAcceptanceVo> invoiceAcceptanceVos = acceptanceVoList.stream().filter(vo -> StringUtils.equals(AcceptanceStatus.S200.getDictCode(), vo.getOrderStatus())).collect(Collectors.toList());
                if (CollectionUtils.isEmpty(invoiceAcceptanceVos)){
                    kmsAuditSummary.setAcceptanceOrderTotal(BigDecimal.ZERO);
                    kmsAuditSummary.setAcceptanceSuccessRate(BigDecimal.ZERO);
                }else {
                    BigDecimal allTotal = new BigDecimal(acceptanceVoList.size());
                    BigDecimal total = new BigDecimal(invoiceAcceptanceVos.size());
                    BigDecimal divide = total.divide(allTotal,4,BigDecimal.ROUND_DOWN).multiply(new BigDecimal(100));
                    kmsAuditSummary.setAcceptanceOrderTotal(new BigDecimal(invoiceAcceptanceVos.size()));
                    kmsAuditSummary.setAcceptanceSuccessRate(divide);
                }
            }else {
                kmsAuditSummary.setAcceptanceSuccessRate(BigDecimal.ZERO);
                kmsAuditSummary.setAcceptanceOrderTotal(BigDecimal.ZERO);
            }
            //设置匹配成功率
            if (kmsAuditSummary.getSapTotal() != null && kmsAuditSummary.getSapTotal().compareTo(BigDecimal.ZERO) != 0){
                kmsAuditSummary.setAuditSuccessRate(kmsAuditSummary.getSapTotalMatched().divide(kmsAuditSummary.getSapTotal(),4,BigDecimal.ROUND_DOWN).multiply(new BigDecimal(100)));
            }else {
                kmsAuditSummary.setAuditSuccessRate(BigDecimal.ZERO);
            }
            //设置差异率
            kmsAuditSummary.setAuditDifferRate(new BigDecimal(100).subtract(kmsAuditSummary.getAuditSuccessRate()));
            List<InvoiceAcceptanceGoodsVo> acceptanceGoodsDirectList = acceptanceDirectMap.get(kmsAuditSummary.getDirectCode());
            //设置差异项
            if (!CollectionUtils.isEmpty(acceptanceGoodsDirectList)){
                List<InvoiceAcceptanceGoodsVo> acceptanceOrderFail = acceptanceGoodsDirectList.stream().filter(vo -> StringUtils.equals(MatchStatusEnum.M100.getDictCode(),vo.getOrderStatus()) && StringUtils.isNotEmpty(vo.getOrderStatusMsg())).collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(acceptanceOrderFail)){
                    //设置失败门店(转换状态100是失败)
                    List<InvoiceAcceptanceGoodsVo> failedDelivery = acceptanceOrderFail.stream().filter(vo -> vo.getOrderStatusMsg().contains("门店")).collect(Collectors.toList());
                    if (!CollectionUtils.isEmpty(failedDelivery)){
                        kmsAuditSummary.setMatchFailedDelivery(new BigDecimal(failedDelivery.size()));
                    }else {
                        kmsAuditSummary.setMatchFailedDelivery(BigDecimal.ZERO);
                    }
                    //设置失败产品
                    List<InvoiceAcceptanceGoodsVo> failedProduct = acceptanceOrderFail.stream().filter(vo -> vo.getOrderStatusMsg().contains("商品")).collect(Collectors.toList());
                    if (!CollectionUtils.isEmpty(failedProduct)){
                        kmsAuditSummary.setMatchFailedProduct(new BigDecimal(failedProduct.size()));
                    }else {
                        kmsAuditSummary.setMatchFailedProduct(BigDecimal.ZERO);
                    }
                    //设置差异类型
                    if (!CollectionUtils.isEmpty(failedDelivery) || !failedProduct.isEmpty()){
                        buffer.append("存在主数据缺失||");
                    }
                }else {
                    kmsAuditSummary.setMatchFailedDelivery(BigDecimal.ZERO);
                    kmsAuditSummary.setMatchFailedProduct(BigDecimal.ZERO);
                }
                if (kmsAuditSummary.getSapTotal().compareTo(BigDecimal.ZERO) != 0 && kmsAuditSummary.getAcceptanceTotal() != null && kmsAuditSummary.getAcceptanceTotal().compareTo(BigDecimal.ZERO) != 0){
                    BigDecimal differ = kmsAuditSummary.getAcceptanceTotal().divide(kmsAuditSummary.getSapTotal(),4,BigDecimal.ROUND_DOWN).multiply(new BigDecimal(100));
                    if (differ.compareTo(new BigDecimal(50)) <= 0){
                        buffer.append("存在虚出库||");
                    }
                }
                if (kmsAuditSummary.getSapTotal().compareTo(kmsAuditSummary.getAcceptanceTotal()) == 0){
                    if (kmsAuditSummary.getAuditSuccessRate().compareTo(BigDecimal.ONE) != 0){
                        buffer.append("存在跨门店验收");
                    }
                }
            }else {
                buffer.append("存在主数据缺失||");
            }
            kmsAuditSummary.setDifferType(String.valueOf(buffer));
            kmsAuditSummary.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            kmsAuditSummary.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
            kmsAuditSummary.setTenantCode(TenantUtils.getTenantCode());
            kmsAuditSummaryList.add(kmsAuditSummary);
            kmsAuditSummary.setTemplateCode(auditTemplate.getTemplateCode());
        }
        return kmsAuditSummaryList;
    }

    /**
     * 创建验证
     *
     * @param kmsAuditSummary
     */
    private void createValidate(KmsAuditSummary kmsAuditSummary) {
        Validate.notNull(kmsAuditSummary, "新增时，对象信息不能为空！");
        kmsAuditSummary.setId(null);
        Validate.notBlank(kmsAuditSummary.getTenantCode(), "新增数据时，租户编号不能为空！");
    }

    /**
     * 修改验证
     *
     * @param kmsAuditSummary
     */
    private void updateValidate(KmsAuditSummary kmsAuditSummary) {
        Validate.notNull(kmsAuditSummary, "修改时，对象信息不能为空！");
        Validate.notBlank(kmsAuditSummary.getId(), "新增数据时，不能为空！");
        Validate.notBlank(kmsAuditSummary.getTenantCode(), "新增数据时，租户编号不能为空！");
    }
}

