package com.biz.crm.kms.business.audit.match.rule.abstracts;

import cn.hutool.core.date.DateUtil;
import com.biz.crm.kms.business.audit.match.local.entity.*;
import com.biz.crm.kms.business.audit.match.local.model.AuditMatchInvoiceModel;
import com.biz.crm.kms.business.audit.match.local.model.MatchConsequenceCalculatedAbstract;
import com.biz.crm.kms.business.audit.match.local.model.MatchConsequenceCalculatedModel;
import com.biz.crm.kms.business.audit.match.local.repository.AuditMatchCrossAcceptanceRepository;
import com.biz.crm.kms.business.audit.match.local.repository.AuditMatchInvoiceRepository;
import com.biz.crm.kms.business.audit.match.local.repository.MatchInvoiceManuRepository;
import com.biz.crm.kms.business.audit.match.local.service.AuditMatchService;
import com.biz.crm.kms.business.audit.match.sdk.constant.AuditMatchConstant;
import com.biz.crm.kms.business.audit.match.sdk.dto.AuditMatchNotReasonDto;
import com.biz.crm.kms.business.audit.match.sdk.enums.AuditMatchCrossExceptionEnum;
import com.biz.crm.kms.business.audit.match.sdk.enums.AuditMatchCrossTypeEnum;
import com.biz.crm.kms.business.audit.match.sdk.enums.MatchParamsEnum;
import com.biz.crm.kms.business.audit.match.sdk.enums.PriceTypeEnum;
import com.biz.crm.kms.business.invoice.acceptance.sdk.enums.AcceptanceStatus;
import com.biz.crm.kms.business.invoice.sdk.service.InvoiceStatisticsVoService;
import com.biz.crm.kms.business.invoice.sdk.vo.InvoiceStatisticsVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 出货数据(第三方稽核匹配数据)=验收单 计算逻辑策略实现
 *
 * @author songjingen
 * @date 2022/10/28 9:45
 */
@Component
@Slf4j
public class AcceptanceEqualToShipmentCalculatedAbstract extends MatchConsequenceCalculatedAbstract {

  @Autowired(required = false)
  private InvoiceStatisticsVoService invoiceStatisticsVoService;
  @Autowired(required = false)
  private AuditMatchInvoiceRepository auditMatchInvoiceRepository;

  @Autowired(required = false)
  private MatchInvoiceManuRepository matchInvoiceManuRepository;

  @Autowired(required = false)
  private AuditMatchService auditMatchService;

  @Autowired(required = false)
  private AuditMatchCrossAcceptanceRepository auditMatchCrossAcceptanceRepository;

  /**
   * 1、this.covertMatchLogicCalculatedModels(auditTemplateSupermarketModels)进行数据转换； 2、获取单据的数据，并且进行赋值 ；3、this.calculate(matchLogicCalculatedModels)执行计算逻辑
   *
   * @param templateCode 模板编码
   * @param calculatedModels       匹配结果数据
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void execute(String templateCode, List<MatchConsequenceCalculatedModel> calculatedModels) {
    Validate.isTrue(!CollectionUtils.isEmpty(calculatedModels), "执行匹配计算逻辑时，需要匹配的数据不存在！");
    //1、查询单据
    List<Date> invoiceDates = calculatedModels.stream().map(MatchConsequenceCalculatedModel::getSapPostingDate).collect(Collectors.toList());
    List<String> goodsCodes = calculatedModels.stream().map(MatchConsequenceCalculatedModel::getSapMaterialCode).collect(Collectors.toList());
    List<String> storeCodes = calculatedModels.stream().map(MatchConsequenceCalculatedModel::getDeliveryCode).collect(Collectors.toList());
    //1.1、查询验收单数据,并且按照匹配条件转成map
    List<InvoiceStatisticsVo> invoiceStatisticsVos = this.invoiceStatisticsVoService.findByType(AuditMatchConstant.ACCEPTANCE, invoiceDates, goodsCodes, storeCodes, null);
    Validate.isTrue(!CollectionUtils.isEmpty(invoiceStatisticsVos), "执行匹配计算逻辑时，未匹配到验收单！");
    Map<String, List<InvoiceStatisticsVo>> groupMap = invoiceStatisticsVos.stream()
            .collect(Collectors.groupingBy(o -> StringUtils.joinWith("", o.getStoreCode(), o.getGoodsCode(), DateUtil.format(o.getInvoiceDate(), "yyyy-MM-dd"))));
    //1.2、查询单据已匹配的稽核数据，并且根据单据类型和单据编码转成map
    List<String> invoiceCodes = invoiceStatisticsVos.stream().map(InvoiceStatisticsVo::getInvoiceCode).collect(Collectors.toList());
    List<AuditMatchInvoice> auditMatchInvoices = this.auditMatchInvoiceRepository.findByTemplateCodeAndInvoiceTypesAndInvoiceCodes(templateCode,
            Lists.newArrayList(AuditMatchConstant.ACCEPTANCE, AuditMatchConstant.RETURN), invoiceCodes);
    Map<String, String> auditCodeMapByInvoiceMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(auditMatchInvoices)) {
      auditCodeMapByInvoiceMap = auditMatchInvoices.stream()
              .collect(Collectors.toMap(o -> StringUtils.joinWith("", o.getInvoiceCode(), o.getInvoiceType()), AuditMatchInvoice::getAuditCode, (a, b) -> a));
    }
    //2、赋值
    ArrayList<AuditMatchInvoiceModel> invoiceModels = new ArrayList<>();
    for (MatchConsequenceCalculatedModel calculatedModel : calculatedModels) {
      BigDecimal invoiceTotalCount = BigDecimal.ZERO;
      BigDecimal invoiceTotalAmount = BigDecimal.ZERO;
      BigDecimal invoiceTotalAmountTaxExclusive = BigDecimal.ZERO;
      String mapKey = StringUtils.joinWith("", calculatedModel.getDeliveryCode(), calculatedModel.getSapMaterialCode(), DateUtil.format(calculatedModel.getSapPostingDate(), "yyyy-MM-dd"));
      List<InvoiceStatisticsVo> statisticsVos = groupMap.get(mapKey);
      if (CollectionUtils.isEmpty(statisticsVos)) {
        continue;
      }
      for (InvoiceStatisticsVo statisticsVo : statisticsVos) {
        //如果当前单据已经匹配了数据并且和当前稽核编码不匹配，则跳过统计
        String key = StringUtils.joinWith("", statisticsVo.getInvoiceCode(), statisticsVo.getInvoiceType());
        String auditCode = auditCodeMapByInvoiceMap.get(key);
        if (StringUtils.isNotBlank(auditCode) && !auditCode.equals(calculatedModel.getAuditCode())) {
          continue;
        }
        invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
        invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
        invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
        //将已使用的单据匹配到稽核数据上
        auditCodeMapByInvoiceMap.put(key, calculatedModel.getAuditCode());
        //绑定单据
        AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
        invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
        invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
        invoiceModel.setTemplateCode(templateCode);
        invoiceModel.setAuditCode(calculatedModel.getAuditCode());
        invoiceModels.add(invoiceModel);
      }
      calculatedModel.setInvoiceTotalCount(invoiceTotalCount);
      calculatedModel.setInvoiceTotalAmount(invoiceTotalAmount);
      calculatedModel.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
      calculatedModel.setAuditMatchInvoiceModels(invoiceModels);
    }
    //执行计算
    this.calculate(calculatedModels,null);
  }

  @Override
  public void executeAuto(AuditTemplate auditTemplate, List<MatchConsequenceCalculatedModel> models) {
    if (CollectionUtils.isEmpty(models)) {
      log.error("执行匹配计算逻辑时，需要匹配的数据不存在！");
      return;
    }
    List<MatchConsequenceCalculatedModel> modelList = new ArrayList<>();
    //1、根据稽核模板获取商超
    List<AuditTemplateSupermarket> templateSupermarkets = auditTemplate.getTemplateSupermarkets();
    if (CollectionUtils.isEmpty(templateSupermarkets)) {
      log.error("未能根据[%s]模板信息查询到商超模板!", auditTemplate.getTemplateCode());
      return;
    }
    Map<String, AuditTemplateSupermarket> auditTemplateSupermarketMap = templateSupermarkets.stream().collect(Collectors.toMap(AuditTemplateSupermarket::getDirectCode, Function.identity()));
    //2、构建查询条件,查询验收单
    List<String> productCodes = Lists.newArrayList();
    List<String> storeCodes = Lists.newArrayList();
    List<String> orderCodes = Lists.newArrayList();
    List<String> soldToPartyCodes = Lists.newArrayList();
    List<String> params = Arrays.stream(auditTemplate.getMatchParams().split(",")).collect(Collectors.toList());
    String maxAcceptanceDate = null;
    String minAcceptanceDate = null;
    if (params.contains(MatchParamsEnum.PRODUCT.getDictCode())) {
      List<String> goodsCodes = models.stream().map(MatchConsequenceCalculatedModel::getSapMaterialCode).distinct().collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(goodsCodes)) {
        productCodes.addAll(goodsCodes);
      }
    }
    if (params.contains(MatchParamsEnum.STORE.getDictCode())) {
      List<String> deliveryCodes = models.stream().map(MatchConsequenceCalculatedModel::getDeliveryCode).distinct().collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(deliveryCodes)) {
        storeCodes.addAll(deliveryCodes);
      }
    }
    if (params.contains(MatchParamsEnum.ORDER.getDictCode())) {
      List<String> relationOrderCodes = models.stream().map(MatchConsequenceCalculatedModel::getRelationOrderCode).distinct().collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(relationOrderCodes)) {
        orderCodes.addAll(relationOrderCodes);
      }
    }
    //3、处理容差时间(选容差时间最大的值，一次性吧数据查出来)
    if (params.contains(MatchParamsEnum.DATE.getDictCode())) {
      //获取最大时间和最小时间
      Date maxDate = models.stream().max(Comparator.comparing(MatchConsequenceCalculatedModel::getSapPostingDate)).get().getSapPostingDate();
      Date minDate = models.stream().min(Comparator.comparing(MatchConsequenceCalculatedModel::getSapPostingDate)).get().getSapPostingDate();
      //获取最大时间容差
      Integer delayDay = templateSupermarkets.stream().filter(entity -> entity.getDelayDays() != null).max(Comparator.comparing(AuditTemplateSupermarket::getDelayDays)).get().getDelayDays();
      //按上下浮动来处理
      Calendar calendar = Calendar.getInstance();
      calendar.setTime(maxDate);
      calendar.add(Calendar.DATE, delayDay);
      maxAcceptanceDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
      calendar.setTime(minDate);
      calendar.add(Calendar.DATE, -1 * delayDay);
      minAcceptanceDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
    }
    if (params.contains(MatchParamsEnum.SOLDTOPARTY.getDictCode())) {
      List<String> soldCodes = models.stream().map(MatchConsequenceCalculatedModel::getSoldToPartyCode).distinct().collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(soldCodes)) {
        soldToPartyCodes.addAll(soldCodes);
      }
    }
    List<String> directCodes = templateSupermarkets.stream().map(AuditTemplateSupermarket::getDirectCode).collect(Collectors.toList());
    //4、查询单据
    List<InvoiceStatisticsVo> invoiceStatisticsVos = this.invoiceStatisticsVoService.findByConditions(AuditMatchConstant.ACCEPTANCE, productCodes, storeCodes, orderCodes, minAcceptanceDate, maxAcceptanceDate, soldToPartyCodes, directCodes);
    List<InvoiceStatisticsVo> successInvoiceStatisticsVos = invoiceStatisticsVos.stream().filter(vo -> StringUtils.equals(AcceptanceStatus.S200.getDictCode(), vo.getOrderStatus())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(successInvoiceStatisticsVos)) {
      if (params.contains(MatchParamsEnum.ORDER.getDictCode())) {
        log.info("未根据条件查询到验收单,该次匹配退出!");
        this.updateNotReason(models, invoiceStatisticsVos);
        return;
      } else {
        log.info("未根据条件查询到验收单,该次匹配退出!");
        return;
      }
    }
    //5、根据条件查询出来的验收单分组
    if (params.contains(MatchParamsEnum.ORDER.getDictCode())) {
      successInvoiceStatisticsVos = successInvoiceStatisticsVos.stream().filter(vo -> StringUtils.isNotEmpty(vo.getOrderCode())).collect(Collectors.toList());
    } else {
      successInvoiceStatisticsVos = successInvoiceStatisticsVos.stream().filter(vo -> StringUtils.isEmpty(vo.getOrderCode())).collect(Collectors.toList());
    }
    if (CollectionUtils.isEmpty(successInvoiceStatisticsVos)) {
      log.info("未根据条件查询到转换成功的验收单,该次匹配退出!");
      this.updateNotReason(models, invoiceStatisticsVos);
      return;
    }
    Map<String, List<InvoiceStatisticsVo>> acceptanceMap = new HashMap<>();
    //产品+售达方+日期(不进行匹配)
    if (CollectionUtils.isEmpty(storeCodes) && CollectionUtils.isEmpty(orderCodes)) {
      return;
    }
    //产品+售达方+日期+送达方
    if (params.size() == 4 && CollectionUtils.isEmpty(orderCodes)) {
      acceptanceMap = successInvoiceStatisticsVos.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("_", vo.getGoodsCode(), vo.getStoreCode(), vo.getSoldToPartyCode())));
    }
    //产品+售达方+日期+采购单号
    if (params.size() == 4 && CollectionUtils.isEmpty(storeCodes)) {
      acceptanceMap = successInvoiceStatisticsVos.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("_", vo.getGoodsCode(), vo.getOrderCode(), vo.getSoldToPartyCode())));
    }
    //全条件
    if (params.size() == 5) {
      acceptanceMap = successInvoiceStatisticsVos.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("_", vo.getGoodsCode(), vo.getStoreCode(), vo.getOrderCode(), vo.getSoldToPartyCode())));
    }
    if (MapUtils.isEmpty(acceptanceMap)) {
      return;
    }
      for (MatchConsequenceCalculatedModel model : models) {
        ArrayList<AuditMatchInvoiceModel> invoiceModels = new ArrayList<>();
        String mapKey = null;
        //产品+售达方+日期+送达方
        if (params.size() == 4 && CollectionUtils.isEmpty(orderCodes)) {
          mapKey = StringUtils.joinWith("_", model.getSapMaterialCode(),model.getDeliveryCode() , model.getSoldToPartyCode());
        }
        //产品+售达方+日期+采购单号
        if (params.size() == 4 && CollectionUtils.isEmpty(storeCodes)) {
          mapKey = StringUtils.joinWith("_", model.getSapMaterialCode(),model.getRelationOrderCode() , model.getSoldToPartyCode());
        }
        //全条件
        if (params.size() == 5) {
          mapKey = StringUtils.joinWith("_", model.getSapMaterialCode(), model.getDeliveryCode() ,model.getRelationOrderCode(), model.getSoldToPartyCode());
        }
        List<InvoiceStatisticsVo> invoiceStatisticsList = acceptanceMap.get(mapKey);
        if (CollectionUtils.isEmpty(invoiceStatisticsList)) {
          modelList.add(model);
          continue;
        }
        AuditTemplateSupermarket auditTemplateSupermarket = auditTemplateSupermarketMap.get(model.getDirectCode());
        if (auditTemplateSupermarket == null) {
          continue;
        }
        model.setPriceValue(auditTemplateSupermarket.getPriceValue());
        model.setPriceType(auditTemplateSupermarket.getPriceType());
        List<InvoiceStatisticsVo> matchInvoices = new ArrayList<>();
        BigDecimal invoiceTotalCount = BigDecimal.ZERO;
        BigDecimal invoiceTotalAmount = BigDecimal.ZERO;
        BigDecimal invoiceTotalAmountTaxExclusive = BigDecimal.ZERO;
        //2、3、2判断有模板条件有无日期,处理容差日期问题
        if (params.contains(MatchParamsEnum.DATE.getDictCode())) {
          if (StringUtils.isNotEmpty(auditTemplateSupermarket.getDelayType())) {
            Integer delayDays = auditTemplateSupermarket.getDelayDays();
            String beginDate = null;
            String endDate = null;
            String posDate = DateUtil.format(model.getSapPostingDate(), "yyyy-MM-dd");
            Calendar calendar = Calendar.getInstance();
            switch (PriceTypeEnum.getByDictCode(auditTemplateSupermarket.getDelayType())) {
              case Z:
                calendar.setTime(model.getSapPostingDate());
                calendar.add(Calendar.DATE, delayDays);
                endDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                for (InvoiceStatisticsVo invoiceStatisticsVo : invoiceStatisticsList) {
                  String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                  if (invoiceDate != null && invoiceDate.compareTo(posDate) >= 0 && invoiceDate.compareTo(endDate) <= 0) {
                    matchInvoices.add(invoiceStatisticsVo);
                  }
                }
                break;
              case F:
                calendar.setTime(model.getSapPostingDate());
                calendar.add(Calendar.DATE, -1 * delayDays);
                beginDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                for (InvoiceStatisticsVo invoiceStatisticsVo : invoiceStatisticsList) {
                  String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                  if (invoiceDate != null && invoiceDate.compareTo(beginDate) >= 0 && invoiceDate.compareTo(posDate) <= 0) {
                    matchInvoices.add(invoiceStatisticsVo);
                  }
                }
                break;
              case UD:
                calendar.setTime(model.getSapPostingDate());
                calendar.add(Calendar.DATE, delayDays);
                endDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                calendar.setTime(model.getSapPostingDate());
                calendar.add(Calendar.DATE, -1 * delayDays);
                beginDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                for (InvoiceStatisticsVo invoiceStatisticsVo : invoiceStatisticsList) {
                  String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                  if (invoiceDate != null && invoiceDate.compareTo(beginDate) >= 0 && invoiceDate.compareTo(endDate) <= 0) {
                    matchInvoices.add(invoiceStatisticsVo);
                  }
                }
                break;
              default:
                break;
            }
          }
        }else {
          matchInvoices.addAll(invoiceStatisticsList);
        }
        if (CollectionUtils.isEmpty(matchInvoices)) {
          modelList.add(model);
          continue;
        }
        if (matchInvoices.size() == 1){
          for (InvoiceStatisticsVo statisticsVo : matchInvoices) {
            invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
            invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
            invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
            AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
            invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
            invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
            invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
            invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
            invoiceModel.setAuditCode(model.getAuditCode());
            invoiceModels.add(invoiceModel);
          }
        }else {
          for (InvoiceStatisticsVo statisticsVo : matchInvoices) {
            if (model.getSapEaCount().compareTo(statisticsVo.getInvoiceCount()) != 0){
              continue;
            }
            invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
            invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
            invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
            AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
            invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
            invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
            invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
            invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
            invoiceModel.setAuditCode(model.getAuditCode());
            invoiceModels.add(invoiceModel);
            break;
          }
        }
        model.setInvoiceTotalCount(invoiceTotalCount);
        model.setInvoiceTotalAmount(invoiceTotalAmount);
        model.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
        model.setAuditMatchInvoiceModels(invoiceModels);
        model.setInvoicePrice(matchInvoices.get(0).getInvoicePrice());
        model.setInvoicePriceNoTax(matchInvoices.get(0).getInvoicePriceNoTax());
      }
    //两种路径、1、普通匹配 2、窜单匹配
    List<MatchConsequenceCalculatedModel> crossModels = new ArrayList<>();
    List<String> verifyCodes = models.stream().map(MatchConsequenceCalculatedModel::getVerifyCode).distinct().collect(Collectors.toList());
    List<AuditMatchCrossAcceptance> crossAuditVo = this.auditMatchCrossAcceptanceRepository.findAuditCrossAcceptanceByVerifyCodes(verifyCodes, TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(crossAuditVo)) {
      List<String> conVerifyCodes = crossAuditVo.stream().map(AuditMatchCrossAcceptance::getVerifyCode).distinct().collect(Collectors.toList());
      crossModels = models.stream().filter(vo -> conVerifyCodes.contains(vo.getVerifyCode())).collect(Collectors.toList());
    }
    models.removeAll(crossModels);
    //普通匹配
    for (MatchConsequenceCalculatedModel model : models) {
      ArrayList<AuditMatchInvoiceModel> invoiceModels = new ArrayList<>();
      String mapKey = null;
      //产品+售达方+日期+送达方
      if (params.size() == 4 && CollectionUtils.isEmpty(orderCodes)) {
        mapKey = StringUtils.joinWith("_", model.getSapMaterialCode(), model.getDeliveryCode(), model.getSoldToPartyCode());
      }
      //产品+售达方+日期+采购单号
      if (params.size() == 4 && CollectionUtils.isEmpty(storeCodes)) {
        mapKey = StringUtils.joinWith("_", model.getSapMaterialCode(), model.getRelationOrderCode(), model.getSoldToPartyCode());
      }
      //全条件
      if (params.size() == 5) {
        mapKey = StringUtils.joinWith("_", model.getSapMaterialCode(), model.getDeliveryCode(), model.getRelationOrderCode(), model.getSoldToPartyCode());
      }
      List<InvoiceStatisticsVo> invoiceStatisticsList = acceptanceMap.get(mapKey);
      if (CollectionUtils.isEmpty(invoiceStatisticsList)) {
        modelList.add(model);
        continue;
      }
      AuditTemplateSupermarket auditTemplateSupermarket = auditTemplateSupermarketMap.get(model.getDirectCode());
      if (auditTemplateSupermarket == null) {
        continue;
      }
      model.setPriceValue(auditTemplateSupermarket.getPriceValue());
      model.setPriceType(auditTemplateSupermarket.getPriceType());
      List<InvoiceStatisticsVo> matchInvoices = new ArrayList<>();
      BigDecimal invoiceTotalCount = BigDecimal.ZERO;
      BigDecimal invoiceTotalAmount = BigDecimal.ZERO;
      BigDecimal invoiceTotalAmountTaxExclusive = BigDecimal.ZERO;
      //2、3、2判断有模板条件有无日期,处理容差日期问题
      if (params.contains(MatchParamsEnum.DATE.getDictCode())) {
        if (StringUtils.isNotEmpty(auditTemplateSupermarket.getDelayType())) {
          Integer delayDays = auditTemplateSupermarket.getDelayDays();
          String beginDate = null;
          String endDate = null;
          String posDate = DateUtil.format(model.getSapPostingDate(), "yyyy-MM-dd");
          Calendar calendar = Calendar.getInstance();
          switch (PriceTypeEnum.getByDictCode(auditTemplateSupermarket.getDelayType())) {
            case Z:
              calendar.setTime(model.getSapPostingDate());
              calendar.add(Calendar.DATE, delayDays);
              endDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
              for (InvoiceStatisticsVo invoiceStatisticsVo : invoiceStatisticsList) {
                String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                if (invoiceDate != null && invoiceDate.compareTo(posDate) >= 0 && invoiceDate.compareTo(endDate) <= 0) {
                  matchInvoices.add(invoiceStatisticsVo);
                }
              }
              break;
            case F:
              calendar.setTime(model.getSapPostingDate());
              calendar.add(Calendar.DATE, -1 * delayDays);
              beginDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
              for (InvoiceStatisticsVo invoiceStatisticsVo : invoiceStatisticsList) {
                String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                if (invoiceDate != null && invoiceDate.compareTo(beginDate) >= 0 && invoiceDate.compareTo(posDate) <= 0) {
                  matchInvoices.add(invoiceStatisticsVo);
                }
              }
              break;
            case UD:
              calendar.setTime(model.getSapPostingDate());
              calendar.add(Calendar.DATE, delayDays);
              endDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
              calendar.setTime(model.getSapPostingDate());
              calendar.add(Calendar.DATE, -1 * delayDays);
              beginDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
              for (InvoiceStatisticsVo invoiceStatisticsVo : invoiceStatisticsList) {
                String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                if (invoiceDate != null && invoiceDate.compareTo(beginDate) >= 0 && invoiceDate.compareTo(endDate) <= 0) {
                  matchInvoices.add(invoiceStatisticsVo);
                }
              }
              break;
            default:
              break;
          }
        }
      } else {
        matchInvoices.addAll(invoiceStatisticsList);
      }
      if (CollectionUtils.isEmpty(matchInvoices)) {
        modelList.add(model);
        continue;
      }
      if (matchInvoices.size() == 1) {
        for (InvoiceStatisticsVo statisticsVo : matchInvoices) {
          invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
          invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
          invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
          AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
          invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
          invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
          invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
          invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
          invoiceModel.setAuditCode(model.getAuditCode());
          invoiceModels.add(invoiceModel);
        }
      } else {
        for (InvoiceStatisticsVo statisticsVo : matchInvoices) {
          if (model.getSapEaCount().compareTo(statisticsVo.getInvoiceCount()) != 0) {
            continue;
          }
          invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
          invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
          invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
          AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
          invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
          invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
          invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
          invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
          invoiceModel.setAuditCode(model.getAuditCode());
          invoiceModels.add(invoiceModel);
          break;
        }
      }
      model.setInvoiceTotalCount(invoiceTotalCount);
      model.setInvoiceTotalAmount(invoiceTotalAmount);
      model.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
      model.setAuditMatchInvoiceModels(invoiceModels);
      model.setInvoicePrice(matchInvoices.get(0).getInvoicePrice());
    }
    //窜单匹配
    if (!CollectionUtils.isEmpty(crossAuditVo)){
      Map<String, List<AuditMatchCrossAcceptance>> crossMap = crossAuditVo.stream().collect(Collectors.groupingBy(AuditMatchCrossAcceptance::getVerifyCode));
      //窜单找验收单
      List<String> crossRelationKaOrderCode = crossAuditVo.stream().filter(vo -> StringUtils.equals(AuditMatchCrossTypeEnum.RELATIONKACODE.getDictCode(), vo.getCrossOrderType())).map(AuditMatchCrossAcceptance::getCrossOrderNumber).collect(Collectors.toList());
      List<String> crossAcceptanceOrderNumber = crossAuditVo.stream().filter(vo -> StringUtils.equals(AuditMatchCrossTypeEnum.ORDERNUMBER.getDictCode(), vo.getCrossOrderType())).map(AuditMatchCrossAcceptance::getCrossOrderNumber).collect(Collectors.toList());
      List<InvoiceStatisticsVo> crossInvoiceStatisticsVos = this.invoiceStatisticsVoService.findCrossByConditions(AuditMatchConstant.ACCEPTANCE, crossRelationKaOrderCode,crossAcceptanceOrderNumber);
      List<InvoiceStatisticsVo> crossSuccessInvoiceStatisticsVos = crossInvoiceStatisticsVos.stream().filter(vo -> StringUtils.equals(AcceptanceStatus.S200.getDictCode(), vo.getOrderStatus())).collect(Collectors.toList());
      Map<String, List<InvoiceStatisticsVo>> orderNumberMap = crossSuccessInvoiceStatisticsVos.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("-", vo.getOrderNumber(), vo.getGoodsCode())));
      Map<String, List<InvoiceStatisticsVo>> orderCodeMap = crossSuccessInvoiceStatisticsVos.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("-", vo.getOrderCode(), vo.getGoodsCode())));
      if (CollectionUtils.isEmpty(crossSuccessInvoiceStatisticsVos)) {
        if (params.contains(MatchParamsEnum.ORDER.getDictCode())) {
          log.info("未根据条件查询到验收单,该次窜单匹配退出!");
          this.updateNotReason(crossModels, crossInvoiceStatisticsVos);
        } else {
          log.info("未根据条件查询到验收单,该次窜单匹配退出!");
        }
      }else {
        for (MatchConsequenceCalculatedModel crossModel : crossModels) {
          AuditTemplateSupermarket auditTemplateSupermarket = auditTemplateSupermarketMap.get(crossModel.getDirectCode());
          if (auditTemplateSupermarket == null) {
            continue;
          }
          crossModel.setPriceValue(auditTemplateSupermarket.getPriceValue());
          crossModel.setPriceType(auditTemplateSupermarket.getPriceType());
          BigDecimal invoiceTotalCount = BigDecimal.ZERO;
          BigDecimal invoiceTotalAmount = BigDecimal.ZERO;
          BigDecimal invoiceTotalAmountTaxExclusive = BigDecimal.ZERO;
          BigDecimal price = BigDecimal.ZERO;
          BigDecimal priceNot = BigDecimal.ZERO;
          List<AuditMatchCrossAcceptance> auditMatchCrossAcceptances = crossMap.get(crossModel.getVerifyCode());
          AuditMatchCrossAcceptance auditMatchCrossAcceptance = auditMatchCrossAcceptances.get(0);
          if (StringUtils.equals(AuditMatchCrossExceptionEnum.WHOLEORDER.getDictCode(),auditMatchCrossAcceptance.getExceptionType())){
            List<AuditMatchInvoiceModel> invoiceModels = new ArrayList<>();
            if (StringUtils.equals(AuditMatchCrossTypeEnum.RELATIONKACODE.getDictCode(),auditMatchCrossAcceptance.getCrossOrderType())){
              String key = StringUtils.joinWith("-",auditMatchCrossAcceptance.getCrossOrderNumber(),auditMatchCrossAcceptance.getSapMaterialCode());
              List<InvoiceStatisticsVo> waitMatchInvoices = orderCodeMap.get(key);
              if (CollectionUtils.isEmpty(waitMatchInvoices)){
                modelList.add(crossModel);
              }else {
                if (waitMatchInvoices.size() == 1){
                  invoiceTotalCount = invoiceTotalCount.add(waitMatchInvoices.get(0).getInvoiceCount());
                  invoiceTotalAmount = invoiceTotalAmount.add(waitMatchInvoices.get(0).getInvoiceAmount());
                  invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(waitMatchInvoices.get(0).getInvoiceAmountTaxExclusive());
                  AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                  invoiceModel.setInvoiceCode(waitMatchInvoices.get(0).getInvoiceCode());
                  invoiceModel.setRelationId(waitMatchInvoices.get(0).getInvoiceCode());
                  invoiceModel.setInvoiceType(waitMatchInvoices.get(0).getInvoiceType());
                  invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                  invoiceModel.setAuditCode(crossModel.getAuditCode());
                  invoiceModels.add(invoiceModel);
                }else {
                  for (InvoiceStatisticsVo statisticsVo : waitMatchInvoices) {
                    if (auditMatchCrossAcceptance.getCrossOrderNum().compareTo(statisticsVo.getInvoiceCount()) != 0) {
                      continue;
                    }
                    invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
                    invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
                    invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
                    AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                    invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
                    invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
                    invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
                    invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                    invoiceModel.setAuditCode(crossModel.getAuditCode());
                    invoiceModels.add(invoiceModel);
                    break;
                  }
                }
                crossModel.setInvoiceTotalCount(invoiceTotalCount);
                crossModel.setInvoiceTotalAmount(invoiceTotalAmount);
                crossModel.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
                crossModel.setAuditMatchInvoiceModels(invoiceModels);
                crossModel.setInvoicePrice(waitMatchInvoices.get(0).getInvoicePrice());
                crossModel.setInvoicePriceNoTax(waitMatchInvoices.get(0).getInvoicePriceNoTax());
                crossModel.setCrossOrderNumber(auditMatchCrossAcceptance.getCrossOrderNumber());
              }
            }else {
              String key = StringUtils.joinWith("-",auditMatchCrossAcceptance.getCrossOrderNumber(),auditMatchCrossAcceptance.getSapMaterialCode());
              List<InvoiceStatisticsVo> waitMatchInvoices = orderNumberMap.get(key);
              if (CollectionUtils.isEmpty(waitMatchInvoices)){
                modelList.add(crossModel);
              }else {
                if (waitMatchInvoices.size() == 1){
                  invoiceTotalCount = invoiceTotalCount.add(waitMatchInvoices.get(0).getInvoiceCount());
                  invoiceTotalAmount = invoiceTotalAmount.add(waitMatchInvoices.get(0).getInvoiceAmount());
                  invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(waitMatchInvoices.get(0).getInvoiceAmountTaxExclusive());
                  AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                  invoiceModel.setInvoiceCode(waitMatchInvoices.get(0).getInvoiceCode());
                  invoiceModel.setRelationId(waitMatchInvoices.get(0).getInvoiceCode());
                  invoiceModel.setInvoiceType(waitMatchInvoices.get(0).getInvoiceType());
                  invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                  invoiceModel.setAuditCode(crossModel.getAuditCode());
                  invoiceModels.add(invoiceModel);
                }else {
                  for (InvoiceStatisticsVo statisticsVo : waitMatchInvoices) {
                    if (auditMatchCrossAcceptance.getCrossOrderNum().compareTo(statisticsVo.getInvoiceCount()) != 0) {
                      continue;
                    }
                    invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
                    invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
                    invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
                    AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                    invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
                    invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
                    invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
                    invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                    invoiceModel.setAuditCode(crossModel.getAuditCode());
                    invoiceModels.add(invoiceModel);
                    break;
                  }
                }
                crossModel.setInvoiceTotalCount(invoiceTotalCount);
                crossModel.setInvoiceTotalAmount(invoiceTotalAmount);
                crossModel.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
                crossModel.setAuditMatchInvoiceModels(invoiceModels);
                crossModel.setInvoicePrice(waitMatchInvoices.get(0).getInvoicePrice());
                crossModel.setInvoicePriceNoTax(waitMatchInvoices.get(0).getInvoicePriceNoTax());
                crossModel.setCrossOrderNumber(auditMatchCrossAcceptance.getCrossOrderNumber());
              }
            }
          }else {
              List<AuditMatchInvoiceModel> invoiceModels = new ArrayList<>();
              for (AuditMatchCrossAcceptance matchCrossAcceptance : auditMatchCrossAcceptances) {
                if (StringUtils.equals(AuditMatchCrossTypeEnum.RELATIONKACODE.getDictCode(),auditMatchCrossAcceptance.getCrossOrderType())){
                  String key = StringUtils.joinWith("-",matchCrossAcceptance.getCrossOrderNumber(),matchCrossAcceptance.getSapMaterialCode());
                  List<InvoiceStatisticsVo> waitMatchInvoices = orderCodeMap.get(key);
                  if (CollectionUtils.isEmpty(waitMatchInvoices)){
                    continue;
                  }else {
                    if (waitMatchInvoices.size() == 1){
                      invoiceTotalCount = invoiceTotalCount.add(waitMatchInvoices.get(0).getInvoiceCount());
                      invoiceTotalAmount = invoiceTotalAmount.add(waitMatchInvoices.get(0).getInvoiceAmount());
                      invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(waitMatchInvoices.get(0).getInvoiceAmountTaxExclusive());
                      AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                      invoiceModel.setInvoiceCode(waitMatchInvoices.get(0).getInvoiceCode());
                      invoiceModel.setRelationId(waitMatchInvoices.get(0).getInvoiceCode());
                      invoiceModel.setInvoiceType(waitMatchInvoices.get(0).getInvoiceType());
                      invoiceModel.setCrossOrderNumber(matchCrossAcceptance.getCrossOrderNumber());
                      invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                      invoiceModel.setAuditCode(crossModel.getAuditCode());
                      invoiceModels.add(invoiceModel);
                    }else {
                      for (InvoiceStatisticsVo statisticsVo : waitMatchInvoices) {
                        if (matchCrossAcceptance.getCrossOrderNum().compareTo(statisticsVo.getInvoiceCount()) != 0) {
                          continue;
                        }
                        invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
                        invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
                        invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
                        AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                        invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
                        invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
                        invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
                        invoiceModel.setCrossOrderNumber(matchCrossAcceptance.getCrossOrderNumber());
                        invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                        invoiceModel.setAuditCode(crossModel.getAuditCode());
                        invoiceModels.add(invoiceModel);
                        break;
                      }
                    }
                  }
                }else {
                  String key = StringUtils.joinWith("-",auditMatchCrossAcceptance.getCrossOrderNumber(),auditMatchCrossAcceptance.getSapMaterialCode());
                  List<InvoiceStatisticsVo> waitMatchInvoices = orderNumberMap.get(key);
                  if (CollectionUtils.isEmpty(waitMatchInvoices)){
                    modelList.add(crossModel);
                  }else {
                    if (waitMatchInvoices.size() == 1){
                      invoiceTotalCount = invoiceTotalCount.add(waitMatchInvoices.get(0).getInvoiceCount());
                      invoiceTotalAmount = invoiceTotalAmount.add(waitMatchInvoices.get(0).getInvoiceAmount());
                      invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(waitMatchInvoices.get(0).getInvoiceAmountTaxExclusive());
                      price = waitMatchInvoices.get(0).getInvoicePrice();
                      priceNot = waitMatchInvoices.get(0).getInvoicePriceNoTax();
                      AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                      invoiceModel.setInvoiceCode(waitMatchInvoices.get(0).getInvoiceCode());
                      invoiceModel.setRelationId(waitMatchInvoices.get(0).getInvoiceCode());
                      invoiceModel.setInvoiceType(waitMatchInvoices.get(0).getInvoiceType());
                      invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                      invoiceModel.setCrossOrderNumber(matchCrossAcceptance.getCrossOrderNumber());
                      invoiceModel.setAuditCode(crossModel.getAuditCode());
                      invoiceModels.add(invoiceModel);
                    }else {
                      for (InvoiceStatisticsVo statisticsVo : waitMatchInvoices) {
                        if (auditMatchCrossAcceptance.getCrossOrderNum().compareTo(statisticsVo.getInvoiceCount()) != 0) {
                          continue;
                        }
                        price = waitMatchInvoices.get(0).getInvoicePrice();
                        priceNot = waitMatchInvoices.get(0).getInvoicePriceNoTax();
                        invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount());
                        invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount());
                        invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive());
                        AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
                        invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
                        invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
                        invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
                        invoiceModel.setCrossOrderNumber(matchCrossAcceptance.getCrossOrderNumber());
                        invoiceModel.setTemplateCode(auditTemplate.getTemplateCode());
                        invoiceModel.setAuditCode(crossModel.getAuditCode());
                        invoiceModels.add(invoiceModel);
                        break;
                      }
                    }
                  }
                }
              }
            crossModel.setInvoiceTotalCount(invoiceTotalCount);
            crossModel.setInvoiceTotalAmount(invoiceTotalAmount);
            crossModel.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
            crossModel.setAuditMatchInvoiceModels(invoiceModels);
            StringBuffer buffer = new StringBuffer();
            for (AuditMatchInvoiceModel invoiceModel : invoiceModels) {
              buffer.append(invoiceModel.getCrossOrderNumber() + "/");
            }
            crossModel.setCrossOrderNumber(String.valueOf(buffer));
            crossModel.setInvoicePrice(price);
            crossModel.setInvoicePriceNoTax(priceNot);
          }
          models.add(crossModel);
        }
      }
    }
    //执行计算
      if (params.contains(MatchParamsEnum.ORDER.getDictCode())){
        //未匹配原因
        List<AuditMatchNotReasonDto> notMatchReason = this.findNotMatchReason(invoiceStatisticsVos, modelList,auditTemplateSupermarketMap);
        this.calculate(models,notMatchReason);
      }else {
        this.calculate(models,null);
      }
  }

  @Override
  public void notAutoExecute(String templateCode, MatchConsequenceCalculatedModel calculatedModel, List<AuditMatchInvoice> auditMatchInvoices) {
    Validate.isTrue(!Objects.isNull(calculatedModel), "执行匹配计算逻辑时，需要匹配的数据不存在！");

    //1、查询稽核单号已匹配的单据
    List<String> auditCodes = Lists.newArrayList(calculatedModel.getAuditCode());
    List<MatchInvoiceManuEntity> matchInvoiceList = this.matchInvoiceManuRepository.findByAuditCodes(auditCodes);
    List<InvoiceStatisticsVo> invoiceStatisticsVos = new ArrayList<>();
    List<String> relationId = new ArrayList<>();
    //2、查询验收单数据,并且按照匹配条件转成map
    if (!CollectionUtils.isEmpty(auditMatchInvoices)){
      relationId = auditMatchInvoices.stream().map(AuditMatchInvoice::getRelationId).collect(Collectors.toList());
      invoiceStatisticsVos = this.invoiceStatisticsVoService.findByRelationId(AuditMatchConstant.ACCEPTANCE,relationId);
      Validate.isTrue(!CollectionUtils.isEmpty(invoiceStatisticsVos), "执行匹配计算逻辑时，未匹配到验收单！");
    }
    //3、赋值
    List<AuditMatchInvoiceModel> invoiceModels = new ArrayList<>();
      BigDecimal invoiceTotalCount = BigDecimal.ZERO;
      BigDecimal invoiceTotalAmount = BigDecimal.ZERO;
      BigDecimal invoiceTotalAmountTaxExclusive = BigDecimal.ZERO;
      if (!CollectionUtils.isEmpty(auditMatchInvoices)){
        for (InvoiceStatisticsVo statisticsVo : invoiceStatisticsVos) {
          invoiceTotalCount = invoiceTotalCount.add(statisticsVo.getInvoiceCount() == null ? BigDecimal.ZERO:statisticsVo.getInvoiceCount());
          invoiceTotalAmount = invoiceTotalAmount.add(statisticsVo.getInvoiceAmount() == null ? BigDecimal.ZERO:statisticsVo.getInvoiceAmount());
          invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(statisticsVo.getInvoiceAmountTaxExclusive() == null ? BigDecimal.ZERO:statisticsVo.getInvoiceAmountTaxExclusive());
          //绑定单据
          AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
          invoiceModel.setInvoiceCode(statisticsVo.getInvoiceCode());
          invoiceModel.setInvoiceType(statisticsVo.getInvoiceType());
          invoiceModel.setTemplateCode(templateCode);
          invoiceModel.setAuditCode(calculatedModel.getAuditCode());
          invoiceModel.setRelationId(statisticsVo.getInvoiceCode());
          invoiceModels.add(invoiceModel);
        }
      }
    for (MatchInvoiceManuEntity matchInvoiceManuEntity : matchInvoiceList) {
      invoiceTotalCount = invoiceTotalCount.add(matchInvoiceManuEntity.getCurCompanyUnitOrderQuantity() == null ? BigDecimal.ZERO:matchInvoiceManuEntity.getCurCompanyUnitOrderQuantity());
      invoiceTotalAmount = invoiceTotalAmount.add(matchInvoiceManuEntity.getAcceptanceAmount() == null ? BigDecimal.ZERO:matchInvoiceManuEntity.getAcceptanceAmount());
      invoiceTotalAmountTaxExclusive = invoiceTotalAmountTaxExclusive.add(matchInvoiceManuEntity.getAcceptanceAmountNot() == null ? BigDecimal.ZERO:matchInvoiceManuEntity.getAcceptanceAmountNot());
      //绑定单据
      AuditMatchInvoiceModel invoiceModel = new AuditMatchInvoiceModel();
      invoiceModel.setInvoiceCode(matchInvoiceManuEntity.getId());
      invoiceModel.setTemplateCode(templateCode);
      invoiceModel.setAuditCode(matchInvoiceManuEntity.getAuditCode());
      invoiceModel.setRelationId(matchInvoiceManuEntity.getId());
      invoiceModels.add(invoiceModel);
    }
      calculatedModel.setInvoiceTotalCount(invoiceTotalCount);
      calculatedModel.setInvoiceTotalAmount(invoiceTotalAmount);
      calculatedModel.setInvoiceTotalAmountTaxExclusive(invoiceTotalAmountTaxExclusive);
      calculatedModel.setAuditMatchInvoiceModels(invoiceModels);
      calculatedModel.setTemplateCode(templateCode);
      calculatedModel.setInvoicePrice(invoiceStatisticsVos.get(0).getInvoicePrice());
      calculatedModel.setInvoicePriceNoTax(invoiceStatisticsVos.get(0).getInvoicePriceNoTax());
    //执行计算
    this.manuCalculate(calculatedModel,relationId);
  }

  private List<AuditMatchNotReasonDto> findNotMatchReason(List<InvoiceStatisticsVo> invoiceStatisticsVos, List<MatchConsequenceCalculatedModel> models,Map<String, AuditTemplateSupermarket> auditTemplateSupermarketMap) {
    List<AuditMatchNotReasonDto> notMatchModel = new ArrayList<>();
    //采购单号
    Map<String, List<InvoiceStatisticsVo>> caseOneMap = invoiceStatisticsVos.stream().collect(Collectors.groupingBy(InvoiceStatisticsVo::getOrderCode));

    for (MatchConsequenceCalculatedModel model : models) {
      AuditTemplateSupermarket auditTemplateSupermarket = auditTemplateSupermarketMap.get(model.getDirectCode());
      Integer delayDays = auditTemplateSupermarket.getDelayDays();
      String caseOneKey = model.getRelationOrderCode();
      String caseTwoKey = StringUtils.joinWith("-",model.getRelationOrderCode(),model.getSapMaterialCode());
      String caseThreeKey = StringUtils.joinWith("-", model.getSapMaterialCode(), model.getSoldToPartyCode(), model.getRelationOrderCode());
      StringBuffer reason = new StringBuffer();
      if (StringUtils.isEmpty(caseOneKey)){
        reason.append("无采购单号!");
      }else {
        //第一条件
        List<InvoiceStatisticsVo> caseOneVo = caseOneMap.get(caseOneKey);
        if (CollectionUtils.isEmpty(caseOneVo)) {
          reason.append("KA单据不存在||");
        } else {
          List<InvoiceStatisticsVo> caseSuccessOneVo = caseOneVo.stream().filter(vo -> StringUtils.equals(AcceptanceStatus.S200.getDictCode(), vo.getOrderStatus())).collect(Collectors.toList());
          if (CollectionUtils.isEmpty(caseSuccessOneVo)) {
            reason.append("单据转换失败||" + caseOneVo.get(0).getOrderStatusMsg());
          } else {
            //采购单号、产品
            Map<String, List<InvoiceStatisticsVo>> caseTwoMap = caseSuccessOneVo.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("-", vo.getOrderCode(), vo.getGoodsCode())));
            List<InvoiceStatisticsVo> caseTwoVo = caseTwoMap.get(caseTwoKey);
            reason.append("转换成功||");
            if (CollectionUtils.isEmpty(caseTwoVo)) {
              reason.append("该采购单下,该产品数据不存在||");
            } else {
              //产品、售达方、采购单号
              Map<String, List<InvoiceStatisticsVo>> caseThreeMap = caseSuccessOneVo.stream().collect(Collectors.groupingBy(vo -> StringUtils.joinWith("-", vo.getGoodsCode(), vo.getSoldToPartyCode(), vo.getOrderCode())));
              List<InvoiceStatisticsVo> caseThreeVo = caseThreeMap.get(caseThreeKey);
              if (CollectionUtils.isEmpty(caseThreeVo)) {
                reason.append("售达方未能匹配||");
              } else {
                List<InvoiceStatisticsVo> dateInvoices = new ArrayList<>();
                String beginDate = null;
                String endDate = null;
                String posDate = DateUtil.format(model.getSapPostingDate(), "yyyy-MM-dd");
                Calendar calendar = Calendar.getInstance();
                switch (PriceTypeEnum.getByDictCode(auditTemplateSupermarket.getDelayType())) {
                  case Z:
                    calendar.setTime(model.getSapPostingDate());
                    calendar.add(Calendar.DATE, delayDays);
                    endDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                    for (InvoiceStatisticsVo invoiceStatisticsVo : caseThreeVo) {
                      String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                      if (invoiceDate != null && invoiceDate.compareTo(posDate) >= 0 && invoiceDate.compareTo(endDate) <= 0) {
                        dateInvoices.add(invoiceStatisticsVo);
                      }
                    }
                    break;
                  case F:
                    calendar.setTime(model.getSapPostingDate());
                    calendar.add(Calendar.DATE, -1 * delayDays);
                    beginDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                    for (InvoiceStatisticsVo invoiceStatisticsVo : caseThreeVo) {
                      String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                      if (invoiceDate != null && invoiceDate.compareTo(beginDate) >= 0 && invoiceDate.compareTo(posDate) <= 0) {
                        dateInvoices.add(invoiceStatisticsVo);
                      }
                    }
                    break;
                  case UD:
                    calendar.setTime(model.getSapPostingDate());
                    calendar.add(Calendar.DATE, delayDays);
                    endDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                    calendar.setTime(model.getSapPostingDate());
                    calendar.add(Calendar.DATE, -1 * delayDays);
                    beginDate = DateUtil.format(calendar.getTime(), "yyyy-MM-dd");
                    for (InvoiceStatisticsVo invoiceStatisticsVo : caseThreeVo) {
                      String invoiceDate = DateUtil.format(invoiceStatisticsVo.getInvoiceDate(), "yyyy-MM-dd");
                      if (invoiceDate != null && invoiceDate.compareTo(beginDate) >= 0 && invoiceDate.compareTo(endDate) <= 0) {
                        dateInvoices.add(invoiceStatisticsVo);
                      }
                    }
                    break;
                  default:
                    break;
                }
                if (CollectionUtils.isEmpty(dateInvoices)){
                  reason.append("超出时间容差范围||");
                }else {
                  if (dateInvoices.size() > 1) {
                    boolean numFlag = false;
                    for (InvoiceStatisticsVo invoiceStatisticsVo : caseThreeVo) {
                      if (model.getSapEaCount().compareTo(invoiceStatisticsVo.getInvoiceCount()) == 0) {
                        numFlag = true;
                      }
                    }
                    if (!numFlag) {
                      reason.append("有多个验收关系,数量未能匹配||");
                    }
                  } else if (dateInvoices.size() == 1) {
                      reason.append("未知原因||");
                  }
                }
              }
            }
          }
        }
      }
      AuditMatchNotReasonDto dto = new AuditMatchNotReasonDto();
      dto.setAuditCode(model.getAuditCode());
      dto.setNotMatchReason(String.valueOf(reason));
      notMatchModel.add(dto);
    }
    return notMatchModel;
  }

  private void updateNotReason(List<MatchConsequenceCalculatedModel> models,List<InvoiceStatisticsVo> invoiceStatisticsVos){
    List<AuditMatchNotReasonDto> notMatchModel = new ArrayList<>();
    Map<String, List<InvoiceStatisticsVo>> caseOneMap = invoiceStatisticsVos.stream().collect(Collectors.groupingBy(InvoiceStatisticsVo::getOrderCode));
    for (MatchConsequenceCalculatedModel model : models) {
      if (caseOneMap.containsKey(model.getRelationOrderCode())){
        List<InvoiceStatisticsVo> invoiceStatisticsList = caseOneMap.get(model.getRelationOrderCode());
        AuditMatchNotReasonDto dto = new AuditMatchNotReasonDto();
        dto.setNotMatchReason("单据转换失败||" + invoiceStatisticsList.get(0).getOrderStatusMsg());
        dto.setAuditCode(model.getAuditCode());
        notMatchModel.add(dto);
      }else {
        AuditMatchNotReasonDto dto = new AuditMatchNotReasonDto();
        dto.setNotMatchReason("KA单据不存在||");
        dto.setAuditCode(model.getAuditCode());
        notMatchModel.add(dto);
      }
    }
    this.auditMatchService.updateNotMatchReason(notMatchModel);
  }
}
