package com.biz.crm.kms.business.direct.account.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.account.local.entity.DirectAccountEntity;
import com.biz.crm.kms.business.direct.account.local.mapper.DirectAccountMapper;

import java.util.List;

import org.springframework.stereotype.Component;

/**
 * 直营账号管理表(repository)
 *
 * @author xi.peng
 * @since 2022-09-23 10:47:58
 */
@Component
public class DirectAccountRepository extends ServiceImpl<DirectAccountMapper, DirectAccountEntity> {

  /**
   * 根据ID获取详情
   *
   * @param id
   * @return
   */
  public DirectAccountEntity findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<DirectAccountEntity> findByIds(List<String> ids) {
    return this.lambdaQuery()
        .eq(DirectAccountEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(DirectAccountEntity::getId, ids)
        .list();
  }

  /**
   * 根据系统编码获取直营账号管理集合
   *
   * @param directCode 系统编码
   * @param tenantCode 租户编码
   * @return
   */
  public DirectAccountEntity findByAccount(String directCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(DirectAccountEntity::getTenantCode, tenantCode)
        .eq(DirectAccountEntity::getEnableStatus,EnableStatusEnum.ENABLE.getCode())
        .eq(DirectAccountEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
        .eq(DirectAccountEntity::getAccount, directCode)
        .one();
  }

  /**
   * 根据系统编码获取直营账号管理集合
   *
   * @param directCode 系统编码
   * @param tenantCode 租户编码
   * @return
   */
  public List<DirectAccountEntity> findByDirectCode(String directCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(DirectAccountEntity::getTenantCode, tenantCode)
        .eq(DirectAccountEntity::getEnableStatus,EnableStatusEnum.ENABLE.getCode())
        .eq(DirectAccountEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
        .eq(DirectAccountEntity::getDirectCode, directCode)
        .list();
  }

  /**
   * 根据系统编码获取直营账号管理集合
   *
   * @param directCodes 系统编码集合
   * @param tenantCode 租户编码
   * @return
   */
  public List<DirectAccountEntity> findByDirectCodes(List<String> directCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(DirectAccountEntity::getTenantCode, tenantCode)
        .eq(DirectAccountEntity::getEnableStatus,EnableStatusEnum.ENABLE.getCode())
        .eq(DirectAccountEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
        .in(DirectAccountEntity::getDirectCode, directCodes)
        .list();
  }

  /**
   * 根据账号集合获取直营账号管理集合
   *
   * @param accountList 账号集合
   * @param tenantCode 租户编码
   * @return
   */
  public List<DirectAccountEntity> findByAccountList(List<String> accountList, String tenantCode) {
    return this.lambdaQuery()
        .eq(DirectAccountEntity::getTenantCode, tenantCode)
        .eq(DirectAccountEntity::getEnableStatus,EnableStatusEnum.ENABLE.getCode())
        .eq(DirectAccountEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
        .in(DirectAccountEntity::getAccount, accountList)
        .list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids
   * @param enable
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .in(DirectAccountEntity::getId, ids)
        .set(DirectAccountEntity::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate().in(DirectAccountEntity::getId, ids)
        .set(DirectAccountEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }
}
