package com.biz.crm.kms.business.direct.account.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.account.local.entity.DirectAccountEntity;
import com.biz.crm.kms.business.direct.account.local.repository.DirectAccountRepository;
import com.biz.crm.kms.business.direct.account.sdk.dto.DirectAccountDto;
import com.biz.crm.kms.business.direct.account.sdk.service.DirectAccountVoService;
import com.biz.crm.kms.business.direct.account.sdk.vo.DirectAccountVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

/**
 * 直营账号管理表(directAccount)表服务实现类
 *
 * @author xi.peng
 * @since 2022-09-23 10:47:58
 */
@Slf4j
@Service
public class DirectAccountVoServiceImpl implements DirectAccountVoService {

  @Autowired(required = false)
  private DirectAccountRepository directAccountRepository;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<DirectAccountVo> findByDirectCode(String directCode) {
    if (StringUtils.isBlank(directCode)) {
      return null;
    }
    List<DirectAccountEntity> entities = this.directAccountRepository.findByDirectCode(directCode, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<DirectAccountVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DirectAccountEntity.class, DirectAccountVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<DirectAccountVo> findByDirectCodes(List<String> directCodes) {
    if (CollectionUtils.isEmpty(directCodes)) {
      return null;
    }
    List<DirectAccountEntity> entities = this.directAccountRepository.findByDirectCodes(directCodes, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<DirectAccountVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DirectAccountEntity.class, DirectAccountVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<DirectAccountVo> findByAccountList(List<String> accountList) {
    if (CollectionUtils.isEmpty(accountList)) {
      return null;
    }
    List<DirectAccountEntity> entities = this.directAccountRepository.findByAccountList(accountList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<DirectAccountVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DirectAccountEntity.class, DirectAccountVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DirectAccountVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    DirectAccountEntity entity = this.directAccountRepository.findById(id);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DirectAccountVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void create(DirectAccountDto dto) {
    this.createValidation(dto);
    DirectAccountEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, DirectAccountEntity.class, HashSet.class, ArrayList.class);
    this.directAccountRepository.save(entity);
  }

  @Override
  @Transactional
  public void update(DirectAccountDto dto) {
    this.updateValidation(dto);
    DirectAccountEntity oldEntity = this.directAccountRepository.findById(dto.getId());
    Validate.notNull(oldEntity, "不存在或已删除！");
    DirectAccountEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, DirectAccountEntity.class, HashSet.class, ArrayList.class);
    this.directAccountRepository.updateById(entity);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DirectAccountEntity> entities = this.directAccountRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.directAccountRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DirectAccountEntity> entities = this.directAccountRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.directAccountRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DirectAccountEntity> entities = this.directAccountRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.directAccountRepository.updateDelFlagByIds(ids);
  }

  private void createValidation(DirectAccountDto dto) {
    this.validation(dto);
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
  }

  private void updateValidation(DirectAccountDto dto) {
    this.validation(dto);
    Validate.notBlank(dto.getId(), "ID不能为空！");
//    Validate.notBlank(dto.getConfirmPassword(), "确认密码不能为空！");
  }

  private void validation(DirectAccountDto dto) {
    Validate.notNull(dto, "对象不能为空！");
    Validate.notBlank(dto.getDirectCode(), "系统编码不能为空！");
    Validate.notBlank(dto.getAccount(), "账号不能为空！");
//    Validate.notBlank(dto.getPassword(), "密码不能为空！");
//    Validate.notBlank(dto.getConfirmPassword(), "确认密码不能为空！");
//    Validate.isTrue(dto.getPassword().equals(dto.getConfirmPassword()), "两次输入的密码不一致！");
    //长度校验
    Validate.isTrue(dto.getAccount().length()<=32, "账号最多32位字符串");
//    Validate.isTrue(dto.getPassword().length()<=32, "密码最多16位字符串");
    if (StringUtils.isNotBlank(dto.getRemark())) {
      Validate.isTrue(dto.getRemark().length()<=400, "备注最多400位字符串");
    }
  }
}
