package com.biz.crm.kms.business.direct.account.local.service.notifier;

import com.biz.crm.kms.business.direct.sdk.listener.DirectOperationListener;
import com.biz.crm.kms.business.direct.account.sdk.service.DirectAccountVoService;
import com.biz.crm.kms.business.direct.account.sdk.vo.DirectAccountVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 系统操作监听器实现
 *
 * @author pengxi
 */
@Slf4j
@Component
public class DirectAccountListenerImpl implements DirectOperationListener {

  @Autowired(required = false) private DirectAccountVoService directAccountVoService;

  /**
   * 启用时，通知上层业务进行业务处理
   *
   * @param directCodes 系统编码集合
   */
  @Override
  @Async("otherThread")
  public void onEnable(List<String> directCodes) {
    List<DirectAccountVo> directAccountVos = this.directAccountVoService.findByDirectCodes(directCodes);
    if (CollectionUtils.isEmpty(directAccountVos)) {
      return;
    }
    List<String> ids = directAccountVos.stream().map(DirectAccountVo::getId).distinct().collect(Collectors.toList());
    this.directAccountVoService.enableBatch(ids);
    log.info("系统启用时被通知启用的直营账号ID集合：{}", ids);
  }

  /**
   * 禁用时，通知上层业务进行业务处理
   *
   * @param directCodes 系统编码集合
   */
  @Override
  @Async("otherThread")
  public void onDisable(List<String> directCodes) {
    List<DirectAccountVo> directAccountVos = this.directAccountVoService.findByDirectCodes(directCodes);
    if (CollectionUtils.isEmpty(directAccountVos)) {
      return;
    }
    List<String> ids = directAccountVos.stream().map(DirectAccountVo::getId).distinct().collect(Collectors.toList());
    this.directAccountVoService.disableBatch(ids);
    log.info("系统禁用时被通知禁用的直营账号ID集合：{}", ids);
  }

  /**
   * 删除时，通知上层业务进行业务处理
   *
   * @param directCodes 系统编码集合
   */
  @Override
  @Async("otherThread")
  public void onDelete(List<String> directCodes) {
    List<DirectAccountVo> directAccountVos = this.directAccountVoService.findByDirectCodes(directCodes);
    if (CollectionUtils.isEmpty(directAccountVos)) {
      return;
    }
    List<String> ids = directAccountVos.stream().map(DirectAccountVo::getId).distinct().collect(Collectors.toList());
    this.directAccountVoService.updateDelFlagByIds(ids);
    log.info("系统删除时被通知删除的直营账号ID集合：{}", ids);
  }
  
}
