package com.biz.crm.kms.business.direct.local.repository;


import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.local.entity.Direct;
import com.biz.crm.kms.business.direct.local.mapper.DirectMapper;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.List;

import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 系统表(Direct)表数据库访问层
 *
 * @author songjingen
 * @since 2022-09-26 16:31:55
 */
@Component
public class DirectRepository extends ServiceImpl<DirectMapper, Direct> {

  /**
   * 根据体系编码查询数据
   *
   * @param directCodes 系统编码集合
   * @return
   */
  public List<Direct> findByDirectCodes(List<String> directCodes) {
    return this.lambdaQuery().eq(Direct::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Direct::getTenantCode, TenantUtils.getTenantCode())
        .in(Direct::getDirectCode, directCodes)
        .list();
  }

  /**
   * 根据id集合查询数据
   *
   * @param idList id集合
   * @return
   */
  public List<Direct> findByIds(List<String> idList) {
    return this.lambdaQuery().eq(Direct::getTenantCode, TenantUtils.getTenantCode())
        .eq(Direct::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(Direct::getId, idList)
        .list();
  }

  /**
   * 根据id集合和启禁用状态查询数据
   *
   * @param statusEnum 启禁用状态
   * @param idList     id集合
   * @return
   */
  public List<Direct> findByEnableStatusAndIds(EnableStatusEnum statusEnum, List<String> idList) {
    return this.lambdaQuery().eq(Direct::getTenantCode, TenantUtils.getTenantCode())
        .eq(Direct::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Direct::getEnableStatus, statusEnum.getCode())
        .in(Direct::getId, idList)
        .list();
  }

  /**
   * 根据商超编码集合查询数据
   *
   * @param supermarketCodes
   * @return
   */
  public List<Direct> findBySupermarketCodes(List<String> supermarketCodes) {
    return this.lambdaQuery().eq(Direct::getTenantCode, TenantUtils.getTenantCode())
        .eq(Direct::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(Direct::getSupermarketCode, supermarketCodes)
        .list();
  }

  /**
   * 根据商超编码和客户编码查询数据
   *
   * @param supermarketCode 商超编码
   * @param customerOrgCode 客户编码
   * @return 结果稽核
   */
  public List<Direct> findBySupermarketCodeAndCustomerOrgCode(String supermarketCode, String customerOrgCode) {
    return this.lambdaQuery().eq(Direct::getTenantCode, TenantUtils.getTenantCode())
        .eq(Direct::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Direct::getSupermarketCode, supermarketCode)
        .eq(Direct::getSalesOrgCode, customerOrgCode)
        .list();
  }

  /**
   * 根据销售组织编码集合和商超编码集合查询数据
   *
   * @param saleOrgCodes
   * @param superMarketCodes
   * @return
   */
  public List<DirectVo> findBySaleOrgCodesAndSuperMarketCodes(List<String> saleOrgCodes, List<String> superMarketCodes){
    if (CollectionUtils.isEmpty(saleOrgCodes) || CollectionUtils.isEmpty(superMarketCodes)){
      return Lists.newArrayList();
    }else {
      return this.baseMapper.findBySaleOrgCodesAndSuperMarketCodes(saleOrgCodes,superMarketCodes,TenantUtils.getTenantCode());
    }
  }

  public Direct findByDirectCode(String directCode) {
    return this.lambdaQuery().eq(Direct::getTenantCode, TenantUtils.getTenantCode())
            .eq(Direct::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(Direct::getDirectCode, directCode)
            .one();
  }

  /**
   *查询系统编码
   * @param businessUnitCode
   * @return
   */
  public List<DirectVo> findDirectCodesByBusinessUnitCode (String businessUnitCode){
    return this.baseMapper.findDirectCodesByBusinessUnitCode(businessUnitCode,TenantUtils.getTenantCode());
  }
}

