package com.biz.crm.kms.business.direct.local.service.internal;


import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.local.entity.Direct;
import com.biz.crm.kms.business.direct.local.repository.DirectRepository;
import com.biz.crm.kms.business.direct.local.service.DirectService;
import com.biz.crm.kms.business.direct.sdk.dto.DirectDto;
import com.biz.crm.kms.business.direct.sdk.dto.DirectLogEventDto;
import com.biz.crm.kms.business.direct.sdk.listener.DirectLogEventListener;
import com.biz.crm.kms.business.direct.sdk.listener.DirectOperationListener;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 系统表(Direct)表服务实现类
 *
 * @author songjingen
 * @since 2022-09-26 16:31:55
 */
@Service("directService")
public class DirectServiceImpl implements DirectService {

  @Autowired
  private DirectRepository directRepository;
  @Autowired(required = false)
  private List<DirectOperationListener> directOperationListeners;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;


  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public Direct findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Direct direct = this.directRepository.getById(id);
    return direct;
  }

  /**
   * 新增数据
   *
   * @param direct 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public Direct create(Direct direct) {
    this.createValidate(direct);
    this.directRepository.saveOrUpdate(direct);
    return direct;
  }

  /**
   * 修改新据
   *
   * @param direct 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public Direct update(Direct direct) {
    Validate.notNull(direct, "修改时，对象信息不能为空！");
    Validate.notBlank(direct.getId(), "修改数据时，不能为空！");
    Direct current = this.findById(direct.getId());
    DirectDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(current, DirectDto.class, HashSet.class, ArrayList.class);
    Validate.notNull(current, "未查询到要修改的数据！");
    String oldDirectCode = current.getDirectCode();
    BeanUtils.copyProperties(direct, current, "id", "tenantCode", "createAccount", "createName", "createTime", "delFlag", "enableStatus");
    this.updateValidate(current);
    this.directRepository.saveOrUpdate(current);
    //添加日志
    DirectDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(current, DirectDto.class, HashSet.class, ArrayList.class);
    DirectLogEventDto logEventDto = new DirectLogEventDto();
    logEventDto.setOriginal(oldDto);
    logEventDto.setNewest(newDto);
    SerializableBiConsumer<DirectLogEventListener, DirectLogEventDto> onUpdate =
            DirectLogEventListener::onUpdate;
    this.nebulaNetEventClient.publish(logEventDto, DirectLogEventListener.class, onUpdate);
    //调用监听器通知上层业务处理
    if (!CollectionUtils.isEmpty(this.directOperationListeners)) {
      for (DirectOperationListener directOperationListener : this.directOperationListeners) {
        directOperationListener.onUpdate(oldDirectCode, direct.getDirectCode());
      }
    }
    return current;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<Direct> directs = this.directRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(directs), "删除数据时，未查询到需要删除的数据！");
    directs.stream().forEach(direct -> direct.setDelFlag(DelFlagStatusEnum.DELETE.getCode()));
    this.directRepository.saveOrUpdateBatch(directs);
    //调用监听器通知上层业务处理
    if (!CollectionUtils.isEmpty(this.directOperationListeners)) {
      List<String> directCodes = directs.stream().map(Direct::getDirectCode).collect(Collectors.toList());
      for (DirectOperationListener directOperationListener : this.directOperationListeners) {
        directOperationListener.onDelete(directCodes);
      }
    }
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    List<Direct> directs = this.directRepository.findByEnableStatusAndIds(EnableStatusEnum.DISABLE, idList);
    Validate.isTrue(!CollectionUtils.isEmpty(directs), "启用数据时，未查询到需要启用的数据！");
    directs.stream().forEach(direct -> direct.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    this.directRepository.saveOrUpdateBatch(directs);
    //调用监听器通知上层业务处理
    if (!CollectionUtils.isEmpty(this.directOperationListeners)) {
      List<String> directCodes = directs.stream().map(Direct::getDirectCode).collect(Collectors.toList());
      for (DirectOperationListener directOperationListener : this.directOperationListeners) {
        directOperationListener.onEnable(directCodes);
      }
    }
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    List<Direct> directs = this.directRepository.findByEnableStatusAndIds(EnableStatusEnum.ENABLE, idList);
    Validate.isTrue(!CollectionUtils.isEmpty(directs), "禁用数据时，未查询到需要禁用的数据！");
    directs.stream().forEach(direct -> direct.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
    this.directRepository.saveOrUpdateBatch(directs);
    //调用监听器通知上层业务处理
    if (!CollectionUtils.isEmpty(this.directOperationListeners)) {
      List<String> directCodes = directs.stream().map(Direct::getDirectCode).collect(Collectors.toList());
      for (DirectOperationListener directOperationListener : this.directOperationListeners) {
        directOperationListener.onDisable(directCodes);
      }
    }
  }

  /**
   * 创建验证
   *
   * @param direct
   */
  private void createValidate(Direct direct) {
    Validate.notNull(direct, "新增时，对象信息不能为空！");
    direct.setId(null);
    direct.setTenantCode(TenantUtils.getTenantCode());
    direct.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    direct.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    Validate.notBlank(direct.getBusinessFormatCode(),"新增数据时，业态不能为空");
    Validate.notBlank(direct.getBusinessUnitCode(),"新增数据时，业务单元不能为空");
    Validate.notBlank(direct.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(direct.getSalesOrgCode(), "新增数据时，销售组织编码不能为空！");
    Validate.notBlank(direct.getSalesOrgName(), "新增数据时，销售组织名称不能为空！");
    Validate.notBlank(direct.getDirectCode(), "新增数据时，系统编码不能为空！");
    Validate.notBlank(direct.getSupermarketCode(), "新增数据时，商超编码不能为空！");
    Validate.isTrue(direct.getDirectCode().length() > 0 && direct.getDirectCode().length() <= 64, "新增数据时，系统编码不能超过64个字符");
    //判断体系编码重复
    List<Direct> directs = this.directRepository.findByDirectCodes(Lists.newArrayList(direct.getDirectCode()));
    Validate.isTrue(CollectionUtils.isEmpty(directs), "新增数据时，系统编码已存在！");
    //验证商超编码和客户组织编码的唯一性
    List<Direct> directList = this.directRepository.findBySupermarketCodeAndCustomerOrgCode(direct.getSupermarketCode(), direct.getSalesOrgCode());
    Validate.isTrue(CollectionUtils.isEmpty(directList), "新增数据时，当前商超已绑定当前销售组织！");
  }

  /**
   * 修改验证
   *
   * @param direct
   */
  private void updateValidate(Direct direct) {
    Validate.notBlank(direct.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(direct.getSalesOrgCode(), "新增数据时，销售组织编码不能为空！");
    Validate.notBlank(direct.getSalesOrgName(), "新增数据时，销售组织名称不能为空！");
    Validate.notBlank(direct.getDirectCode(), "修改数据时，系统编码不能为空！");
    Validate.isTrue(direct.getDirectCode().length() > 0 && direct.getDirectCode().length() <= 64, "新增数据时，系统编码不能超过64个字符");
    //判断体系编码重复
    List<Direct> directs = this.directRepository.findByDirectCodes(Lists.newArrayList(direct.getDirectCode()));
    if (!CollectionUtils.isEmpty(directs)) {
      Set<String> ids = directs.stream().map(Direct::getId).collect(Collectors.toSet());
      Validate.isTrue(ids.contains(direct.getId()), "修改数据时，系统编码已存在！");
    }
    //验证商超编码和客户组织编码的唯一性
    List<Direct> directList = this.directRepository.findBySupermarketCodeAndCustomerOrgCode(direct.getSupermarketCode(), direct.getSalesOrgCode());
    if (!CollectionUtils.isEmpty(directList)) {
      Set<String> ids = directList.stream().map(Direct::getId).collect(Collectors.toSet());
      Validate.isTrue(ids.contains(direct.getId()), "修改数据时，当前商超已绑定当前销售组织！");
    }
  }
}

