package com.biz.crm.kms.business.direct.product.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.product.local.entity.DirectProductEntity;
import com.biz.crm.kms.business.direct.product.local.mapper.DirectProductMapper;

import java.util.Collections;
import java.util.List;
import java.util.Objects;

import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

/**
 * 直营上架产品管理表(repository)
 *
 * @author xi.peng
 * @since 2022-09-23 10:47:58
 */
@Component
public class DirectProductRepository extends ServiceImpl<DirectProductMapper, DirectProductEntity> {

  /**
   * 条件查询列表
   *
   * @param dto
   * @return
   */
  public List<DirectProductVo> findByListConditions(DirectProductDto dto) {
    return this.baseMapper.findByConditions(dto);
  }

  /**
   * 根据ID获取详情
   *
   * @param id
   * @return
   */
  public DirectProductEntity findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<DirectProductEntity> findByIds(List<String> ids) {
    return this.lambdaQuery()
        .eq(DirectProductEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(DirectProductEntity::getId, ids)
        .list();
  }

  /**
   * 根据系统编码获取直营上架产品管理集合
   *
   * @param directCode 系统编码
   * @param tenantCode 租户编码
   * @return
   */
  public List<DirectProductEntity> findByDirectCode(String directCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(DirectProductEntity::getTenantCode, tenantCode)
        .eq(DirectProductEntity::getDirectCode, directCode)
        .list();
  }

  /**
   * 根据系统编码获取直营上架产品管理集合
   *
   * @param directCodes 系统编码集合
   * @param tenantCode 租户编码
   * @return
   */
  public List<DirectProductEntity> findByDirectCodes(List<String> directCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(DirectProductEntity::getTenantCode, tenantCode)
        .in(DirectProductEntity::getDirectCode, directCodes)
        .list();
  }

  /**
   * 根据系统编码获取直营上架产品管理集合
   *
   * @param dto
   * @return
   */
  public List<DirectProductEntity> findByDirectProductDto(DirectProductDto dto) {
    if (Objects.isNull(dto)){
      return Collections.emptyList();
    }
    if (StringUtils.isBlank(dto.getTenantCode()) &&
            StringUtils.isBlank(dto.getDirectCode()) &&
            StringUtils.isBlank(dto.getDeliveryPartyCode()) &&
            CollectionUtils.isEmpty(dto.getDirectCodes()) &&
            CollectionUtils.isEmpty(dto.getDeliveryPartyCodes()) &&
            StringUtils.isBlank(dto.getSellPartyCode()) &&
            StringUtils.isBlank(dto.getProductCode()) &&
            StringUtils.isBlank(dto.getKaProductCode()) &&
            StringUtils.isBlank(dto.getIsDefault()) &&
            CollectionUtils.isEmpty(dto.getProductCodes()) &&
            CollectionUtils.isEmpty(dto.getKaProductCodes()) &&
            CollectionUtils.isEmpty(dto.getSellPartyCodes())) {
      return Collections.emptyList();
    }
    return this.lambdaQuery()
        .eq(DirectProductEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(StringUtils.isNotBlank(dto.getEnableStatus()), DirectProductEntity::getEnableStatus, dto.getEnableStatus())
        .eq(StringUtils.isNotBlank(dto.getOnShelfStatus()), DirectProductEntity::getOnShelfStatus, dto.getOnShelfStatus())
        .eq(StringUtils.isNotBlank(dto.getTenantCode()), DirectProductEntity::getTenantCode, dto.getTenantCode())
        .eq(StringUtils.isNotBlank(dto.getDirectCode()), DirectProductEntity::getDirectCode, dto.getDirectCode())
        .eq(StringUtils.isNotBlank(dto.getDeliveryPartyCode()), DirectProductEntity::getDeliveryPartyCode, dto.getDeliveryPartyCode())
        .in(CollectionUtils.isNotEmpty(dto.getDirectCodes()), DirectProductEntity::getDirectCode, dto.getDirectCodes())
        .in(CollectionUtils.isNotEmpty(dto.getDeliveryPartyCodes()), DirectProductEntity::getDeliveryPartyCode, dto.getDeliveryPartyCodes())
        .eq(StringUtils.isNotBlank(dto.getSellPartyCode()), DirectProductEntity::getSellPartyCode, dto.getSellPartyCode())
        .eq(StringUtils.isNotBlank(dto.getProductCode()), DirectProductEntity::getProductCode, dto.getProductCode())
        .eq(StringUtils.isNotBlank(dto.getKaProductCode()), DirectProductEntity::getKaProductCode, dto.getKaProductCode())
        .eq(StringUtils.isNotBlank(dto.getIsDefault()), DirectProductEntity::getIsDefault, dto.getIsDefault())
        .in(CollectionUtils.isNotEmpty(dto.getProductCodes()), DirectProductEntity::getProductCode, dto.getProductCodes())
        .in(CollectionUtils.isNotEmpty(dto.getKaProductCodes()), DirectProductEntity::getKaProductCode, dto.getKaProductCodes())
        .in(CollectionUtils.isNotEmpty(dto.getSellPartyCodes()),DirectProductEntity::getSellPartyCode,dto.getSellPartyCodes())
        .list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids
   * @param enable
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .in(DirectProductEntity::getId, ids)
        .set(DirectProductEntity::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate().in(DirectProductEntity::getId, ids)
        .set(DirectProductEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 根据id删除数据(用于导入)
   * @param ids
   */
  public void deleteByIdsToImport(List<String> ids){
    this.lambdaUpdate().in(DirectProductEntity::getId, ids)
            .remove();
  }

  /**
   * 查询产品编码
   *
   * @return
   */
  public List<String> findProductCodes() {
    return this.baseMapper.findProductCodes();
  }
}
