package com.biz.crm.kms.business.direct.product.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.kms.business.direct.product.local.entity.DirectProductEntity;
import com.biz.crm.kms.business.direct.product.local.entity.DirectProductUnitEntity;
import com.biz.crm.kms.business.direct.product.local.repository.DirectProductRepository;
import com.biz.crm.kms.business.direct.product.local.repository.DirectProductUnitRepository;
import com.biz.crm.kms.business.direct.product.sdk.constant.ProductConstant;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductLogEventDto;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductUnitDto;
import com.biz.crm.kms.business.direct.product.sdk.event.DirectProductVoLogEventListener;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductUnitVoService;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductUnitVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgKmsDirectProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.liveobject.resolver.UUIDGenerator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

/**
 * 直营上架产品管理表(DirectProduct)表服务实现类
 *
 * @author xi.peng
 * @since 2022-09-23 10:47:58
 */
@Slf4j
@Service
public class DirectProductVoServiceImpl implements DirectProductVoService {

  @Autowired(required = false)
  private DirectProductRepository directProductRepository;

  @Autowired(required = false)
  private DirectProductUnitRepository directProductUnitRepository;

  @Autowired(required = false)
  private DirectProductUnitVoService directProductUnitVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private MasterDataMdgService masterDataMdgService;

  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  @Autowired(required = false)
  private DirectStoreVoService directStoreVoService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private DirectVoService directVoService;

  @Autowired(required = false)
  private RedisService redisService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  public List<DirectProductVo> findByListConditions(DirectProductDto dto) {
    dto = Optional.ofNullable(dto).orElse(new DirectProductDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.directProductRepository.findByListConditions(dto);
  }

  @Override
  public List<DirectProductVo> findByDirectProductDto(DirectProductDto dto) {
    Validate.notBlank(dto.getTenantCode(), "租户编号不能为空！");
    List<DirectProductEntity> productEntities = this.directProductRepository.findByDirectProductDto(dto);
    if (CollectionUtils.isEmpty(productEntities)) {
      return Lists.newArrayList();
    }
    List<DirectProductVo> productVos = (List<DirectProductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productEntities, DirectProductEntity.class, DirectProductVo.class, HashSet.class, ArrayList.class);
    List<DirectProductUnitVo> unitVos = this.directProductUnitVoService.findDirectProductIds(productVos.stream().map(DirectProductVo::getId).distinct().collect(Collectors.toList()), dto.getOrderType());
    Map<String, List<DirectProductUnitVo>> unitVoMap = Optional.ofNullable(unitVos).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(DirectProductUnitVo::getDirectProductId));
    productVos.forEach(productVo-> productVo.setUnitList(unitVoMap.get(productVo.getId())));
    return productVos;
  }

  @Override
  public List<DirectProductVo> findByDirectCode(String directCode) {
    if (StringUtils.isBlank(directCode)) {
      return null;
    }
    List<DirectProductEntity> entities = this.directProductRepository.findByDirectCode(directCode, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<DirectProductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DirectProductEntity.class, DirectProductVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<DirectProductVo> findByDirectCodes(List<String> directCodes) {
    if (CollectionUtils.isEmpty(directCodes)) {
      return null;
    }
    List<DirectProductEntity> entities = this.directProductRepository.findByDirectCodes(directCodes, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<DirectProductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DirectProductEntity.class, DirectProductVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DirectProductVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    DirectProductEntity entity = this.directProductRepository.findById(id);
    if (entity == null) {
      return null;
    }
    DirectProductVo directProductVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, DirectProductVo.class, HashSet.class, ArrayList.class);
    List<DirectProductUnitVo> directProductUnitVoList = this.directProductUnitVoService.findDirectProductIds(Collections.singletonList(entity.getId()), null);
    directProductVo.setUnitList(directProductUnitVoList);
    return directProductVo;
  }

  @Override
  @Transactional
  public void create(DirectProductDto dto) {
    this.createValidation(dto);
    if(StringUtils.isNotEmpty(dto.getDirectCode())){
      DirectVo directVo = this.directVoService.findByDirectCode(dto.getDirectCode());
      if(directVo!=null){
        dto.setBusinessFormatCode(directVo.getBusinessFormatCode());
        dto.setBusinessUnitCode(directVo.getBusinessUnitCode());
      }
    }
    DirectProductEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, DirectProductEntity.class, HashSet.class, ArrayList.class);
    this.directProductRepository.save(entity);
    // 创建商品单位
    if (CollectionUtils.isNotEmpty(dto.getUnitList())) {
      dto.getUnitList().forEach(label-> label.setDirectProductId(entity.getId()));
      this.directProductUnitVoService.createBatch(dto.getUnitList());
    }
    DirectProductLogEventDto logEventDto = new DirectProductLogEventDto();
    logEventDto.setOriginal(null);
    logEventDto.setNewest(dto);
    SerializableBiConsumer<DirectProductVoLogEventListener, DirectProductLogEventDto> onCreate =
            DirectProductVoLogEventListener::onCreate;
    this.nebulaNetEventClient.publish(logEventDto, DirectProductVoLogEventListener.class, onCreate);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void importBatch(List<DirectProductDto> dtoList) {
//    List<String> deleteId = new ArrayList<>();
//    log.info("开始校验重复数据!");
//    //校验数据库重复数据(覆盖操作)
//    List<String> deliveryCodes = dtoList.stream().distinct().map(DirectProductDto::getDeliveryPartyCode).collect(Collectors.toList());
//    List<String> sellPartyCodes = dtoList.stream().distinct().map(DirectProductDto::getSellPartyCode).collect(Collectors.toList());
//    List<String> directCodeList = dtoList.stream().distinct().map(DirectProductDto::getDirectCode).collect(Collectors.toList());
//    List<String> productCodeList = dtoList.stream().distinct().map(DirectProductDto::getProductCode).collect(Collectors.toList());
//    DirectProductDto dto = new DirectProductDto();
//    dto.setDeliveryPartyCodes(deliveryCodes);
//    dto.setSellPartyCodes(sellPartyCodes);
//    dto.setDirectCodes(directCodeList);
//    dto.setProductCodes(productCodeList);
//    List<DirectProductVo> extProductVos = this.findByListConditions(dto);
//    if (!CollectionUtils.isEmpty(extProductVos)){
//      log.info("开始匹配已存在数据!");
//      Map<String, List<DirectProductVo>> directProductVoMap = extProductVos.stream().collect(Collectors.groupingBy(
//              vo -> StringUtils.joinWith("_", vo.getDirectCode(), vo.getDeliveryPartyCode(), vo.getSellPartyCode(), vo.getProductCode())));
//      for (DirectProductDto productDto : dtoList) {
//        String extKey = StringUtils.joinWith("_",productDto.getDirectCode(),productDto.getDeliveryPartyCode(),productDto.getSellPartyCode(),productDto.getProductCode());
//        if (directProductVoMap.containsKey(extKey)) {
//          deleteId.addAll(directProductVoMap.get(extKey).stream().map(DirectProductVo::getId).distinct().collect(Collectors.toList()));
//        }
//      }
//      if (CollectionUtils.isNotEmpty(deleteId)){
//        deleteId = deleteId.stream().filter(id ->StringUtils.isNotEmpty(id)).collect(Collectors.toList());
//        log.info("id数据：{}", deleteId);
//        log.info("开始删除上架产品单位数据!");
//        //删除上架产品关系
//        this.directProductUnitRepository.deleteByDirectProductIds(deleteId);
//        log.info("开始删除上架产品数据!");
//        log.info("id数据：{}", deleteId);
//        //删除上架产品
//        this.directProductRepository.deleteByIdsToImport(deleteId);
//      }
//    }
//    log.info("开始赋值!");
//    if(CollectionUtils.isNotEmpty(directCodeList)){
//      List<DirectVo> directVoList = this.directVoService.findByDirectCodes(directCodeList);
//      if (CollectionUtils.isNotEmpty(directVoList)){
//        Map<String, DirectVo> directVoMap = directVoList.stream().collect(Collectors.toMap(DirectVo::getDirectCode, Function.identity(), (a, b) -> a));
//        for (DirectProductDto productDto : dtoList) {
//          if (directVoMap.containsKey(productDto.getDirectCode())){
//            DirectVo directVo = directVoMap.get(productDto.getDirectCode());
//            productDto.setBusinessFormatCode(directVo.getBusinessFormatCode());
//            productDto.setBusinessUnitCode(directVo.getBusinessUnitCode());
//          }
//        }
//      }
//    }
    for (DirectProductDto productDto : dtoList) {
      log.info("开始保存商品数据!");
      DirectProductEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(productDto, DirectProductEntity.class, HashSet.class, ArrayList.class);
      this.directProductRepository.save(entity);
      // 创建商品单位
      log.info("开始保存商品单位数据!");
      if (CollectionUtils.isNotEmpty(productDto.getUnitList())) {
        log.info("单位数据：{}", productDto.getUnitList());
//        productDto.getUnitList().forEach(label-> label.setDirectProductId(entity.getId()));
        this.directProductUnitVoService.createBatch(productDto.getUnitList());
      }
//      DirectProductLogEventDto logEventDto = new DirectProductLogEventDto();
//      logEventDto.setOriginal(null);
//      logEventDto.setNewest(productDto);
//      SerializableBiConsumer<DirectProductVoLogEventListener, DirectProductLogEventDto> onCreate =
//              DirectProductVoLogEventListener::onCreate;
//      this.nebulaNetEventClient.publish(logEventDto, DirectProductVoLogEventListener.class, onCreate);
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void update(DirectProductDto dto) {
    this.updateValidation(dto);
    DirectProductEntity oldEntity = this.directProductRepository.findById(dto.getId());
    Validate.notNull(oldEntity, "不存在或已删除！");
    DirectProductEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, DirectProductEntity.class, HashSet.class, ArrayList.class);
    DirectProductVo oldVo =  this.findDetailById(dto.getId());
    this.directProductRepository.updateById(entity);
    // 删除原商品单位
    this.directProductUnitVoService.deleteByDirectProductId(dto.getId());
    // 创建新商品单位
    if (CollectionUtils.isNotEmpty(dto.getUnitList())) {
      dto.getUnitList().forEach(label-> label.setDirectProductId(entity.getId()));
      this.directProductUnitVoService.createBatch(dto.getUnitList());
    }
    DirectProductDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldVo,DirectProductDto.class,HashSet.class,ArrayList.class);
    //修改业务日志
    DirectProductLogEventDto logEventDto = new DirectProductLogEventDto();
    logEventDto.setOriginal(oldDto);
    logEventDto.setNewest(dto);
    SerializableBiConsumer<DirectProductVoLogEventListener, DirectProductLogEventDto> onUpdate =
            DirectProductVoLogEventListener::onUpdate;
    this.nebulaNetEventClient.publish(logEventDto, DirectProductVoLogEventListener.class, onUpdate);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DirectProductEntity> entities = this.directProductRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.directProductRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
//    this.enableSaveLog(ids);
  }

//  @Async("otherThread")
//  @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
//  public void enableSaveLog(List<String> ids){
//    ids.forEach(aa ->{
//      //启用业务日志
//      DirectProductLogEventDto logEventDto = new DirectProductLogEventDto();
//      DirectProductVo oldVo =  this.findDetailById(aa);
//      DirectProductDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldVo,DirectProductDto.class,HashSet.class,ArrayList.class);
//      logEventDto.setOriginal(oldDto);
//      DirectProductDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(oldDto, DirectProductDto.class, null, null);
//      newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
//      logEventDto.setNewest(newDto);
//      SerializableBiConsumer<DirectProductVoLogEventListener, DirectProductLogEventDto> onEnable =
//              DirectProductVoLogEventListener::onEnable;
//      this.nebulaNetEventClient.publish(logEventDto, DirectProductVoLogEventListener.class, onEnable);
//    });
//  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    log.info("开始查询上架产品的的数据={}", JSON.toJSONString(ids));
    List<DirectProductEntity> entities = this.directProductRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    log.info("开始禁用上架产品产品={}", JSON.toJSONString(ids));
    this.directProductRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
//    log.info("禁用上架产品结束，开始写入日志");
//    this.disableSaveLog(ids);
  }


//  @Async("otherThread")
//  @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
//  public void disableSaveLog(List<String> ids){
//    log.info("禁用上架产品结束，进入写入日志");
//    ids.forEach(aa ->{
//      //启用业务日志
//      DirectProductLogEventDto logEventDto = new DirectProductLogEventDto();
//      DirectProductVo oldVo =  this.findDetailById(aa);
//      DirectProductDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldVo,DirectProductDto.class,HashSet.class,ArrayList.class);
//      logEventDto.setOriginal(oldDto);
//      DirectProductDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(oldDto, DirectProductDto.class, null, null);
//      newDto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
//      logEventDto.setNewest(newDto);
//      SerializableBiConsumer<DirectProductVoLogEventListener, DirectProductLogEventDto> onEnable =
//              DirectProductVoLogEventListener::onEnable;
//      this.nebulaNetEventClient.publish(logEventDto, DirectProductVoLogEventListener.class, onEnable);
//    });
//    log.info("禁用上架产品结束，写入日志结束");
//  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<DirectProductEntity> entities = this.directProductRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.directProductRepository.updateDelFlagByIds(ids);

    ids.forEach(aa ->{
      //启用业务日志
      DirectProductLogEventDto logEventDto = new DirectProductLogEventDto();
      DirectProductVo oldVo =  this.findDetailById(aa);
      DirectProductDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldVo,DirectProductDto.class,HashSet.class,ArrayList.class);
      logEventDto.setOriginal(oldDto);
      DirectProductDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(oldDto, DirectProductDto.class, null, null);
      newDto.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
      logEventDto.setNewest(newDto);
      SerializableBiConsumer<DirectProductVoLogEventListener, DirectProductLogEventDto> onEnable =
              DirectProductVoLogEventListener::onEnable;
      this.nebulaNetEventClient.publish(logEventDto, DirectProductVoLogEventListener.class, onEnable);

    });
  }


  @DynamicTaskService(cornExpression = "0 0 23 * * ?", taskDesc = "自动拉取MDG上架产品数据")
  public void authPullKmsDirectProductList(){
    MasterDataMdgBaseDto dto = new MasterDataMdgBaseDto();
    this.pullKmsDirectProductList(dto);
  }

  /**
   * 批量拉取 MDG 上架产品数据
   */
  @Override
  public void pullKmsDirectProductList(MasterDataMdgBaseDto dto) {
    loginUserService.refreshAuthentication(null);
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    if (ObjectUtils.isEmpty(dto)) {
      dto = new MasterDataMdgBaseDto();
    }
    if (StringUtils.isEmpty(dto.getPageNum())) {
      dto.setPageNum(ProductConstant.STORE_PAGE_NUM.toString());
    }
    if (StringUtils.isEmpty(dto.getPageSize())) {
      dto.setPageSize(ProductConstant.STORE_PAGE_SIZE.toString());
    }
    if (StringUtils.isEmpty(dto.getDs())) {
      String date = DateUtil.getDate("yyyyMMdd");
      Integer i = Integer.parseInt(date) - 1;
      dto.setDs(i.toString());
    }
    boolean lock = redisMutexService.tryLock(ProductConstant.STORE_PULL_LOCK, TimeUnit.SECONDS, ProductConstant.STORE_REDIS_TIME);
    try {
      List<DirectVo> allDirect = this.directVoService.findAllDirect();
      Set<String> directCodes = Sets.newHashSet();
      Map<String,List<DirectVo>> directMap = Maps.newHashMap();
      for(DirectVo directVo : allDirect){
        if("DY00000010".equals(directVo.getBusinessUnitCode()) && StringUtils.isNotEmpty(directVo.getCustomerRetailerName())){
          directCodes.add(directVo.getDirectCode());
          if(CollectionUtils.isNotEmpty(directMap.get(directVo.getCustomerRetailerName()))){
            List<DirectVo> directVoList = directMap.get(directVo.getCustomerRetailerName());
            directVoList.add(directVo);
            directMap.put(directVo.getCustomerRetailerName(),directVoList);
          }else{
            List<DirectVo> directVoList = Lists.newArrayList();
            directVoList.add(directVo);
            directMap.put(directVo.getCustomerRetailerName(),directVoList);
          }
        }
      }
      this.pullKmsDirectProduct(dto,directCodes,directMap);
    }catch (Exception e){
      log.error("拉取上架产品数据错误",e);
    }finally {
      if(lock){
        redisMutexService.unlock(ProductConstant.STORE_PULL_LOCK);
      }
    }
  }
  /**
   * 每天晚上11点缓存主数据产品税率
   */
  @DynamicTaskService(cornExpression = "0 0 23 * * ?", taskDesc = "缓存主数据产品税率")
  @Override
  public void cacheProductTaxRate() {
    List<String> codes = directProductRepository.findProductCodes();
    ProductQueryDto dto = new ProductQueryDto();
    dto.setProductCodeList(codes);
    List<ProductVo> productVos = this.productVoService.findByQueryDto(dto);
    if (CollectionUtils.isEmpty(productVos)){
      return;
    }
    Map<String, BigDecimal> map = productVos.stream().collect(Collectors.toMap(ProductVo::getProductCode, ProductVo::getRate));
    productVos.clear();
    redisService.set(ProductConstant.CACHE_PRODUCT_TAX_RATE,map,ProductConstant.CACHE_EXPIRE_TIME);
    codes.clear();

  }

  /**
   * 上架产品数据转换
   */
  private void pullKmsDirectProduct(MasterDataMdgBaseDto dto,Set<String> directCodes,Map<String,List<DirectVo>> directMap){
    List<DirectProductEntity> productList = Lists.newArrayList();
    List<DirectProductUnitDto> unitDtoList = Lists.newArrayList();
    //获取到需要同步的产品数据
    List<MasterDataMdgKmsDirectProductVo> directProducts = masterDataMdgService.pullKmsDirectProductList(dto);
    if(CollectionUtils.isEmpty(directProducts)){
      return;
    }
    //组装同步到的产品数据
    Set<String> storeCodes = Sets.newHashSet();
    Set<String> productCodes = Sets.newHashSet();
    Set<String> kaProductCodes = Sets.newHashSet();
    Set<String> terminalCodes = Sets.newHashSet();
//    Set<String> partyCodes = Sets.newHashSet();
    Map<String,Boolean> defaultMap = Maps.newHashMap();
    Map<String,Set<Integer>> pullMap = Maps.newHashMap();
    Set<Integer> pullItem = Sets.newHashSet();
    String key = "";
    for(MasterDataMdgKmsDirectProductVo productVo : directProducts){
      storeCodes.add(productVo.getStoreCode());
      productCodes.add(productVo.getSapCode());
      kaProductCodes.add(productVo.getItemCode());
      key = productVo.getStoreCode()+"-"+productVo.getItemCode();
      if(StringUtils.isNotEmpty(productVo.getRetailerRegion())){
        key = key + "-"+productVo.getRetailerRegion();
      }
      if(pullMap.get(key)==null){
        pullItem = Sets.newHashSet();
        pullItem.add(productVo.getSubcontractingCoefficient());
        pullMap.put(key,pullItem);
      }else{
        pullItem = pullMap.get(key);
        pullItem.add(productVo.getSubcontractingCoefficient());
        pullMap.put(key,pullItem);
      }
    }
    //获取对应的直营门店的信息
    DirectStoreConditionDto storeDto = new DirectStoreConditionDto();
    storeDto.setSupermarketStoreCodes(storeCodes);
    storeDto.setDirectCodes(directCodes);
    List<DirectStoreVo> storeVoList = this.directStoreVoService.findByDirectStoreConditionDto(storeDto);
    Map<String,DirectStoreVo> storeMap = Maps.newHashMap();
    if(!CollectionUtils.isEmpty(storeVoList)){
      for(DirectStoreVo store : storeVoList){
        if(StringUtils.isNotEmpty(store.getRetailerRegion())){
          storeMap.put(store.getDirectCode()+"-"+store.getSupermarketStoreCode()+"-"+store.getRetailerRegion(),store);
        }else{
          storeMap.put(store.getDirectCode()+"-"+store.getSupermarketStoreCode()+"-",store);
        }
        terminalCodes.add(store.getTerminalCode());
//        partyCodes.add(store.getSoldToPartyCode());
      }
    }else{
      if(directProducts.size() == 1000){
        Integer pageNum = Integer.parseInt(dto.getPageNum());
        pageNum = pageNum+1;
        dto.setPageNum(pageNum.toString());
        this.pullKmsDirectProduct(dto,directCodes,directMap);
      }else{
        return;
      }
    }
    //获取对应的MDM产品主数据信息
    List<ProductVo> mdmProductVos = productVoService.findByProductCodes(new ArrayList<>(productCodes));
    Map<String,ProductVo> mdmProductVoMap = Maps.newHashMap();
    if(CollectionUtils.isNotEmpty(mdmProductVos)){
      for(ProductVo productVo : mdmProductVos){
        mdmProductVoMap.put(productVo.getProductCode(),productVo);
      }
    }
    //获取对应的KMS产品数据
    DirectProductDto productDto = new DirectProductDto();
    productDto.setTenantCode(TenantUtils.getTenantCode());
//    productDto.setProductCodes(new ArrayList<>(productCodes));
    productDto.setDeliveryPartyCodes(new ArrayList<>(terminalCodes));
//    productDto.setSellPartyCodes(new ArrayList<>(partyCodes));
    productDto.setKaProductCodes(new ArrayList<>(kaProductCodes));
    List<DirectProductVo> kmsProductVos = this.findByDirectProductDto(productDto);
    Map<String,DirectProductVo> kmsProductVoMap = Maps.newHashMap();
    for(DirectProductVo product : kmsProductVos){
        kmsProductVoMap.put(product.getDirectCode()+"-"+product.getKaProductCode()+"-"+product.getDeliveryPartyCode(),product);
          if("Y".equals(product.getIsDefault())){
            defaultMap.put(product.getDirectCode()+"-"+product.getKaProductCode()+"-"+product.getDeliveryPartyCode(),true);
          }
    }
    Map<String,String> checkMap = Maps.newHashMap();
    for(MasterDataMdgKmsDirectProductVo productVo :directProducts){
      //校验该零售商对应的系统是否存在
      if(CollectionUtils.isNotEmpty(directMap.get(productVo.getRetailer()))){
        List<DirectVo> directVoList = directMap.get(productVo.getRetailer());
        for(DirectVo directVo : directVoList){
          //校验该直营门店是否存在
          if(storeMap.get(directVo.getDirectCode()+"-"+productVo.getStoreCode()+"-"+productVo.getRetailerRegion())==null){
            continue;
          }
          //校验产品主数据是否存在
          if(mdmProductVoMap.get(productVo.getSapCode())==null){
            continue;
          }
          //校验导入数据中是否有重复数据。相同数据，单位系统不同的，已经合并过了
          if(checkMap.get(productVo.getStoreCode()+"-"+productVo.getSapCode()+"-"+productVo.getItemCode())!=null){
           continue;
          }else{
            checkMap.put(productVo.getStoreCode()+"-"+productVo.getSapCode()+"-"+productVo.getItemCode(),"1");
          }
          DirectStoreVo storeVo = storeMap.get(directVo.getDirectCode()+"-"+productVo.getStoreCode() + "-" + productVo.getRetailerRegion());
          DirectProductEntity productEntity = new DirectProductEntity();
          if(kmsProductVoMap.get(storeVo.getDirectCode()+"-"+productVo.getItemCode()+"-"+storeVo.getTerminalCode())!=null){
            DirectProductVo oldProdcut = kmsProductVoMap.get(storeVo.getDirectCode()+"-"+productVo.getItemCode()+"-"+storeVo.getTerminalCode());
            productEntity.setId(oldProdcut.getId());
            if(defaultMap.get(storeVo.getDirectCode()+"-"+productVo.getItemCode()+"-"+storeVo.getTerminalCode())!=null){
              productEntity.setIsDefault(oldProdcut.getIsDefault());
            }else{
              productEntity.setIsDefault("Y");
              defaultMap.put(storeVo.getDirectCode()+"-"+productVo.getItemCode()+"-"+storeVo.getTerminalCode(),true);
            }
            if(CollectionUtils.isNotEmpty(oldProdcut.getUnitList())){
              for(DirectProductUnitVo unitVo : oldProdcut.getUnitList()){
                key = storeVo.getSupermarketStoreCode()+"-"+oldProdcut.getKaProductCode();
                if(StringUtils.isNotEmpty(storeVo.getRetailerRegion())){
                  key = key + "-"+storeVo.getRetailerRegion();
                }
                if(pullMap.get(key)!=null){
                  pullItem = pullMap.get(key);
                  if(pullItem.contains(unitVo.getKaUnitQuantity().intValue())){
                    pullItem.remove(unitVo.getKaUnitQuantity().intValue());
                    pullMap.put(key,pullItem);
                  }
                }
              }
            }
          }else {
            productEntity.setId(UUID.randomUUID().toString().replaceAll("-",""));
            if(defaultMap.get(storeVo.getDirectCode()+"-"+productVo.getItemCode()+"-"+storeVo.getTerminalCode())!=null){
              productEntity.setIsDefault("N");
            }else{
              productEntity.setIsDefault("Y");
              defaultMap.put(storeVo.getDirectCode()+"-"+productVo.getItemCode()+"-"+storeVo.getTerminalCode(),true);
            }
          }
          productEntity.setTenantCode(TenantUtils.getTenantCode());
          productEntity.setDirectCode(storeVo.getDirectCode());
          productEntity.setOnShelfStatus("Y");
          productEntity.setKaProductCode(productVo.getItemCode());
          productEntity.setKaProductName(productVo.getItemName());
          productEntity.setProductCode(productVo.getSapCode());
          productEntity.setProductName(productVo.getSapName());
          productEntity.setSellPartyCode(storeVo.getSoldToPartyCode());
          productEntity.setSellPartyName(storeVo.getSoldToPartyName());
          productEntity.setBusinessFormatCode(directVo.getBusinessFormatCode());
          productEntity.setBusinessUnitCode(directVo.getBusinessUnitCode());
          productEntity.setDeliveryPartyCode(storeVo.getTerminalCode());
          productEntity.setDeliveryPartyName(storeVo.getTerminalName());
          productList.add(productEntity);
//      this.directProductRepository.saveOrUpdate(productEntity);
          List<String> orderType = Lists.newArrayList();
          orderType.add("acceptance");
          orderType.add("return");
          orderType.add("sale");
          orderType.add("stock");
          orderType.add("expensesheet");
          orderType.add("statement");

          key = productVo.getStoreCode()+"-"+productVo.getItemCode();
          if(StringUtils.isNotEmpty(productVo.getRetailerRegion())){
            key = key + "-"+storeVo.getRetailerRegion();
          }
          pullItem = pullMap.get(key);

          for(Integer unitNum : pullItem){
            for(String ordertypeStr : orderType){
              DirectProductUnitDto unitDto = new DirectProductUnitDto();
              unitDto.setDirectProductId(productEntity.getId());
              unitDto.setUnitQuantity(new BigDecimal(1));
              unitDto.setOrderType(ordertypeStr);
              unitDto.setUnitCode(mdmProductVoMap.get(productVo.getSapCode()).getBaseUnit());
              unitDto.setUnitName(mdmProductVoMap.get(productVo.getSapCode()).getBaseUnitDes());
              unitDto.setKaUnitQuantity(new BigDecimal(unitNum));
              unitDtoList.add(unitDto);
            }
          }
        }
      }else{
        continue;
      }
    }
    if(CollectionUtils.isNotEmpty(productList)){
      this.directProductRepository.saveOrUpdateBatch(productList);
      if(CollectionUtils.isNotEmpty(unitDtoList)){
        this.directProductUnitVoService.createBatch(unitDtoList);
      }
    }
    if(directProducts.size() == 1000){
      Integer pageNum = Integer.parseInt(dto.getPageNum());
      pageNum = pageNum+1;
      dto.setPageNum(pageNum.toString());
      this.pullKmsDirectProduct(dto,directCodes,directMap);
    }else{
      return;
    }
  }

  private void createValidation(DirectProductDto dto) {
    this.validation(dto);
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
  }

  private void updateValidation(DirectProductDto dto) {
    this.validation(dto);
    Validate.notBlank(dto.getId(), "ID不能为空！");
  }

  private void validation(DirectProductDto dto) {
    Validate.notNull(dto, "对象不能为空！");
    Validate.notBlank(dto.getDirectCode(), "系统编码不能为空！");
    Validate.notBlank(dto.getSellPartyCode(), "售达方编码不能为空！");
    Validate.notBlank(dto.getSellPartyName(), "售达方名称不能为空！");
    Validate.notBlank(dto.getDeliveryPartyCode(), "送达方编码不能为空！");
    //TODO 临时屏蔽 2022-12-17 10:38:23
//    Validate.notBlank(dto.getDeliveryPartyName(), "送达方名称不能为空！");
    Validate.notBlank(dto.getProductCode(), "产品编码不能为空！");
    Validate.notBlank(dto.getProductName(), "产品名称不能为空！");
    Validate.notBlank(dto.getKaProductCode(), "商超产品编码不能为空！");
    Validate.notBlank(dto.getIsDefault(), "默认物料不能为空！");
    Validate.notBlank(dto.getOnShelfStatus(), "是否上架产品不能为空！");
    // 长度校验
    Validate.isTrue(dto.getKaProductCode().length()<=64, "客户产品编码最多64位字符串");
    if (StringUtils.isNotBlank(dto.getRemark())) {
      Validate.isTrue(dto.getRemark().length()<=400, "备注最多400位字符串");
    }
    // 新增或编辑做重复性校验
    if (BooleanEnum.TRUE.getNumStr().equals(dto.getIsDefault())) {
      // 报文中是否默认物料为”是“时，需要校验产品组（直营+售达方+KA产品+送达方）最多一条默认物料记录
      DirectProductDto directProductDto = new DirectProductDto();
      directProductDto.setTenantCode(TenantUtils.getTenantCode());
      directProductDto.setDirectCode(dto.getDirectCode());
      directProductDto.setDeliveryPartyCode(dto.getDeliveryPartyCode());
      directProductDto.setSellPartyCode(dto.getSellPartyCode());
      directProductDto.setKaProductCode(dto.getKaProductCode());
      directProductDto.setIsDefault(BooleanEnum.TRUE.getNumStr());
      List<DirectProductEntity> directProductEntities = this.directProductRepository.findByDirectProductDto(directProductDto);
      // 编辑时，排除自己
      if (StringUtils.isNotBlank(dto.getId())) {
        directProductEntities = directProductEntities.stream().filter(dp->!dto.getId().equals(dp.getId())).collect(Collectors.toList());
      }
      Validate.isTrue(CollectionUtils.isEmpty(directProductEntities),
          String.format("产品组：系统[%s]售达方[%s]送达方[%s]客户产品[%s]已存在默认物料记录！",
              dto.getDirectCode(), dto.getSellPartyCode(), dto.getDeliveryPartyCode(),dto.getKaProductCode()));
    } else {
      // 报文中是否默认物料为”否“时，需要校验产品组（直营+售达方+送达方+企业产品+KA产品）最多一条非默认物料记录
      DirectProductDto directProductDto = new DirectProductDto();
      directProductDto.setTenantCode(TenantUtils.getTenantCode());
      directProductDto.setDirectCode(dto.getDirectCode());
      directProductDto.setDeliveryPartyCode(dto.getDeliveryPartyCode());
      directProductDto.setSellPartyCode(dto.getSellPartyCode());
      directProductDto.setProductCode(dto.getProductCode());
      directProductDto.setKaProductCode(dto.getKaProductCode());
      directProductDto.setIsDefault(BooleanEnum.FALSE.getNumStr());
      List<DirectProductEntity> directProductEntities = this.directProductRepository.findByDirectProductDto(directProductDto);
      // 编辑时，排除自己
      if (StringUtils.isNotBlank(dto.getId())) {
        directProductEntities = directProductEntities.stream().filter(dp->!dto.getId().equals(dp.getId())).collect(Collectors.toList());
      }
      Validate.isTrue(CollectionUtils.isEmpty(directProductEntities),
          String.format("产品组：系统[%s]售达方[%s]送达方[%s]企业产品[%s]客户产品[%s]已存在非默认物料记录！",
              dto.getDirectCode(), dto.getSellPartyCode(), directProductDto.getDeliveryPartyCode(),dto.getProductCode(), dto.getKaProductCode()));
    }
  }

}
