package com.biz.crm.kms.business.direct.product.local.service.notifier;

import com.alibaba.fastjson.JSON;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.listener.DirectOperationListener;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 系统操作监听器实现
 *
 * @author pengxi
 */
@Slf4j
@Component
public class DirectProductListenerImpl implements DirectOperationListener {

  @Autowired(required = false) private DirectProductVoService directProductVoService;

  /**
   * 启用时，通知上层业务进行业务处理
   *
   * @param directCodes 系统编码集合
   */
  @Override
  @Async("otherThread")
  public void onEnable(List<String> directCodes) {
    List<DirectProductVo> directProductVos = this.directProductVoService.findByDirectCodes(directCodes);
    if (CollectionUtils.isEmpty(directProductVos)) {
      return;
    }
    List<String> ids = directProductVos.stream().map(DirectProductVo::getId).distinct().collect(Collectors.toList());
    this.directProductVoService.enableBatch(ids);
    log.info("系统启用时被通知启用的直营上架产品ID集合：{}", ids);
  }

  /**
   * 禁用时，通知上层业务进行业务处理
   *
   * @param directCodes 系统编码集合
   */
  @Override
  @Async("otherThread")
  public void onDisable(List<String> directCodes) {
    log.info("进入禁用上架产品的方法={}", JSON.toJSONString(directCodes));
    List<DirectProductVo> directProductVos = this.directProductVoService.findByDirectCodes(directCodes);
    log.info("进入禁用上架产品的方法，查询出来的上架产品数据量为={}",directProductVos.size());
    if (CollectionUtils.isEmpty(directProductVos)) {
      return;
    }
    List<String> ids = directProductVos.stream().map(DirectProductVo::getId).distinct().collect(Collectors.toList());
    log.info("开始调用禁用上架的方法={}", JSON.toJSONString(ids));
    this.directProductVoService.disableBatch(ids);
    log.info("系统禁用时被通知禁用的直营上架产品ID集合：{}", ids);
  }

  /**
   * 删除时，通知上层业务进行业务处理
   *
   * @param directCodes 系统编码集合
   */
  @Override
  @Async("otherThread")
  public void onDelete(List<String> directCodes) {
    List<DirectProductVo> directProductVos = this.directProductVoService.findByDirectCodes(directCodes);
    if (CollectionUtils.isEmpty(directProductVos)) {
      return;
    }
    List<String> ids = directProductVos.stream().map(DirectProductVo::getId).distinct().collect(Collectors.toList());
    this.directProductVoService.updateDelFlagByIds(ids);
    log.info("系统删除时被通知删除的直营上架产品ID集合：{}", ids);
  }
  
}
