package com.biz.crm.kms.business.direct.store.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.kms.business.direct.store.local.entity.DirectStoreEntity;
import com.biz.crm.kms.business.direct.store.local.mapper.DirectStoreMapper;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreDto;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.grab.rule.local.model.DirectStoreConditionModel;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 直营门店表的数据库访问类 {@link DirectStoreEntity}
 *
 * @author ning.zhang
 * @date 2022-09-28 10:28:05
 */
@Component
public class DirectStoreRepository extends ServiceImpl<DirectStoreMapper, DirectStoreEntity> {

  /**
   * 多条件查询直营门店列表
   *
   * @param model 条件model
   * @return 直营门店列表
   */
  public List<DirectStoreEntity> findByDirectStoreConditionModel(DirectStoreConditionModel model) {
    if (StringUtils.isBlank(model.getTenantCode())) {
      return Collections.emptyList();
    }
    if (CollectionUtils.isEmpty(model.getIds()) &&
            CollectionUtils.isEmpty(model.getDirectCodes()) &&
            CollectionUtils.isEmpty(model.getSupermarketStoreCodes()) &&
            CollectionUtils.isEmpty(model.getTerminalCodes()) &&
            CollectionUtils.isEmpty(model.getSoldToPartyCodes()) &&
            StringUtils.isBlank(model.getDirectCode()) &&
            StringUtils.isBlank(model.getTerminalCode()) &&
            StringUtils.isBlank(model.getSupermarketStoreCode())) {
      return Collections.emptyList();
    }
    return this.baseMapper.findByDirectStoreConditionModel(model);
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(DirectStoreEntity::getId, ids)
        .set(DirectStoreEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

    public Page<DirectStoreVo> findByConditions(Pageable pageable, DirectStoreConditionDto dto) {
      if (pageable == null) {
        pageable = PageRequest.of(1, 50);
      }
      dto.setTenantCode(TenantUtils.getTenantCode());
      Page<DirectStoreVo> page = new Page<DirectStoreVo>(pageable.getPageNumber(), pageable.getPageSize());
      return this.baseMapper.findByConditions(page, dto);
    }

  /**
   * 通过送达方编码查找售达方
   * @param deliveryCodes
   * @param tenantCode
   * @return
   */
  public List<DirectStoreEntity> findByDeliveryCodes(List<String> deliveryCodes,String tenantCode){

    if (CollectionUtils.isEmpty(deliveryCodes)){
      return new ArrayList<>();
    }
      return this.lambdaQuery()
              .eq(DirectStoreEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
              .eq(DirectStoreEntity::getTenantCode,tenantCode)
              .in(DirectStoreEntity::getTerminalCode,deliveryCodes)
              .list();
    }

  /**
   * 通过系统找送达方
   * @param dto
   * @return
   */
  public List<DirectStoreEntity> findSoldToPartyByDirectCode(DirectStoreDto dto){
      if (ObjectUtils.isEmpty(dto) || StringUtils.isEmpty(dto.getDirectCode())){
        return new ArrayList<>();
      }
      return this.baseMapper.findSoldToPartyByDirectCode(dto,TenantUtils.getTenantCode());
    }

  /**
   * 通过系统集合找售达方编码
   * @param directCodes
   * @return
   */
  public List<DirectStoreVo> findSoldToPartyByDirectCodes(List<String> directCodes) {
      if (CollectionUtils.isEmpty(directCodes)) {
        return Lists.newArrayList();
      } else {
        return this.baseMapper.findSoldToPartyByDirectCodes(directCodes, TenantUtils.getTenantCode());
      }
    }
}
